"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessAnalyzer = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [access-analyzer](https://docs.aws.amazon.com/service-authorization/latest/reference/list_iamaccessanalyzer.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class AccessAnalyzer extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [access-analyzer](https://docs.aws.amazon.com/service-authorization/latest/reference/list_iamaccessanalyzer.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'access-analyzer';
        this.accessLevelList = {
            "Write": [
                "ApplyArchiveRule",
                "CreateAnalyzer",
                "CreateArchiveRule",
                "DeleteAnalyzer",
                "DeleteArchiveRule",
                "StartResourceScan",
                "UpdateArchiveRule",
                "UpdateFindings"
            ],
            "Read": [
                "GetAnalyzedResource",
                "GetAnalyzer",
                "GetArchiveRule",
                "GetFinding",
                "ListAnalyzedResources",
                "ListFindings",
                "ListTagsForResource"
            ],
            "List": [
                "ListAnalyzers",
                "ListArchiveRules"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to apply an archive rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ApplyArchiveRule.html
     */
    toApplyArchiveRule() {
        this.to('access-analyzer:ApplyArchiveRule');
        return this;
    }
    /**
     * Grants permission to create an analyzer.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_CreateAnalyzer.html
     */
    toCreateAnalyzer() {
        this.to('access-analyzer:CreateAnalyzer');
        return this;
    }
    /**
     * Grants permission to create an archive rule for the specified analyzer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_CreateArchiveRule.html
     */
    toCreateArchiveRule() {
        this.to('access-analyzer:CreateArchiveRule');
        return this;
    }
    /**
     * Grants permission to delete the specified analyzer.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_DeleteAnalyzer.html
     */
    toDeleteAnalyzer() {
        this.to('access-analyzer:DeleteAnalyzer');
        return this;
    }
    /**
     * Grants permission to delete archive rules for the specified analyzer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_DeleteArchiveRule.html
     */
    toDeleteArchiveRule() {
        this.to('access-analyzer:DeleteArchiveRule');
        return this;
    }
    /**
     * Grants permission to retrieve information about an analyzed resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_GetAnalyzedResource.html
     */
    toGetAnalyzedResource() {
        this.to('access-analyzer:GetAnalyzedResource');
        return this;
    }
    /**
     * Grants permission to retrieve information about analyzers.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_GetAnalyzer.html
     */
    toGetAnalyzer() {
        this.to('access-analyzer:GetAnalyzer');
        return this;
    }
    /**
     * Grants permission to retrieve information about archive rules for the specified analyzer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_GetArchiveRule.html
     */
    toGetArchiveRule() {
        this.to('access-analyzer:GetArchiveRule');
        return this;
    }
    /**
     * Grants permission to retrieve findings.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_GetFinding.html
     */
    toGetFinding() {
        this.to('access-analyzer:GetFinding');
        return this;
    }
    /**
     * Grants permission to retrieve a list of resources that have been analyzed.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListAnalyzedResources.html
     */
    toListAnalyzedResources() {
        this.to('access-analyzer:ListAnalyzedResources');
        return this;
    }
    /**
     * Grants permission to retrieves a list of analyzers.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListAnalyzers.html
     */
    toListAnalyzers() {
        this.to('access-analyzer:ListAnalyzers');
        return this;
    }
    /**
     * Grants permission to retrieve a list of archive rules from an analyzer.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListArchiveRules.html
     */
    toListArchiveRules() {
        this.to('access-analyzer:ListArchiveRules');
        return this;
    }
    /**
     * Grants permission to retrieve a list of findings from an analyzer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListFindings.html
     */
    toListFindings() {
        this.to('access-analyzer:ListFindings');
        return this;
    }
    /**
     * Grants permission to retrieve a list of tags applied to a resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('access-analyzer:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to start a scan of the policies applied to a resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_StartResourceScan.html
     */
    toStartResourceScan() {
        this.to('access-analyzer:StartResourceScan');
        return this;
    }
    /**
     * Grants permission to add a tag to a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('access-analyzer:TagResource');
        return this;
    }
    /**
     * Grants permission to remove a tag from a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('access-analyzer:UntagResource');
        return this;
    }
    /**
     * Grants permission to modify an archive rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_UpdateArchiveRule.html
     */
    toUpdateArchiveRule() {
        this.to('access-analyzer:UpdateArchiveRule');
        return this;
    }
    /**
     * Grants permission to modify findings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_UpdateFindings.html
     */
    toUpdateFindings() {
        this.to('access-analyzer:UpdateFindings');
        return this;
    }
    /**
     * Adds a resource of type Analyzer to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-getting-started.html#permission-resources
     *
     * @param analyzerName - Identifier for the analyzerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAnalyzer(analyzerName, account, region, partition) {
        var arn = 'arn:${Partition}:access-analyzer:${Region}:${Account}:analyzer/${AnalyzerName}';
        arn = arn.replace('${AnalyzerName}', analyzerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ArchiveRule to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-getting-started.html#permission-resources
     *
     * @param analyzerName - Identifier for the analyzerName.
     * @param ruleName - Identifier for the ruleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onArchiveRule(analyzerName, ruleName, account, region, partition) {
        var arn = 'arn:${Partition}:access-analyzer:${Region}:${Account}:analyzer/${AnalyzerName}/archive-rule/${RuleName}';
        arn = arn.replace('${AnalyzerName}', analyzerName);
        arn = arn.replace('${RuleName}', ruleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.AccessAnalyzer = AccessAnalyzer;
//# sourceMappingURL=data:application/json;base64,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