"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkFirewall = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [network-firewall](https://docs.aws.amazon.com/service-authorization/latest/reference/list_networkfirewall.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class NetworkFirewall extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [network-firewall](https://docs.aws.amazon.com/service-authorization/latest/reference/list_networkfirewall.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'network-firewall';
        this.accessLevelList = {
            "Write": [
                "AssociateFirewallPolicy",
                "AssociateSubnets",
                "CreateFirewall",
                "CreateFirewallPolicy",
                "CreateRuleGroup",
                "DeleteFirewall",
                "DeleteFirewallPolicy",
                "DeleteResourcePolicy",
                "DeleteRuleGroup",
                "DisassociateSubnets",
                "PutResourcePolicy",
                "UpdateFirewallDeleteProtection",
                "UpdateFirewallDescription",
                "UpdateFirewallPolicy",
                "UpdateFirewallPolicyChangeProtection",
                "UpdateLoggingConfiguration",
                "UpdateRuleGroup",
                "UpdateSubnetChangeProtection"
            ],
            "Read": [
                "DescribeFirewall",
                "DescribeFirewallPolicy",
                "DescribeLoggingConfiguration",
                "DescribeResourcePolicy",
                "DescribeRuleGroup"
            ],
            "List": [
                "ListFirewallPolicies",
                "ListFirewalls",
                "ListRuleGroups",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to create an association between a firewall policy and a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_AssociateFirewallPolicy.html
     */
    toAssociateFirewallPolicy() {
        this.to('network-firewall:AssociateFirewallPolicy');
        return this;
    }
    /**
     * Grants permission to associate VPC subnets to a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_AssociateSubnets.html
     */
    toAssociateSubnets() {
        this.to('network-firewall:AssociateSubnets');
        return this;
    }
    /**
     * Grants permission to create an AWS Network Firewall firewall
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_CreateFirewall.html
     */
    toCreateFirewall() {
        this.to('network-firewall:CreateFirewall');
        return this;
    }
    /**
     * Grants permission to create an AWS Network Firewall firewall policy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_CreateFirewallPolicy.html
     */
    toCreateFirewallPolicy() {
        this.to('network-firewall:CreateFirewallPolicy');
        return this;
    }
    /**
     * Grants permission to create an AWS Network Firewall rule group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_CreateRuleGroup.html
     */
    toCreateRuleGroup() {
        this.to('network-firewall:CreateRuleGroup');
        return this;
    }
    /**
     * Grants permission to delete a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DeleteFirewall.html
     */
    toDeleteFirewall() {
        this.to('network-firewall:DeleteFirewall');
        return this;
    }
    /**
     * Grants permission to delete a firewall policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DeleteFirewallPolicy.html
     */
    toDeleteFirewallPolicy() {
        this.to('network-firewall:DeleteFirewallPolicy');
        return this;
    }
    /**
     * Grants permission to delete a resource policy for a firewall policy or rule group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        this.to('network-firewall:DeleteResourcePolicy');
        return this;
    }
    /**
     * Grants permission to delete a rule group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DeleteRuleGroup.html
     */
    toDeleteRuleGroup() {
        this.to('network-firewall:DeleteRuleGroup');
        return this;
    }
    /**
     * Grants permission to retrieve the data objects that define a firewall
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeFirewall.html
     */
    toDescribeFirewall() {
        this.to('network-firewall:DescribeFirewall');
        return this;
    }
    /**
     * Grants permission to retrieve the data objects that define a firewall policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeFirewallPolicy.html
     */
    toDescribeFirewallPolicy() {
        this.to('network-firewall:DescribeFirewallPolicy');
        return this;
    }
    /**
     * Grants permission to describe the logging configuration of a firewall
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeLoggingConfiguration.html
     */
    toDescribeLoggingConfiguration() {
        this.to('network-firewall:DescribeLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to describe a resource policy for a firewall policy or rule group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeResourcePolicy.html
     */
    toDescribeResourcePolicy() {
        this.to('network-firewall:DescribeResourcePolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the data objects that define a rule group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeRuleGroup.html
     */
    toDescribeRuleGroup() {
        this.to('network-firewall:DescribeRuleGroup');
        return this;
    }
    /**
     * Grants permission to disassociate VPC subnets from a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/APIReference/API_DisassociateSubnets.html
     */
    toDisassociateSubnets() {
        this.to('network-firewall:DisassociateSubnets');
        return this;
    }
    /**
     * Grants permission to retrieve the metadata for firewall policies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListFirewallPolicies.html
     */
    toListFirewallPolicies() {
        this.to('network-firewall:ListFirewallPolicies');
        return this;
    }
    /**
     * Grants permission to retrieve the metadata for firewalls
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListFirewalls.html
     */
    toListFirewalls() {
        this.to('network-firewall:ListFirewalls');
        return this;
    }
    /**
     * Grants permission to retrieve the metadata for rule groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListRuleGroups.html
     */
    toListRuleGroups() {
        this.to('network-firewall:ListRuleGroups');
        return this;
    }
    /**
     * Grants permission to retrieve the tags for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('network-firewall:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to put a resource policy for a firewall policy or rule group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        this.to('network-firewall:PutResourcePolicy');
        return this;
    }
    /**
     * Grants permission to attach tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('network-firewall:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('network-firewall:UntagResource');
        return this;
    }
    /**
     * Grants permission to add or remove delete protection for a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateFirewallDeleteProtection.html
     */
    toUpdateFirewallDeleteProtection() {
        this.to('network-firewall:UpdateFirewallDeleteProtection');
        return this;
    }
    /**
     * Grants permission to modify the description for a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateFirewallDescription.html
     */
    toUpdateFirewallDescription() {
        this.to('network-firewall:UpdateFirewallDescription');
        return this;
    }
    /**
     * Grants permission to modify a firewall policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateFirewallPolicy.html
     */
    toUpdateFirewallPolicy() {
        this.to('network-firewall:UpdateFirewallPolicy');
        return this;
    }
    /**
     * Grants permission to add or remove firewall policy change protection for a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateFirewallPolicyChangeProtection.html
     */
    toUpdateFirewallPolicyChangeProtection() {
        this.to('network-firewall:UpdateFirewallPolicyChangeProtection');
        return this;
    }
    /**
     * Grants permission to modify the logging configuration of a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateLoggingConfiguration.html
     */
    toUpdateLoggingConfiguration() {
        this.to('network-firewall:UpdateLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to modify a rule group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateRuleGroup.html
     */
    toUpdateRuleGroup() {
        this.to('network-firewall:UpdateRuleGroup');
        return this;
    }
    /**
     * Grants permission to add or remove subnet change protection for a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateSubnetChangeProtection.html
     */
    toUpdateSubnetChangeProtection() {
        this.to('network-firewall:UpdateSubnetChangeProtection');
        return this;
    }
    /**
     * Adds a resource of type Firewall to the statement
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_Firewall.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFirewall(name, account, region, partition) {
        var arn = 'arn:${Partition}:network-firewall:${Region}:${Account}:firewall/${Name}';
        arn = arn.replace('${Name}', name);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type FirewallPolicy to the statement
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_FirewallPolicyResponse.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFirewallPolicy(name, account, region, partition) {
        var arn = 'arn:${Partition}:network-firewall:${Region}:${Account}:firewall-policy/${Name}';
        arn = arn.replace('${Name}', name);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type StatefulRuleGroup to the statement
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_RuleGroupResponse.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStatefulRuleGroup(name, account, region, partition) {
        var arn = 'arn:${Partition}:network-firewall:${Region}:${Account}:stateful-rulegroup/${Name}';
        arn = arn.replace('${Name}', name);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type StatelessRuleGroup to the statement
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_RuleGroupResponse.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStatelessRuleGroup(name, account, region, partition) {
        var arn = 'arn:${Partition}:network-firewall:${Region}:${Account}:stateless-rulegroup/${Name}';
        arn = arn.replace('${Name}', name);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.NetworkFirewall = NetworkFirewall;
//# sourceMappingURL=data:application/json;base64,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