"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Servicequotas = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [servicequotas](https://docs.aws.amazon.com/service-authorization/latest/reference/list_servicequotas.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Servicequotas extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [servicequotas](https://docs.aws.amazon.com/service-authorization/latest/reference/list_servicequotas.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'servicequotas';
        this.accessLevelList = {
            "Write": [
                "AssociateServiceQuotaTemplate",
                "DeleteServiceQuotaIncreaseRequestFromTemplate",
                "DisassociateServiceQuotaTemplate",
                "PutServiceQuotaIncreaseRequestIntoTemplate",
                "RequestServiceQuotaIncrease"
            ],
            "Read": [
                "GetAWSDefaultServiceQuota",
                "GetAssociationForServiceQuotaTemplate",
                "GetRequestedServiceQuotaChange",
                "GetServiceQuota",
                "GetServiceQuotaIncreaseRequestFromTemplate",
                "ListAWSDefaultServiceQuotas",
                "ListRequestedServiceQuotaChangeHistory",
                "ListRequestedServiceQuotaChangeHistoryByQuota",
                "ListServiceQuotaIncreaseRequestsInTemplate",
                "ListServiceQuotas",
                "ListServices"
            ]
        };
    }
    /**
     * Grants permission to associate the Service Quotas template with your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_AssociateServiceQuotaTemplate.html
     */
    toAssociateServiceQuotaTemplate() {
        this.to('servicequotas:AssociateServiceQuotaTemplate');
        return this;
    }
    /**
     * Grants permission to remove the specified service quota from the service quota template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_DeleteServiceQuotaIncreaseRequestFromTemplate.html
     */
    toDeleteServiceQuotaIncreaseRequestFromTemplate() {
        this.to('servicequotas:DeleteServiceQuotaIncreaseRequestFromTemplate');
        return this;
    }
    /**
     * Grants permission to disassociate the Service Quotas template from your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_DisassociateServiceQuotaTemplate.html
     */
    toDisassociateServiceQuotaTemplate() {
        this.to('servicequotas:DisassociateServiceQuotaTemplate');
        return this;
    }
    /**
     * Grants permission to return the details for the specified service quota, including the AWS default value
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetAWSDefaultServiceQuota.html
     */
    toGetAWSDefaultServiceQuota() {
        this.to('servicequotas:GetAWSDefaultServiceQuota');
        return this;
    }
    /**
     * Grants permission to retrieve the ServiceQuotaTemplateAssociationStatus value, which tells you if the Service Quotas template is associated with an organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetAssociationForServiceQuotaTemplate.html
     */
    toGetAssociationForServiceQuotaTemplate() {
        this.to('servicequotas:GetAssociationForServiceQuotaTemplate');
        return this;
    }
    /**
     * Grants permission to retrieve the details for a particular service quota increase request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetRequestedServiceQuotaChange.html
     */
    toGetRequestedServiceQuotaChange() {
        this.to('servicequotas:GetRequestedServiceQuotaChange');
        return this;
    }
    /**
     * Grants permission to return the details for the specified service quota, including the applied value
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetServiceQuota.html
     */
    toGetServiceQuota() {
        this.to('servicequotas:GetServiceQuota');
        return this;
    }
    /**
     * Grants permission to retrieve the details for a service quota increase request from the service quota template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetServiceQuotaIncreaseRequestFromTemplate.html
     */
    toGetServiceQuotaIncreaseRequestFromTemplate() {
        this.to('servicequotas:GetServiceQuotaIncreaseRequestFromTemplate');
        return this;
    }
    /**
     * Grants permission to list all default service quotas for the specified AWS service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListAWSDefaultServiceQuotas.html
     */
    toListAWSDefaultServiceQuotas() {
        this.to('servicequotas:ListAWSDefaultServiceQuotas');
        return this;
    }
    /**
     * Grants permission to request a list of the changes to quotas for a service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListRequestedServiceQuotaChangeHistory.html
     */
    toListRequestedServiceQuotaChangeHistory() {
        this.to('servicequotas:ListRequestedServiceQuotaChangeHistory');
        return this;
    }
    /**
     * Grants permission to request a list of the changes to specific service quotas
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListRequestedServiceQuotaChangeHistoryByQuota.html
     */
    toListRequestedServiceQuotaChangeHistoryByQuota() {
        this.to('servicequotas:ListRequestedServiceQuotaChangeHistoryByQuota');
        return this;
    }
    /**
     * Grants permission to return a list of the service quota increase requests from the service quota template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServiceQuotaIncreaseRequestsInTemplate
     */
    toListServiceQuotaIncreaseRequestsInTemplate() {
        this.to('servicequotas:ListServiceQuotaIncreaseRequestsInTemplate');
        return this;
    }
    /**
     * Grants permission to list all service quotas for the specified AWS service, in that account, in that Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServiceQuotas.html
     */
    toListServiceQuotas() {
        this.to('servicequotas:ListServiceQuotas');
        return this;
    }
    /**
     * Grants permission to list the AWS services available in Service Quotas
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServices.html
     */
    toListServices() {
        this.to('servicequotas:ListServices');
        return this;
    }
    /**
     * Grants permission to define and add a quota to the service quota template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifService()
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_PutServiceQuotaIncreaseRequestIntoTemplate.html
     */
    toPutServiceQuotaIncreaseRequestIntoTemplate() {
        this.to('servicequotas:PutServiceQuotaIncreaseRequestIntoTemplate');
        return this;
    }
    /**
     * Grants permission to submit the request for a service quota increase
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifService()
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_RequestServiceQuotaIncrease.html
     */
    toRequestServiceQuotaIncrease() {
        this.to('servicequotas:RequestServiceQuotaIncrease');
        return this;
    }
    /**
     * Adds a resource of type quota to the statement
     *
     * https://docs.aws.amazon.com/servicequotas/latest/userguide/quota-as-resource.html
     *
     * @param serviceCode - Identifier for the serviceCode.
     * @param quotaCode - Identifier for the quotaCode.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onQuota(serviceCode, quotaCode, account, region, partition) {
        var arn = 'arn:${Partition}:servicequotas:${Region}:${Account}:${ServiceCode}/${QuotaCode}';
        arn = arn.replace('${ServiceCode}', serviceCode);
        arn = arn.replace('${QuotaCode}', quotaCode);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters or restricts access to a specified AWS service
     *
     * Applies to actions:
     * - .toPutServiceQuotaIncreaseRequestIntoTemplate()
     * - .toRequestServiceQuotaIncrease()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifService(value, operator) {
        return this.if(`servicequotas:service`, value, operator || 'StringLike');
    }
}
exports.Servicequotas = Servicequotas;
//# sourceMappingURL=data:application/json;base64,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