"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProwlerAudit = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// import { Stack, Duration, RemovalPolicy, CustomResource } from 'aws-cdk-lib';
// eslint-disable-next-line no-duplicate-imports
// import { aws_iam as iam, aws_logs as logs, aws_s3 as s3, aws_codebuild as codebuild, aws_lambda as lambda, custom_resources as cr } from 'aws-cdk-lib';
// import { Construct } from 'constructs';
const codebuild = require("@aws-cdk/aws-codebuild");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const statement = require("cdk-iam-floyd");
/**
 * Creates a CodeBuild project to audit an AWS account with Prowler and stores the html report in a S3 bucket.
 *
 * This will run onece at the beginning and on a schedule afterwards. Partial contribution from https://github.com/stevecjones
 *
 * @stability stable
 */
class ProwlerAudit extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(parent, id, props) {
        var _b, _c, _d, _e;
        super(parent, id);
        // defaults
        this.serviceName = (props === null || props === void 0 ? void 0 : props.serviceName) ? props.serviceName : 'prowler';
        this.logsRetentionInDays = (props === null || props === void 0 ? void 0 : props.logsRetentionInDays) ? props.logsRetentionInDays : logs.RetentionDays.THREE_DAYS;
        this.enableScheduler = (props === null || props === void 0 ? void 0 : props.enableScheduler) ? props.enableScheduler : false;
        this.prowlerScheduler = (props === null || props === void 0 ? void 0 : props.prowlerScheduler) ? props.prowlerScheduler : 'cron(0 22 * * ? *)';
        this.prowlerOptions = (props === null || props === void 0 ? void 0 : props.prowlerOptions) ? props.prowlerOptions : '-M text,junit-xml,html,csv,json';
        this.prowlerVersion = (props === null || props === void 0 ? void 0 : props.prowlerVersion) ? props.prowlerVersion : '2.5.0';
        const reportBucket = (_b = props === null || props === void 0 ? void 0 : props.reportBucket) !== null && _b !== void 0 ? _b : new s3.Bucket(this, 'ReportBucket', {
            //bucketName: `${'123456'}-prowler-reports`,
            autoDeleteObjects: true,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        const reportGroup = new codebuild.ReportGroup(this, 'reportGroup', { /**reportGroupName: 'testReportGroup', */ removalPolicy: core_1.RemovalPolicy.DESTROY });
        reportGroup;
        const prowlerBuild = this.codebuildProject = new codebuild.Project(this, 'prowlerBuild', {
            description: 'Run Prowler assessment',
            timeout: core_1.Duration.hours(5),
            environment: {
                environmentVariables: {
                    BUCKET_REPORT: { value: reportBucket.bucketName || '' },
                    BUCKET_PREFIX: { value: (_c = props === null || props === void 0 ? void 0 : props.reportBucketPrefix) !== null && _c !== void 0 ? _c : '' },
                    PROWLER_OPTIONS: { value: this.prowlerOptions || '' },
                },
                buildImage: codebuild.LinuxBuildImage.fromCodeBuildImageId('aws/codebuild/amazonlinux2-x86_64-standard:3.0'),
            },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        'runtime-versions': {
                            python: 3.8,
                        },
                        'commands': [
                            'echo "Installing Prowler and dependencies..."',
                            'pip3 install detect-secrets',
                            'yum -y install jq',
                            'curl "https://awscli.amazonaws.com/awscli-exe-linux-x86_64.zip" -o "awscliv2.zip"',
                            'unzip awscliv2.zip',
                            './aws/install',
                            `git clone -b ${this.prowlerVersion} https://github.com/toniblyx/prowler`,
                        ],
                    },
                    build: {
                        commands: [
                            `echo "Running Prowler as ./prowler ${this.prowlerOptions} && echo OK || echo FAILED"`,
                            'cd prowler',
                            `./prowler ${this.prowlerOptions} && echo OK || echo FAILED`,
                        ],
                    },
                    post_build: {
                        commands: [
                            'echo "Uploading reports to S3..." ',
                            'aws s3 cp --sse AES256 output/ s3://$BUCKET_REPORT/$BUCKET_PREFIX --recursive',
                            'echo "Done!"',
                        ],
                    },
                },
                reports: {
                    [reportGroup.reportGroupName]: {
                        'files': ['**/*'],
                        'base-directory': 'prowler/junit-reports',
                        'file-format': 'JunitXml',
                    },
                },
            }),
        });
        (_d = prowlerBuild.role) === null || _d === void 0 ? void 0 : _d.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('SecurityAudit'));
        (_e = prowlerBuild.role) === null || _e === void 0 ? void 0 : _e.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('job-function/ViewOnlyAccess'));
        // prowlerBuild.addToRolePolicy(new statement.Dax().allow().to());
        prowlerBuild.addToRolePolicy(new statement.Ds().allow().toListAuthorizedApplications());
        prowlerBuild.addToRolePolicy(new statement.Ec2().allow().toGetEbsEncryptionByDefault());
        prowlerBuild.addToRolePolicy(new statement.Ecr().allow().toDescribeImageScanFindings().toDescribeImages().toDescribeRegistry());
        prowlerBuild.addToRolePolicy(new statement.Tag().allow().toGetTagKeys());
        prowlerBuild.addToRolePolicy(new statement.Lambda().allow().toGetFunction());
        prowlerBuild.addToRolePolicy(new statement.Glue().allow().toSearchTables().toGetConnections());
        prowlerBuild.addToRolePolicy(new statement.Apigateway().allow().toGET());
        prowlerBuild.addToRolePolicy(new iam.PolicyStatement({ actions: ['support:Describe*'], resources: ['*'] }));
        reportBucket.grantPut(prowlerBuild);
        const myRole = new iam.Role(this, 'MyRole', { assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com') });
        const prowlerStartBuildLambda = new lambda.Function(this, 'Lambda', {
            runtime: lambda.Runtime.PYTHON_3_6,
            timeout: core_1.Duration.seconds(120),
            handler: 'index.lambda_handler',
            code: lambda.Code.fromInline(`import boto3
import cfnresponse
from botocore.exceptions import ClientError
def lambda_handler(event,context):
  props = event['ResourceProperties']
  codebuild_client = boto3.client('codebuild')
  if (event['RequestType'] == 'Create' or event['RequestType'] == 'Update'):
    try:
        response = codebuild_client.start_build(projectName=props['Build'])
        print(response)
        print("Respond: SUCCESS")
        cfnresponse.send(event, context, cfnresponse.SUCCESS, {})
    except Exception as ex:
        print(ex.response['Error']['Message'])
        cfnresponse.send(event, context, cfnresponse.FAILED, ex.response)
      `),
        });
        prowlerStartBuildLambda.addToRolePolicy(new statement.Codebuild().allow().toStartBuild()); // onResource project ...
        const myProvider = new cr.Provider(this, 'MyProvider', {
            onEventHandler: prowlerStartBuildLambda,
            logRetention: this.logsRetentionInDays,
            role: myRole,
        });
        new core_1.CustomResource(this, 'Resource1', {
            serviceToken: myProvider.serviceToken,
            properties: {
                Build: prowlerBuild.projectName,
                RERUN_PROWLER: Boolean(this.node.tryGetContext('reRunProwler')) ? Date.now().toString() : '',
            },
        });
        if (this.enableScheduler) {
            const prowlerSchedulerLambda = new lambda.Function(this, 'ScheduleLambda', {
                runtime: lambda.Runtime.PYTHON_3_6,
                timeout: core_1.Duration.seconds(120),
                handler: 'index.lambda_handler',
                environment: {
                    buildName: prowlerBuild.projectName,
                },
                code: lambda.Code.fromInline(`import boto3
        import os
        def lambda_handler(event,context):
          codebuild_client = boto3.client('codebuild')
          print("Running Prowler scheduled!: " + os.environ['buildName'])
          project_name = os.environ['buildName']
          response = codebuild_client.start_build(projectName=project_name)
          print(response)
          print("Respond: SUCCESS")
        `),
            });
            new events.Rule(this, 'Rule', {
                description: 'A schedule for the Lambda function that triggers Prowler in CodeBuild.',
                targets: [new targets.LambdaFunction(prowlerSchedulerLambda)],
                schedule: events.Schedule.expression(this.prowlerScheduler || ''),
            });
        }
    }
}
exports.ProwlerAudit = ProwlerAudit;
_a = JSII_RTTI_SYMBOL_1;
ProwlerAudit[_a] = { fqn: "cdk-prowler.ProwlerAudit", version: "1.119.2" };
//# sourceMappingURL=data:application/json;base64,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