"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
/**
 * The record type.
 */
var RecordType;
(function (RecordType) {
    /**
     * route traffic to a resource, such as a web server, using an IPv4 address in dotted decimal
     * notation
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#AFormat
     */
    RecordType["A"] = "A";
    /**
     * route traffic to a resource, such as a web server, using an IPv6 address in colon-separated
     * hexadecimal format
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#AAAAFormat
     */
    RecordType["AAAA"] = "AAAA";
    /**
     * A CAA record specifies which certificate authorities (CAs) are allowed to issue certificates
     * for a domain or subdomain
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#CAAFormat
     */
    RecordType["CAA"] = "CAA";
    /**
     * A CNAME record maps DNS queries for the name of the current record, such as acme.example.com,
     * to another domain (example.com or example.net) or subdomain (acme.example.com or zenith.example.org).
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#CNAMEFormat
     */
    RecordType["CNAME"] = "CNAME";
    /**
     * An MX record specifies the names of your mail servers and, if you have two or more mail servers,
     * the priority order.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#MXFormat
     */
    RecordType["MX"] = "MX";
    /**
     * A Name Authority Pointer (NAPTR) is a type of record that is used by Dynamic Delegation Discovery
     * System (DDDS) applications to convert one value to another or to replace one value with another.
     * For example, one common use is to convert phone numbers into SIP URIs.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#NAPTRFormat
     */
    RecordType["NAPTR"] = "NAPTR";
    /**
     * An NS record identifies the name servers for the hosted zone
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#NSFormat
     */
    RecordType["NS"] = "NS";
    /**
     * A PTR record maps an IP address to the corresponding domain name.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#PTRFormat
     */
    RecordType["PTR"] = "PTR";
    /**
     * A start of authority (SOA) record provides information about a domain and the corresponding Amazon
     * Route 53 hosted zone
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#SOAFormat
     */
    RecordType["SOA"] = "SOA";
    /**
     * SPF records were formerly used to verify the identity of the sender of email messages.
     * Instead of an SPF record, we recommend that you create a TXT record that contains the applicable value.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#SPFFormat
     */
    RecordType["SPF"] = "SPF";
    /**
     * An SRV record Value element consists of four space-separated values. The first three values are
     * decimal numbers representing priority, weight, and port. The fourth value is a domain name.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#SRVFormat
     */
    RecordType["SRV"] = "SRV";
    /**
     * A TXT record contains one or more strings that are enclosed in double quotation marks (").
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#TXTFormat
     */
    RecordType["TXT"] = "TXT";
})(RecordType = exports.RecordType || (exports.RecordType = {}));
/**
 * Type union for a record that accepts multiple types of target.
 */
class RecordTarget {
    /**
     *
     * @param values correspond with the chosen record type (e.g. for 'A' Type, specify one ore more IP addresses)
     * @param aliasTarget alias for targets such as CloudFront distribution to route traffic to
     */
    constructor(values, aliasTarget) {
        this.values = values;
        this.aliasTarget = aliasTarget;
    }
    /**
     * Use string values as target.
     */
    static fromValues(...values) {
        return new RecordTarget(values);
    }
    /**
     * Use an alias as target.
     */
    static fromAlias(aliasTarget) {
        return new RecordTarget(undefined, aliasTarget);
    }
    /**
     * Use ip addresses as target.
     */
    static fromIpAddresses(...ipAddresses) {
        return RecordTarget.fromValues(...ipAddresses);
    }
}
exports.RecordTarget = RecordTarget;
/**
 * A record set.
 */
class RecordSet extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const ttl = props.target.aliasTarget ? undefined : ((props.ttl && props.ttl.toSeconds()) || 1800).toString();
        const recordSet = new route53_generated_1.CfnRecordSet(this, 'Resource', {
            hostedZoneId: props.zone.hostedZoneId,
            name: util_1.determineFullyQualifiedDomainName(props.recordName || props.zone.zoneName, props.zone),
            type: props.recordType,
            resourceRecords: props.target.values,
            aliasTarget: props.target.aliasTarget && props.target.aliasTarget.bind(this),
            ttl,
            comment: props.comment,
        });
        this.domainName = recordSet.ref;
    }
}
exports.RecordSet = RecordSet;
/**
 * Target for a DNS A Record
 *
 * @deprecated Use RecordTarget
 */
class AddressRecordTarget extends RecordTarget {
}
exports.AddressRecordTarget = AddressRecordTarget;
/**
 * A DNS A record
 *
 * @resource AWS::Route53::RecordSet
 */
class ARecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.A,
            target: props.target,
        });
    }
}
exports.ARecord = ARecord;
/**
 * A DNS AAAA record
 *
 * @resource AWS::Route53::RecordSet
 */
class AaaaRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.AAAA,
            target: props.target,
        });
    }
}
exports.AaaaRecord = AaaaRecord;
/**
 * A DNS CNAME record
 *
 * @resource AWS::Route53::RecordSet
 */
class CnameRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.CNAME,
            target: RecordTarget.fromValues(props.domainName),
        });
    }
}
exports.CnameRecord = CnameRecord;
/**
 * A DNS TXT record
 *
 * @resource AWS::Route53::RecordSet
 */
class TxtRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.TXT,
            target: RecordTarget.fromValues(...props.values.map(v => JSON.stringify(v))),
        });
    }
}
exports.TxtRecord = TxtRecord;
/**
 * A DNS SRV record
 *
 * @resource AWS::Route53::RecordSet
 */
class SrvRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.SRV,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.priority} ${v.weight} ${v.port} ${v.hostName}`)),
        });
    }
}
exports.SrvRecord = SrvRecord;
/**
 * The CAA tag.
 */
var CaaTag;
(function (CaaTag) {
    /**
     * Explicity authorizes a single certificate authority to issue a
     * certificate (any type) for the hostname.
     */
    CaaTag["ISSUE"] = "issue";
    /**
     * Explicity authorizes a single certificate authority to issue a
     * wildcard certificate (and only wildcard) for the hostname.
     */
    CaaTag["ISSUEWILD"] = "issuewild";
    /**
     * Specifies a URL to which a certificate authority may report policy
     * violations.
     */
    CaaTag["IODEF"] = "iodef";
})(CaaTag = exports.CaaTag || (exports.CaaTag = {}));
/**
 * A DNS CAA record
 *
 * @resource AWS::Route53::RecordSet
 */
class CaaRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.CAA,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.flag} ${v.tag} "${v.value}"`)),
        });
    }
}
exports.CaaRecord = CaaRecord;
/**
 * A DNS Amazon CAA record.
 *
 * A CAA record to restrict certificate authorities allowed
 * to issue certificates for a domain to Amazon only.
 *
 * @resource AWS::Route53::RecordSet
 */
class CaaAmazonRecord extends CaaRecord {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            values: [
                {
                    flag: 0,
                    tag: CaaTag.ISSUE,
                    value: 'amazon.com',
                },
            ],
        });
    }
}
exports.CaaAmazonRecord = CaaAmazonRecord;
/**
 * A DNS MX record
 *
 * @resource AWS::Route53::RecordSet
 */
class MxRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.MX,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.priority} ${v.hostName}`)),
        });
    }
}
exports.MxRecord = MxRecord;
/**
 * A record to delegate further lookups to a different set of name servers.
 */
class ZoneDelegationRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.NS,
            target: RecordTarget.fromValues(...core_1.Token.isUnresolved(props.nameServers)
                ? props.nameServers // Can't map a string-array token!
                : props.nameServers.map(ns => (core_1.Token.isUnresolved(ns) || ns.endsWith('.')) ? ns : `${ns}.`)),
            ttl: props.ttl || core_1.Duration.days(2),
        });
    }
}
exports.ZoneDelegationRecord = ZoneDelegationRecord;
//# sourceMappingURL=data:application/json;base64,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