import tempfile
import xml.etree.ElementTree as ET
from pathlib import Path

import pytest

from silkworm.pipelines import CSVPipeline, XMLPipeline
from silkworm.spiders import Spider


@pytest.mark.anyio("asyncio")
async def test_xml_pipeline_creates_valid_xml():
    with tempfile.TemporaryDirectory() as tmpdir:
        xml_path = Path(tmpdir) / "test.xml"
        pipeline = XMLPipeline(xml_path)
        spider = Spider()

        await pipeline.open(spider)
        await pipeline.process_item({"text": "Hello", "author": "John"}, spider)
        await pipeline.process_item({"text": "World", "author": "Jane"}, spider)
        await pipeline.close(spider)

        # Verify XML is valid
        tree = ET.parse(xml_path)
        root = tree.getroot()

        assert root.tag == "items"
        items = list(root)
        assert len(items) == 2

        # Check first item
        assert items[0].tag == "item"
        assert items[0].find("text").text == "Hello"
        assert items[0].find("author").text == "John"

        # Check second item
        assert items[1].tag == "item"
        assert items[1].find("text").text == "World"
        assert items[1].find("author").text == "Jane"


@pytest.mark.anyio("asyncio")
async def test_xml_pipeline_custom_elements():
    with tempfile.TemporaryDirectory() as tmpdir:
        xml_path = Path(tmpdir) / "test.xml"
        pipeline = XMLPipeline(xml_path, root_element="quotes", item_element="quote")
        spider = Spider()

        await pipeline.open(spider)
        await pipeline.process_item({"text": "Test"}, spider)
        await pipeline.close(spider)

        tree = ET.parse(xml_path)
        root = tree.getroot()

        assert root.tag == "quotes"
        assert root[0].tag == "quote"


@pytest.mark.anyio("asyncio")
async def test_xml_pipeline_handles_nested_dict():
    with tempfile.TemporaryDirectory() as tmpdir:
        xml_path = Path(tmpdir) / "test.xml"
        pipeline = XMLPipeline(xml_path)
        spider = Spider()

        await pipeline.open(spider)
        await pipeline.process_item(
            {"user": {"name": "Alice", "age": 30}, "active": True}, spider
        )
        await pipeline.close(spider)

        tree = ET.parse(xml_path)
        root = tree.getroot()
        item = root[0]

        assert item.find("user/name").text == "Alice"
        assert item.find("user/age").text == "30"
        assert item.find("active").text == "True"


@pytest.mark.anyio("asyncio")
async def test_xml_pipeline_handles_list():
    with tempfile.TemporaryDirectory() as tmpdir:
        xml_path = Path(tmpdir) / "test.xml"
        pipeline = XMLPipeline(xml_path)
        spider = Spider()

        await pipeline.open(spider)
        await pipeline.process_item({"tags": ["python", "web", "scraping"]}, spider)
        await pipeline.close(spider)

        tree = ET.parse(xml_path)
        root = tree.getroot()
        item = root[0]

        tags_elem = item.find("tags")
        tag_items = tags_elem.findall("item")
        assert len(tag_items) == 3
        assert tag_items[0].text == "python"
        assert tag_items[1].text == "web"
        assert tag_items[2].text == "scraping"


@pytest.mark.anyio("asyncio")
async def test_xml_pipeline_not_opened_raises_error():
    pipeline = XMLPipeline("test.xml")
    spider = Spider()

    with pytest.raises(RuntimeError, match="XMLPipeline not opened"):
        await pipeline.process_item({"test": "data"}, spider)


@pytest.mark.anyio("asyncio")
async def test_csv_pipeline_creates_valid_csv():
    with tempfile.TemporaryDirectory() as tmpdir:
        csv_path = Path(tmpdir) / "test.csv"
        pipeline = CSVPipeline(csv_path)
        spider = Spider()

        await pipeline.open(spider)
        await pipeline.process_item({"text": "Hello", "author": "John"}, spider)
        await pipeline.process_item({"text": "World", "author": "Jane"}, spider)
        await pipeline.close(spider)

        # Read and verify CSV
        content = csv_path.read_text()
        lines = content.strip().split("\n")

        assert len(lines) == 3  # header + 2 data rows
        # Check header has both fields (order may vary)
        header_fields = set(lines[0].split(","))
        assert header_fields == {"text", "author"}
        assert "Hello" in content
        assert "John" in content
        assert "World" in content
        assert "Jane" in content


@pytest.mark.anyio("asyncio")
async def test_csv_pipeline_with_custom_fieldnames():
    with tempfile.TemporaryDirectory() as tmpdir:
        csv_path = Path(tmpdir) / "test.csv"
        pipeline = CSVPipeline(csv_path, fieldnames=["author", "text"])
        spider = Spider()

        await pipeline.open(spider)
        await pipeline.process_item({"text": "Hello", "author": "John"}, spider)
        await pipeline.close(spider)

        content = csv_path.read_text()
        lines = content.strip().split("\n")

        # Verify header order matches custom fieldnames
        assert lines[0] == "author,text"
        assert "John,Hello" in content


@pytest.mark.anyio("asyncio")
async def test_csv_pipeline_flattens_nested_dict():
    with tempfile.TemporaryDirectory() as tmpdir:
        csv_path = Path(tmpdir) / "test.csv"
        pipeline = CSVPipeline(csv_path)
        spider = Spider()

        await pipeline.open(spider)
        await pipeline.process_item(
            {"name": "Alice", "address": {"city": "NYC", "zip": "10001"}}, spider
        )
        await pipeline.close(spider)

        content = csv_path.read_text()

        assert "address_city" in content
        assert "address_zip" in content
        assert "NYC" in content
        assert "10001" in content


@pytest.mark.anyio("asyncio")
async def test_csv_pipeline_converts_list_to_string():
    with tempfile.TemporaryDirectory() as tmpdir:
        csv_path = Path(tmpdir) / "test.csv"
        pipeline = CSVPipeline(csv_path)
        spider = Spider()

        await pipeline.open(spider)
        await pipeline.process_item(
            {"author": "John", "tags": ["python", "web", "scraping"]}, spider
        )
        await pipeline.close(spider)

        content = csv_path.read_text()

        assert "python, web, scraping" in content


@pytest.mark.anyio("asyncio")
async def test_csv_pipeline_not_opened_raises_error():
    pipeline = CSVPipeline("test.csv")
    spider = Spider()

    with pytest.raises(RuntimeError, match="CSVPipeline not opened"):
        await pipeline.process_item({"test": "data"}, spider)


@pytest.mark.anyio("asyncio")
async def test_csv_pipeline_handles_extra_fields():
    with tempfile.TemporaryDirectory() as tmpdir:
        csv_path = Path(tmpdir) / "test.csv"
        pipeline = CSVPipeline(csv_path, fieldnames=["author"])
        spider = Spider()

        await pipeline.open(spider)
        # Item has extra field "text" that's not in fieldnames
        await pipeline.process_item({"text": "Hello", "author": "John"}, spider)
        await pipeline.close(spider)

        content = csv_path.read_text()
        lines = content.strip().split("\n")

        # Only author should be in output
        assert lines[0] == "author"
        assert "John" in content
        # text should not be in output
        assert "Hello" not in content
