"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const policy_1 = require("./policy");
/**
 * Reference to a new or existing Amazon SQS queue
 */
class QueueBase extends core_1.Resource {
    /**
     * Adds a statement to the IAM resource policy associated with this queue.
     *
     * If this queue was created in this stack (`new Queue`), a queue policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the queue is improted (`Queue.import`), then this is a no-op.
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.QueuePolicy(this, 'Policy', { queues: [this] });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
        }
    }
    /**
     * Grant permissions to consume messages from a queue
     *
     * This will grant the following permissions:
     *
     *   - sqs:ChangeMessageVisibility
     *   - sqs:DeleteMessage
     *   - sqs:ReceiveMessage
     *   - sqs:GetQueueAttributes
     *   - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant consume rights to
     */
    grantConsumeMessages(grantee) {
        const ret = this.grant(grantee, 'sqs:ReceiveMessage', 'sqs:ChangeMessageVisibility', 'sqs:GetQueueUrl', 'sqs:DeleteMessage', 'sqs:GetQueueAttributes');
        if (this.encryptionMasterKey) {
            this.encryptionMasterKey.grantDecrypt(grantee);
        }
        return ret;
    }
    /**
     * Grant access to send messages to a queue to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - sqs:SendMessage
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantSendMessages(grantee) {
        const ret = this.grant(grantee, 'sqs:SendMessage', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
        if (this.encryptionMasterKey) {
            this.encryptionMasterKey.grantEncrypt(grantee);
        }
        return ret;
    }
    /**
     * Grant an IAM principal permissions to purge all messages from the queue.
     *
     * This will grant the following permissions:
     *
     *  - sqs:PurgeQueue
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantPurge(grantee) {
        return this.grant(grantee, 'sqs:PurgeQueue', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
    }
    /**
     * Grant the actions defined in queueActions to the identity Principal given
     * on this SQS queue resource.
     *
     * @param grantee Principal to grant right to
     * @param actions The actions to grant
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions,
            resourceArns: [this.queueArn],
            resource: this,
        });
    }
}
exports.QueueBase = QueueBase;
//# sourceMappingURL=data:application/json;base64,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