"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const sqs = require("../lib");
function testGrant(action, ...expectedActions) {
    const stack = new core_1.Stack();
    const queue = new sqs.Queue(stack, 'MyQueue');
    const principal = new iam.User(stack, 'User');
    action(queue, principal);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
        "PolicyDocument": {
            "Statement": [
                {
                    "Action": expectedActions,
                    "Effect": "Allow",
                    "Resource": {
                        "Fn::GetAtt": [
                            "MyQueueE6CA6235",
                            "Arn"
                        ]
                    }
                }
            ],
            "Version": "2012-10-17"
        }
    }));
}
module.exports = {
    'default properties'(test) {
        const stack = new core_1.Stack();
        const q = new sqs.Queue(stack, 'Queue');
        test.deepEqual(q.fifo, false);
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Queue4A7E3555": {
                    "Type": "AWS::SQS::Queue"
                }
            }
        });
        test.done();
    },
    'with a dead letter queue'(test) {
        const stack = new core_1.Stack();
        const dlq = new sqs.Queue(stack, 'DLQ');
        new sqs.Queue(stack, 'Queue', { deadLetterQueue: { queue: dlq, maxReceiveCount: 3 } });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "DLQ581697C4": {
                    "Type": "AWS::SQS::Queue"
                },
                "Queue4A7E3555": {
                    "Type": "AWS::SQS::Queue",
                    "Properties": {
                        "RedrivePolicy": {
                            "deadLetterTargetArn": {
                                "Fn::GetAtt": [
                                    "DLQ581697C4",
                                    "Arn"
                                ]
                            },
                            "maxReceiveCount": 3
                        }
                    }
                }
            }
        });
        test.done();
    },
    'addToPolicy will automatically create a policy for this queue'(test) {
        const stack = new core_1.Stack();
        const queue = new sqs.Queue(stack, 'MyQueue');
        queue.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['sqs:*'],
            principals: [new iam.ArnPrincipal('arn')]
        }));
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyQueueE6CA6235": {
                    "Type": "AWS::SQS::Queue"
                },
                "MyQueuePolicy6BBEDDAC": {
                    "Type": "AWS::SQS::QueuePolicy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:*",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "AWS": "arn"
                                    },
                                    "Resource": "*"
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "Queues": [
                            {
                                "Ref": "MyQueueE6CA6235"
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'export and import': {
        'importing works correctly'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            const imports = sqs.Queue.fromQueueArn(stack, 'Imported', 'arn:aws:sqs:us-east-1:123456789012:queue1');
            // THEN
            // "import" returns an IQueue bound to `Fn::ImportValue`s.
            test.deepEqual(stack.resolve(imports.queueArn), 'arn:aws:sqs:us-east-1:123456789012:queue1');
            test.deepEqual(stack.resolve(imports.queueUrl), { 'Fn::Join': ['', ['https://sqs.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/', { Ref: 'AWS::AccountId' }, '/queue1']] });
            test.deepEqual(stack.resolve(imports.queueName), 'queue1');
            test.done();
        },
        'importing fifo and standard queues are detected correctly'(test) {
            const stack = new core_1.Stack();
            const stdQueue = sqs.Queue.fromQueueArn(stack, 'StdQueue', 'arn:aws:sqs:us-east-1:123456789012:queue1');
            const fifoQueue = sqs.Queue.fromQueueArn(stack, 'FifoQueue', 'arn:aws:sqs:us-east-1:123456789012:queue2.fifo');
            test.deepEqual(stdQueue.fifo, false);
            test.deepEqual(fifoQueue.fifo, true);
            test.done();
        },
    },
    'grants': {
        'grantConsumeMessages'(test) {
            testGrant((q, p) => q.grantConsumeMessages(p), 'sqs:ReceiveMessage', 'sqs:ChangeMessageVisibility', 'sqs:GetQueueUrl', 'sqs:DeleteMessage', 'sqs:GetQueueAttributes');
            test.done();
        },
        'grantSendMessages'(test) {
            testGrant((q, p) => q.grantSendMessages(p), 'sqs:SendMessage', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
            test.done();
        },
        'grantPurge'(test) {
            testGrant((q, p) => q.grantPurge(p), 'sqs:PurgeQueue', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
            test.done();
        },
        'grant() is general purpose'(test) {
            testGrant((q, p) => q.grant(p, 'service:hello', 'service:world'), 'service:hello', 'service:world');
            test.done();
        },
        'grants also work on imported queues'(test) {
            const stack = new core_1.Stack();
            const queue = sqs.Queue.fromQueueAttributes(stack, 'Import', {
                queueArn: 'arn:aws:sqs:us-east-1:123456789012:queue1',
                queueUrl: 'https://queue-url'
            });
            const user = new iam.User(stack, 'User');
            queue.grantPurge(user);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": [
                                "sqs:PurgeQueue",
                                "sqs:GetQueueAttributes",
                                "sqs:GetQueueUrl"
                            ],
                            "Effect": "Allow",
                            "Resource": "arn:aws:sqs:us-east-1:123456789012:queue1"
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        }
    },
    'queue encryption': {
        'encryptionMasterKey can be set to a custom KMS key'(test) {
            const stack = new core_1.Stack();
            const key = new kms.Key(stack, 'CustomKey');
            const queue = new sqs.Queue(stack, 'Queue', { encryptionMasterKey: key });
            test.same(queue.encryptionMasterKey, key);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::Queue', {
                "KmsMasterKeyId": { "Fn::GetAtt": ["CustomKey1E6D0D07", "Arn"] }
            }));
            test.done();
        },
        'a kms key will be allocated if encryption = kms but a master key is not specified'(test) {
            const stack = new core_1.Stack();
            new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.KMS });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::Queue', {
                "KmsMasterKeyId": {
                    "Fn::GetAtt": [
                        "QueueKey39FCBAE6",
                        "Arn"
                    ]
                }
            }));
            test.done();
        },
        'it is possible to use a managed kms key'(test) {
            const stack = new core_1.Stack();
            new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.KMS_MANAGED });
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "Queue4A7E3555": {
                        "Type": "AWS::SQS::Queue",
                        "Properties": {
                            "KmsMasterKeyId": "alias/aws/sqs"
                        }
                    }
                }
            });
            test.done();
        },
        'grant also affects key on encrypted queue'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const queue = new sqs.Queue(stack, 'Queue', {
                encryption: sqs.QueueEncryption.KMS
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('someone')
            });
            // WHEN
            queue.grantSendMessages(role);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": [
                                "sqs:SendMessage",
                                "sqs:GetQueueAttributes",
                                "sqs:GetQueueUrl"
                            ],
                            "Effect": "Allow",
                            "Resource": { "Fn::GetAtt": ["Queue4A7E3555", "Arn"] }
                        },
                        {
                            "Action": [
                                "kms:Encrypt",
                                "kms:ReEncrypt*",
                                "kms:GenerateDataKey*"
                            ],
                            "Effect": "Allow",
                            "Resource": { "Fn::GetAtt": ["QueueKey39FCBAE6", "Arn"] }
                        }
                    ],
                    "Version": "2012-10-17"
                },
            }));
            test.done();
        },
    },
    'test ".fifo" suffixed queues register as fifo'(test) {
        const stack = new core_1.Stack();
        const queue = new sqs.Queue(stack, 'Queue', {
            queueName: 'MyQueue.fifo'
        });
        test.deepEqual(queue.fifo, true);
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Queue4A7E3555": {
                    "Type": "AWS::SQS::Queue",
                    "Properties": {
                        "QueueName": "MyQueue.fifo",
                        "FifoQueue": true
                    }
                }
            }
        });
        test.done();
    },
    'test a fifo queue is observed when the "fifo" property is specified'(test) {
        const stack = new core_1.Stack();
        const queue = new sqs.Queue(stack, 'Queue', {
            fifo: true
        });
        test.deepEqual(queue.fifo, true);
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Queue4A7E3555": {
                    "Type": "AWS::SQS::Queue",
                    "Properties": {
                        "FifoQueue": true
                    }
                }
            }
        });
        test.done();
    },
    'test metrics'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const queue = new sqs.Queue(stack, 'Queue');
        // THEN
        test.deepEqual(stack.resolve(queue.metricNumberOfMessagesSent()), {
            dimensions: { QueueName: { 'Fn::GetAtt': ['Queue4A7E3555', 'QueueName'] } },
            namespace: 'AWS/SQS',
            metricName: 'NumberOfMessagesSent',
            period: core_1.Duration.minutes(5),
            statistic: 'Sum'
        });
        test.deepEqual(stack.resolve(queue.metricSentMessageSize()), {
            dimensions: { QueueName: { 'Fn::GetAtt': ['Queue4A7E3555', 'QueueName'] } },
            namespace: 'AWS/SQS',
            metricName: 'SentMessageSize',
            period: core_1.Duration.minutes(5),
            statistic: 'Average'
        });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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