# -*- coding: utf-8 -*-
#
# Copyright 2018-2020- Swiss Data Science Center (SDSC)
# A partnership between École Polytechnique Fédérale de Lausanne (EPFL) and
# Eidgenössische Technische Hochschule Zürich (ETHZ).
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
r"""Recreate files created by the "run" command.

Recreating files
~~~~~~~~~~~~~~~~

Assume you have run a step 2 that uses a stochastic algorithm, so each run
will be slightly different. The goal is to regenerate output ``C`` several
times to compare the output. In this situation it is not possible to simply
call :ref:`cli-update` since the input file ``A`` has not been modified
after the execution of step 2.

.. code-block:: text

    A-[step 1]-B-[step 2*]-C

Recreate a specific output file by running:

  .. code-block:: console

     $ renku rerun C

If you would like to recreate a file which was one of several produced by
a tool, then these files must be recreated as well. See the explanation in
:ref:`updating siblings <cli-update-with-siblings>`.
"""

import os
import sys
import uuid
from pathlib import Path

import click
from git import Actor

from renku.core.commands.client import pass_local_client
from renku.core.commands.cwl_runner import execute
from renku.core.commands.graph import Graph
from renku.core.commands.options import option_siblings
from renku.core.models.locals import with_reference
from renku.core.models.provenance.activities import WorkflowRun
from renku.core.models.workflow.converters.cwl import CWLConverter
from renku.version import __version__, version_url


def show_inputs(client, workflow):
    """Show workflow inputs and exit."""
    for input_ in workflow.inputs:
        click.echo("{id}: {default}".format(id=input_._id, default=input_.consumes.path,))
    sys.exit(0)


def edit_inputs(client, workflow):
    """Edit workflow inputs."""
    for input_ in workflow.inputs:
        new_path = click.prompt("{0._id}".format(input_), default=input_.consumes.path,)
        input_.consumes.path = str(Path(os.path.abspath(new_path)).relative_to(client.path))

    for step in workflow.subprocesses:
        for argument in step.process.arguments:
            argument.value = click.prompt("{0._id}".format(argument), default=argument.value,)

    return workflow


@click.command()
@click.option("--revision", default="HEAD")
@click.option(
    "--from",
    "roots",
    type=click.Path(exists=True, dir_okay=False),
    multiple=True,
    help="Start an execution from this file.",
)
@option_siblings
@click.option(
    "--default-inputs", "inputs", default=True, flag_value=lambda _, workflow: workflow, help="Use default inputs.",
)
@click.option(
    "--show-inputs", "inputs", flag_value=show_inputs, help=show_inputs.__doc__,
)
@click.option(
    "--edit-inputs", "inputs", flag_value=edit_inputs, help=edit_inputs.__doc__,
)
@click.argument(
    "paths", type=click.Path(exists=True, dir_okay=True), nargs=-1, required=True,
)
@pass_local_client(
    clean=True, requires_migration=True, commit=True,
)
def rerun(client, revision, roots, siblings, inputs, paths):
    """Recreate files generated by a sequence of ``run`` commands."""
    graph = Graph(client)
    outputs = graph.build(paths=paths, revision=revision)

    # Check or extend siblings of outputs.
    outputs = siblings(graph, outputs)
    output_paths = {node.path for node in outputs}

    # Normalize and check all starting paths.
    roots = {graph.normalize_path(root) for root in roots}
    output_paths -= roots
    outputs = [o for o in outputs if o.path not in roots]

    # Generate workflow and check inputs.
    # NOTE The workflow creation is done before opening a new file.
    workflow = inputs(client, graph.as_workflow(input_paths=roots, output_paths=output_paths, outputs=outputs,))

    wf, path = CWLConverter.convert(workflow, client)

    # Don't compute paths if storage is disabled.
    if client.check_external_storage():
        # Make sure all inputs are pulled from a storage.
        paths_ = (i.consumes.path for i in workflow.inputs)
        client.pull_paths_from_storage(*paths_)

    # Execute the workflow and relocate all output files.
    # FIXME get new output paths for edited tools
    # output_paths = {path for _, path in workflow.iter_output_files()}
    execute(
        client, path, output_paths=output_paths,
    )

    paths = [o.produces.path for o in workflow.outputs]

    client.repo.git.add(*paths)

    if client.repo.is_dirty():
        commit_msg = ("renku rerun: " "committing {} newly added files").format(len(paths))

        committer = Actor("renku {0}".format(__version__), version_url)

        client.repo.index.commit(
            commit_msg, committer=committer, skip_hooks=True,
        )

    workflow_name = "{0}_rerun.yaml".format(uuid.uuid4().hex)

    path = client.workflow_path / workflow_name

    workflow.update_id_and_label_from_commit_path(client, client.repo.head.commit, path)

    with with_reference(path):
        run = WorkflowRun.from_run(workflow, client, path)
        run.to_yaml()
        client.add_to_activity_index(run)
