import gzip
import json
from io import BytesIO
from typing import List, Optional, Tuple

from cortex_xdr_client.api.authentication import Authentication
from cortex_xdr_client.api.base_api import BaseAPI
from cortex_xdr_client.api.models.exceptions import InvalidResponseException, UnsuccessfulQueryStatusException
from cortex_xdr_client.api.models.filters import new_request_data


class XQLAPI(BaseAPI):
    def __init__(self, auth: Authentication, fqdn: str, timeout: Tuple[int, int]) -> None:
        super(XQLAPI, self).__init__(auth, fqdn, "xql", timeout)

    def _get_start_xql_filter(self, query: str, time_period: int = None, from_date: int = None,
                              to_date: int = None, tenants: List[str] = None, params: dict = {}) -> dict:
        filter = {"query": query}
        timeframe = {}
        if tenants is not None and len(tenants) > 0:
            filter["tenants"] = tenants
        if from_date is not None and to_date is not None:
            timeframe = {"from": from_date, "to": to_date}
        elif time_period is not None:
            timeframe = {"relativeTime": time_period}
        filter["timeframe"] = timeframe
        return self.extend(filter, params)

    def _get_xql_results_filter(self, query_id: str, limit: int = None, params: dict = {}) -> dict:
        filter = {"query_id": query_id}
        if limit is not None and limit <= 1000:
            filter["limit"] = limit
        return self.extend(filter, params)

    # https://docs.paloaltonetworks.com/cortex/cortex-xdr/cortex-xdr-api/cortex-xdr-apis/xql-apis/start-xql-query.html
    def start_xql_query(self, query: str, time_period: int = None, from_date: int = None,
                        to_date: int = None, tenants: List[str] = None, params: dict = {}) -> Optional[str]:
        """
        Starts an XQL Query.
        :param query: String of the XQL query.
        :param time_period: Relative Unix timestamp representing the last X hours.
        :param from_date: Absolute Unix timestamp representing a date
        :param to_date: Absolute Unix timestamp representing a date
        :param tenants: List of strings used for running APIs on local and Managed Security tenants.
        :param params: Dictionary of parameters to be passed to the request data
        :return: String representing the unique ID generate by the response to Start XQL Query API.
        """
        """Starts an XQL Query and returns the ID of the query. """
        filters = self._get_start_xql_filter(query, time_period, from_date, to_date, tenants, params)
        request_data = new_request_data(other=filters)
        response = self._call(call_name="start_xql_query", json_value=request_data)
        resp_json = response.json()
        if "reply" not in resp_json:
            raise InvalidResponseException(response, ["reply"])
        return resp_json["reply"]

    # https://docs.paloaltonetworks.com/cortex/cortex-xdr/cortex-xdr-api/cortex-xdr-apis/xql-apis/get-xql-query-results.html
    def get_query_results(self, query_id: str, limit: int = None, params: dict = {}) -> Optional[dict]:
        """
        Returns the results of an XQL Query.
        :param query_id: teger representing the unique execution ID generated by the response to Start an XQL Query API
        :param limit: Integer representing the maximum number of results to return. Max 1000.
        :param params: Dictionary of parameters to be passed to the request data
        :return: Dictionary of results
        """
        filters = self._get_xql_results_filter(query_id, limit, params)
        request_data = new_request_data(other=filters)
        response = self._call(call_name="get_query_results", json_value=request_data)
        resp_json = response.json()
        if "reply" not in resp_json:
            raise InvalidResponseException(response, ["reply"])

        reply = resp_json["reply"]
        if "number_of_results" not in reply:
            raise InvalidResponseException(response, ["number_of_results"])

        if "status" not in reply:
            raise InvalidResponseException(response, ["status"])

        if "SUCCESS" != reply["status"]:
            raise UnsuccessfulQueryStatusException(reply["status"])

        if reply["number_of_results"] <= 1000:
            return reply["results"]

        stream_id = reply["results"]["stream_id"]
        return self.get_query_results_stream(stream_id)

    # https://docs.paloaltonetworks.com/cortex/cortex-xdr/cortex-xdr-api/cortex-xdr-apis/xql-apis/get-xql-query-exported-data.html
    def get_query_results_stream(self, stream_id: str) -> Optional[dict]:
        """
        Returns the results of an XQL Query.
        :param stream_id: Integer representing the unique ID generate by the response to Get XQL Query Results API.
        :return:  Dictionary of results
        """
        request_data = {
            "request_data": {
                "stream_id":          stream_id,
                "is_gzip_compressed": True,
            }
        }
        response = self._call(call_name="get_query_results_stream", json_value=request_data,
                              header_params={"Accept-Encoding": "gzip"})
        buffer = BytesIO(response.content)
        data = gzip.GzipFile(fileobj=buffer).read().decode("utf-8")
        logs = [json.loads(line) for line in data.splitlines() if line.strip() != ""]
        return {"data": logs}
