"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeletePhoneNumber = exports.CreatePhoneNumber = void 0;
/* eslint-disable import/no-extraneous-dependencies */
const client_chime_sdk_voice_1 = require("@aws-sdk/client-chime-sdk-voice");
const client_ssm_1 = require("@aws-sdk/client-ssm");
const chimeSDKVoiceClient = new client_chime_sdk_voice_1.ChimeSDKVoiceClient({
    region: 'us-east-1',
});
const ssmClient = new client_ssm_1.SSMClient({ region: process.env.AWS_REGION });
let searchAvailableNumbersParam;
let searchAvailablePhoneNumberResponse;
let createPhoneNumberOrderResponse;
let checkPhoneNumberOrderCount = 0;
let getParameterCommandOutput;
let getPhoneNumberResponse;
exports.CreatePhoneNumber = async (uid, props) => {
    searchAvailableNumbersParam = {
        ...(props.phoneAreaCode && { AreaCode: props.phoneAreaCode }),
        ...(props.phoneState && { State: props.phoneState }),
        ...(props.phoneCity && { City: props.phoneCity }),
        ...(props.phoneCountry && { Country: props.phoneCountry }),
        ...(props.phoneNumberTollFreePrefix && {
            TollFreePrefix: props.phoneNumberTollFreePrefix,
        }),
    };
    console.info(`Search Available Numbers: ${JSON.stringify(searchAvailableNumbersParam)}`);
    try {
        searchAvailablePhoneNumberResponse = await chimeSDKVoiceClient.send(new client_chime_sdk_voice_1.SearchAvailablePhoneNumbersCommand(searchAvailableNumbersParam));
    }
    catch (error) {
        if (error instanceof Error) {
            console.error(error);
            throw error;
        }
    }
    if (searchAvailablePhoneNumberResponse &&
        searchAvailablePhoneNumberResponse.E164PhoneNumbers &&
        searchAvailablePhoneNumberResponse.E164PhoneNumbers.length === 0) {
        throw new Error('No numbers were found with this search parameters.  Please try a different search.');
    }
    try {
        createPhoneNumberOrderResponse = await chimeSDKVoiceClient.send(new client_chime_sdk_voice_1.CreatePhoneNumberOrderCommand({
            ProductType: props.phoneProductType,
            E164PhoneNumbers: [
                searchAvailablePhoneNumberResponse.E164PhoneNumbers[0],
            ],
        }));
    }
    catch (error) {
        if (error instanceof Error) {
            console.error(error);
            throw error;
        }
    }
    while ((await checkPhoneNumber(createPhoneNumberOrderResponse.PhoneNumberOrder.PhoneNumberOrderId)) !== 'Successful') {
        checkPhoneNumberOrderCount++;
        if (checkPhoneNumberOrderCount > 10) {
            throw new Error('Could not get a valid phone number from Amazon Chime SDK');
        }
        await sleep(10000);
    }
    try {
        await ssmClient.send(new client_ssm_1.PutParameterCommand({
            Name: '/chime/phoneNumber' + uid,
            Value: searchAvailablePhoneNumberResponse.E164PhoneNumbers[0],
            Description: 'Phone Number',
            Overwrite: true,
            Type: 'String',
        }));
    }
    catch (error) {
        if (error instanceof Error) {
            console.error(error);
            throw error;
        }
    }
    return {
        phoneNumber: searchAvailablePhoneNumberResponse.E164PhoneNumbers[0],
    };
};
exports.DeletePhoneNumber = async (uid) => {
    try {
        getParameterCommandOutput = await ssmClient.send(new client_ssm_1.GetParameterCommand({ Name: '/chime/phoneNumber' + uid }));
    }
    catch (error) {
        if (error instanceof Error) {
            console.error(error);
            throw error;
        }
    }
    try {
        getPhoneNumberResponse = await chimeSDKVoiceClient.send(new client_chime_sdk_voice_1.GetPhoneNumberCommand({
            PhoneNumberId: getParameterCommandOutput.Parameter.Value,
        }));
    }
    catch (error) {
        if (error instanceof Error) {
            console.error(error);
            throw error;
        }
    }
    try {
        if (getPhoneNumberResponse.PhoneNumber.Status === 'Assigned') {
            if (getPhoneNumberResponse.PhoneNumber.ProductType === 'VoiceConnector') {
                await chimeSDKVoiceClient.send(new client_chime_sdk_voice_1.DisassociatePhoneNumbersFromVoiceConnectorCommand({
                    E164PhoneNumbers: [
                        getPhoneNumberResponse.PhoneNumber.E164PhoneNumber,
                    ],
                    VoiceConnectorId: getPhoneNumberResponse.PhoneNumber.Associations[0].Value,
                }));
            }
            else if (getPhoneNumberResponse.PhoneNumber.ProductType ===
                'SipMediaApplication') {
                await chimeSDKVoiceClient.send(new client_chime_sdk_voice_1.UpdateSipRuleCommand({
                    Name: getPhoneNumberResponse.PhoneNumber.E164PhoneNumber,
                    SipRuleId: getPhoneNumberResponse.PhoneNumber.Associations[0].Value,
                    Disabled: true,
                }));
            }
            else {
                throw new Error('Could not disassociate number');
            }
        }
    }
    catch (error) {
        if (error instanceof Error) {
            console.error(error);
            throw error;
        }
    }
    await sleep(10000);
    try {
        await chimeSDKVoiceClient.send(new client_chime_sdk_voice_1.DeletePhoneNumberCommand({
            PhoneNumberId: getParameterCommandOutput.Parameter.Value,
        }));
        await ssmClient.send(new client_ssm_1.DeleteParameterCommand({ Name: '/chime/phoneNumber' + uid }));
    }
    catch (error) { }
};
async function checkPhoneNumber(phoneOrderId) {
    try {
        const getPhoneNumberOrderResponse = await chimeSDKVoiceClient.send(new client_chime_sdk_voice_1.GetPhoneNumberOrderCommand({ PhoneNumberOrderId: phoneOrderId }));
        console.info(`Get Phone Number Order: ${JSON.stringify(getPhoneNumberOrderResponse)}`);
        return getPhoneNumberOrderResponse.PhoneNumberOrder?.Status;
    }
    catch (error) {
        if (error instanceof Error) {
            console.error(error);
            throw error;
        }
        return false;
    }
}
const sleep = (ms) => new Promise((r) => setTimeout(r, ms));
//# sourceMappingURL=data:application/json;base64,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