#     Copyright 2016-present CERN – European Organization for Nuclear Research
#
#     Licensed under the Apache License, Version 2.0 (the "License");
#     you may not use this file except in compliance with the License.
#     You may obtain a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#     Unless required by applicable law or agreed to in writing, software
#     distributed under the License is distributed on an "AS IS" BASIS,
#     WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#     See the License for the specific language governing permissions and
#     limitations under the License.

import abc
from typing import List, Optional

from qf_lib.backtesting.signals.signal import Signal
from qf_lib.common.tickers.tickers import Ticker
from qf_lib.containers.dataframe.qf_dataframe import QFDataFrame
from qf_lib.containers.series.qf_series import QFSeries


class SignalsRegister(metaclass=abc.ABCMeta):
    """
    Used to save signals processed by the Position Sizer and to be able to later to analyze all signals generated by
    the model and to present them in a readable form.
    """
    @abc.abstractmethod
    def save_signals(self, signals: List[Signal]):
        raise NotImplementedError()

    @abc.abstractmethod
    def get_signals(self) -> QFDataFrame:
        """
        Returns a QFDataFrame with all generated signals. The columns names consist of ticker names (in the following
        form: NameOfTicker@Model) and the rows are indexed by the time of signals generation (Signal objects do not
        store the timestamp).

        Returns
        --------
        QFDataFrame
            QFDataFrame with all generated signals
        """
        raise NotImplementedError()

    @abc.abstractmethod
    def get_signals_for_ticker(self, ticker: Optional[Ticker], alpha_model=None) -> QFSeries:
        """
        Returns a QFSeries with all generated signals for the given ticker. In case of a future ticker, it returns
        signals that were generated for any specific ticker, that belongs to the futures chain.

        Parameters
        ----------
        ticker: Ticker
            ticker for which the data frame should be generated
        alpha_model
            alpha model for which the generated signals should be returned (it checks whether the name of the
            alpha model matches the one in the signals using the str function)

        Returns
        --------
        QFDataFrame
            QFDataFrame with all generated signals
        """
        raise NotImplementedError()

    def _generate_ticker_name(self, signal: Signal) -> str:
        ticker = signal.ticker
        model_name = "" if signal.alpha_model is None else str(signal.alpha_model)
        return ticker.name + "@" + model_name
