"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubOIDCStack = exports.TokenActions = exports.ProviderUrl = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const base_1 = require("./base");
const utils_1 = require("../common/utils");
var ProviderUrl;
(function (ProviderUrl) {
    ProviderUrl["GITHUB"] = "https://token.actions.githubusercontent.com";
})(ProviderUrl = exports.ProviderUrl || (exports.ProviderUrl = {}));
var TokenActions;
(function (TokenActions) {
    TokenActions[TokenActions["ALL"] = 0] = "ALL";
    TokenActions[TokenActions["ALL_BRANCH"] = 1] = "ALL_BRANCH";
    TokenActions[TokenActions["ALL_TAGS"] = 2] = "ALL_TAGS";
    TokenActions[TokenActions["CUSTOM"] = 3] = "CUSTOM";
})(TokenActions = exports.TokenActions || (exports.TokenActions = {}));
class GithubOIDCStack extends base_1.BaseStack {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.githubUser = props.githubUser;
        this.githubRepository = props.githubRepository;
        this.tokenAction = props.tokenAction;
        let token = this.createTokenAction(props.tokenAction, props.githubUser, props.githubRepository, props.tokenActionCustom);
        this.oidcRole = this.createOidcRole(ProviderUrl.GITHUB, token);
        this.cdkBootstrapRole = this.createCdkBootstrapRole();
        this.cdkDeployRoleManagedPolicies = props.cdkDeployRoleManagedPolicies;
        this.cdkDeployRolePolicyStatements = props.cdkDeployRolePolicyStatements;
        this.cdkDeployRole = this.createCdkDeployRole(this.cdkDeployRoleManagedPolicies, this.cdkDeployRolePolicyStatements);
    }
    createTokenAction(tokenAction, githubUser, githubRepository, tokenActionCustom) {
        if (tokenAction === TokenActions.ALL) {
            return `repo:${githubUser}/*`;
        }
        if (tokenAction === TokenActions.ALL_BRANCH) {
            return `repo:${githubUser}/${githubRepository}:ref:refs/heads/*`;
        }
        if (tokenAction === TokenActions.ALL_TAGS) {
            return `repo:${githubUser}/${githubRepository}:ref:refs/tags/*`;
        }
        if (tokenAction === TokenActions.CUSTOM) {
            return `repo:${githubUser}/${githubRepository}:ref:refs/${tokenActionCustom}`;
        }
        throw new Error('tokenAction not found');
    }
    createCdkDeployRole(managed_policies, policy_statements) {
        let basePolicy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: [`arn:aws:iam::${this.account}:role/cdk-*`],
                    actions: [
                        'sts:AssumeRole',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: [`arn:aws:iam::${this.account}:role/cdk-*`],
                    actions: [
                        'iam:PassRole',
                        'iam:CreateRole',
                        'iam:DeleteRole',
                        'iam:UpdateRole',
                        'iam:DetachRolePolicy',
                        'iam:AttachRolePolicy',
                        'iam:DeleteRolePolicy',
                        'iam:PutRolePolicy',
                        'ssm:GetParameters',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
            ],
        });
        let role = new iam.Role(this, 'cdk-oidc-deploy-role', {
            roleName: 'cdk-oidc-deploy-role',
            assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
            inlinePolicies: {
                CDKDeployBasePolicy: basePolicy,
            },
        });
        if (policy_statements) {
            for (let index = 0; index < policy_statements.length; index++) {
                role.addToPolicy(policy_statements[index]);
            }
        }
        if (managed_policies) {
            for (let index = 0; index < managed_policies.length; index++) {
                role.addManagedPolicy(managed_policies[index]);
            }
        }
        new aws_cdk_lib_1.CfnOutput(this, 'cdk-oidc-deploy-role-output', {
            value: role.roleArn,
            description: 'Role for cdk deploy role arn',
            exportName: 'cdk-oidc-deploy-role',
        });
        utils_1.addBaseTags(role);
        return role;
    }
    createCdkBootstrapRole() {
        let policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: [`arn:aws:iam::${this.account}:role/cdk-*`],
                    actions: [
                        'sts:AssumeRole',
                        'iam:*Role*',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: [`arn:aws:cloudformation:${this.region}:${this.account}:stack/CDKToolkit/*`],
                    actions: [
                        'cloudformation:*',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: [
                        's3:*',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: [`arn:aws:ecr:${this.region}:${this.account}:repository/cdk-*`],
                    actions: [
                        'ecr:SetRepositoryPolicy',
                        'ecr:GetLifecyclePolicy',
                        'ecr:PutImageScanningConfiguration',
                        'ecr:DescribeRepositories',
                        'ecr:CreateRepository',
                        'ecr:DeleteRepository',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: [`arn:aws:ssm:${this.region}:${this.account}:parameter/cdk-bootstrap/*`],
                    actions: [
                        'ssm:GetParameter*',
                        'ssm:PutParameter*',
                        'ssm:DeleteParameter*',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
            ],
        });
        let role = new iam.Role(this, 'cdk-oidc-bootstrap-role', {
            roleName: 'cdk-oidc-bootstrap-role',
            assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
            inlinePolicies: {
                CDKBootstrapPolicy: policy,
            },
        });
        new aws_cdk_lib_1.CfnOutput(this, 'cdk-oidc-bootstrap-role-output', {
            value: role.roleArn,
            description: 'Role for cdk bootstrap role arn',
            exportName: 'cdk-oidc-bootstrap-role',
        });
        utils_1.addBaseTags(role);
        return role;
    }
    createOidcRole(providerUrl, token) {
        let oidcProvider = new iam.OpenIdConnectProvider(this, 'oidc-provider', {
            url: providerUrl,
            clientIds: ['sts.amazonaws.com'],
        });
        let role = new iam.Role(this, 'oidc-role', {
            roleName: 'github-oidc-workflow-role',
            assumedBy: new iam.WebIdentityPrincipal(oidcProvider.openIdConnectProviderArn, {
                StringLike: {
                    'token.actions.githubusercontent.com:sub': token,
                },
            }),
        });
        role.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'iam:PassRole',
                'ssm:GetParameter*',
                'cloudformation:*',
                's3:*',
                'ecr:*',
            ],
            effect: iam.Effect.ALLOW,
        }));
        role.addToPolicy(new iam.PolicyStatement({
            resources: [`arn:aws:iam::${this.account}:role/cdk-*`],
            actions: ['sts:AssumeRole'],
            effect: iam.Effect.ALLOW,
        }));
        new aws_cdk_lib_1.CfnOutput(this, 'github-oidc-workflow-role-output', {
            value: role.roleArn,
            description: 'Role for OIDC Github Workflow role arn',
            exportName: 'github-oidc-workflow-role',
        });
        utils_1.addBaseTags(role);
        return role;
    }
}
exports.GithubOIDCStack = GithubOIDCStack;
_a = JSII_RTTI_SYMBOL_1;
GithubOIDCStack[_a] = { fqn: "neulabs-cdk-constructs.stacks.GithubOIDCStack", version: "0.1.10" };
//# sourceMappingURL=data:application/json;base64,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