from dataclasses import dataclass
from os import mkdir
from os.path import exists
from time import sleep

from requests import get
from requests.exceptions import ConnectionError

from modules.logger import banner
from modules.random_user_agent import random_user_agent
from modules.utils import get_terminal_width


@dataclass
class ExploitInfo:
    Platform: str
    PublishDate: str
    Type: str
    ExploitDBID: int
    Author: str
    Metasploit: bool
    Verified: bool
    Link: str


def GetExploitInfo(CVEID, log) -> list[ExploitInfo]:
    sleep(0.75)
    try:
        apidata = get(
            f"https://www.exploit-db.com/search?cve={CVEID}",
            headers={
                "X-Requested-With": "XMLHttpRequest",
                "User-Agent": next(random_user_agent(log)),
            },
        ).json()
    except ConnectionError:
        log.logger(
            "error",
            "Connection error raised while trying"
            + f" to fetch information about: {CVEID}",
        )
        return []
    except Exception as e:
        log.logger("error", f"An error occured while parsing API response.")
        return []
    else:
        ExploitInfos = []
        for exploit in apidata["data"]:
            Exploit = ExploitInfo(
                Platform=exploit["platform_id"],
                PublishDate=exploit["date_published"],
                Type=exploit["type_id"],
                ExploitDBID=int(exploit["id"]),
                Author=exploit["author"]["name"],
                Metasploit=exploit["author"]["name"] == "Metasploit",
                Verified=exploit["verified"] == "1",
                Link=f"https://www.exploit-db.com/download/{exploit['id']}",
            )
            ExploitInfos.append(Exploit)

        return ExploitInfos


def GetExploitContents(ExploitLink, log) -> tuple:
    sleep(0.75)
    user_agent = next(random_user_agent(log))
    try:
        apiresponse = get(
            ExploitLink,
            headers={
                "X-Requested-With": "XMLHttpRequest",
                "User-Agent": user_agent,
            },
        )
        content = apiresponse.content
        filename = apiresponse.headers["Content-Disposition"].lstrip(
            'attachment; filename="'
        )
    except ConnectionError:
        log.logger(
            "error", f"Connection error raised while trying to fetch: {ExploitLink}"
        )
        return None, None
    except KeyError:
        log.logger(
            "error", f"Unable to retrieve contents of {ExploitLink} {user_agent}"
        )
        return None, None
    else:
        return content, filename


def GetExploitAsFile(vulnerability, log, console, status) -> None:
    SoftwareName = vulnerability.title
    CVEs = vulnerability.CVEs
    term_width = get_terminal_width()

    if not exists("exploits"):
        mkdir("exploits")

    printed_software = []
    for CVE in CVEs:
        Exploits = GetExploitInfo(CVE, log)
        if len(Exploits) == 0:
            continue
        status.stop()
        if SoftwareName not in printed_software:
            console.print(f"┌─[yellow][ {SoftwareName} ][/yellow]\n│")
            printed_software.append(SoftwareName)

        console.print(f"│\n├─────┤ [red]{str(CVE)}[/red]\n│")

        for exploit in Exploits:
            content, filename = GetExploitContents(exploit.Link, log)
            if content is None:
                continue

            if not exists(f"exploits/{SoftwareName}"):
                mkdir(f"exploits/{SoftwareName}")

            if not exists(f"exploits/{SoftwareName}/{CVE}"):
                mkdir(f"exploits/{SoftwareName}/{CVE}")

            with open(f"exploits/{SoftwareName}/{CVE}/{filename}", "wb") as exploitfile:
                console.print(
                    f"├──────────# [white]exploits/{SoftwareName}/{CVE}/{filename}[/white]\n"
                    + f"│\t\t [cyan]Platform: [/cyan] {exploit.Platform}\n"
                    + f"│\t\t [cyan]Type: [/cyan] {exploit.Type}\n"
                    + f"│\t\t [cyan]Author: [/cyan] {exploit.Author}\n"
                    + f"│\t\t [cyan]Date: [/cyan] [bright_cyan]{exploit.PublishDate}[/bright_cyan]\n"
                    + f"│\t\t [cyan]Metasploit: [/cyan] {exploit.Metasploit}\n"
                    + f"│\t\t [cyan]Verified: [/cyan]{exploit.Verified}\n"
                    + f"│\t\t [cyan]Link: [/cyan] {exploit.Link}\n│"
                )
                exploitfile.write(content)

    if SoftwareName in printed_software:
        console.print("└" + "─" * (term_width - 1) + "\n")


def GetExploitsFromArray(VulnsArray, log, console, console2, target=None) -> None:
    if target:
        banner(f"Downloading exploits for {target}...", "blue", console)
    else:
        banner(f"Downloading exploits...", "blue", console)

    with console2.status(
        "[red]Downloading exploits ...[/red]", spinner="bouncingBar"
    ) as status:
        for vulnerability in VulnsArray:
            status.start()
            status.update(
                f"[white]Downloading exploits for[/white] "
                + f"[red]{vulnerability.title}[/red] [white]...[/white]"
            )
            try:
                GetExploitAsFile(vulnerability, log, console, status)
            except KeyboardInterrupt:
                log.logger("warning", f"Skipping exploits for {vulnerability.title}")
