from lemur.common.utils import parse_certificate

VALID_USER_HEADER_TOKEN = {
    "Authorization": "Basic "
    + "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpYXQiOjE1MjE2NTIwMjIsImV4cCI6MjM4NTY1MjAyMiwic3ViIjoxfQ.uK4PZjVAs0gt6_9h2EkYkKd64nFXdOq-rHsJZzeQicc",
    "Content-Type": "application/json",
}


VALID_ADMIN_HEADER_TOKEN = {
    "Authorization": "Basic "
    + "eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9.eyJpYXQiOjE1MjE2NTE2NjMsInN1YiI6MiwiYWlkIjoxfQ.wyf5PkQNcggLrMFqxDfzjY-GWPw_XsuWvU2GmQaC5sg",
    "Content-Type": "application/json",
}


VALID_ADMIN_API_TOKEN = {
    "Authorization": "Basic "
    + "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJzdWIiOjIsImFpZCI6MSwiaWF0IjoxNDM1MjMzMzY5fQ.umW0I_oh4MVZ2qrClzj9SfYnQl6cd0HGzh9EwkDW60I",
    "Content-Type": "application/json",
}


#: CN=LemurTrust Unittests Root CA 2018
ROOTCA_CERT_STR = """\
-----BEGIN CERTIFICATE-----
MIIEFjCCAv6gAwIBAgIQbIbX/Ap0Roqzf5HeN5akmzANBgkqhkiG9w0BAQsFADCB
pDEqMCgGA1UEAwwhTGVtdXJUcnVzdCBVbml0dGVzdHMgUm9vdCBDQSAyMDE4MSMw
IQYDVQQKDBpMZW11clRydXN0IEVudGVycHJpc2VzIEx0ZDEmMCQGA1UECwwdVW5p
dHRlc3RpbmcgT3BlcmF0aW9ucyBDZW50ZXIxCzAJBgNVBAYTAkVFMQwwCgYDVQQI
DANOL0ExDjAMBgNVBAcMBUVhcnRoMB4XDTE3MTIzMTIyMDAwMFoXDTQ3MTIzMTIy
MDAwMFowgaQxKjAoBgNVBAMMIUxlbXVyVHJ1c3QgVW5pdHRlc3RzIFJvb3QgQ0Eg
MjAxODEjMCEGA1UECgwaTGVtdXJUcnVzdCBFbnRlcnByaXNlcyBMdGQxJjAkBgNV
BAsMHVVuaXR0ZXN0aW5nIE9wZXJhdGlvbnMgQ2VudGVyMQswCQYDVQQGEwJFRTEM
MAoGA1UECAwDTi9BMQ4wDAYDVQQHDAVFYXJ0aDCCASIwDQYJKoZIhvcNAQEBBQAD
ggEPADCCAQoCggEBAL8laXtLXyM64t5dz2B9q+4VvOsChefBi2PlGudqxDuRN3l0
Kmcfun6x2Gng24pTlGdtmiTEWA0a2F8HRLv4YBWhuYleVeBPtf1fF1/SuYgkJOWT
7S5qk/od/tUOLHS0Y067st3FydnFQTKpAuYveEkxleFrMS8hX8cuEgbER+8ybiXK
n4GsyM/om6lsTyBoaLp5yTAoQb4jAWDbiz1xcjPSkvH2lm7rLGtKoylCYwxRsMh2
nZcRr1OXVhYHXwpYHVB/jVAjy7PAWQ316hi6mpPYbBV+yfn2GUfGuytqyoXLEsrM
3iEEAkU0mJjQmYsCDM3r7ONHTM+UFEk47HCZJccCAwEAAaNCMEAwDwYDVR0TAQH/
BAUwAwEB/zAOBgNVHQ8BAf8EBAMCAQYwHQYDVR0OBBYEFFL12SFeOTTDdGKsHKoz
eByGHY6nMA0GCSqGSIb3DQEBCwUAA4IBAQAJfe0/uAHobkxth38dqrSFmTo+D5/T
MlRt3hdgjlah6sD2+/DObCyut/XhQWCgTNWyRi4xTKgLh5KSoeJ9EMkADGEgDkU2
vjBg5FmGZsxg6bqjxehK+2HvASJoTH8r41xmTioav7a2i3wNhaNSntw2QRTQBQED
OIzHRpPDQ2quErjA8nSifE2xmAAr3g+FuookTTJuv37s2cS59zRYsg+WC3+TtPpR
ssvobJ6Xe2D4cCVjUmsqtFEztMgdqgmlcWyGdUKeXdi7CMoeTb4uO+9qRQq46wYW
n7K1z+W0Kp5yhnnPAoOioAP4vjASDx3z3RnLaZvMmcO7YdCIwhE5oGV0
-----END CERTIFICATE-----
"""
ROOTCA_CERT = parse_certificate(ROOTCA_CERT_STR)
ROOTCA_KEY = """\
-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEAvyVpe0tfIzri3l3PYH2r7hW86wKF58GLY+Ua52rEO5E3eXQq
Zx+6frHYaeDbilOUZ22aJMRYDRrYXwdEu/hgFaG5iV5V4E+1/V8XX9K5iCQk5ZPt
LmqT+h3+1Q4sdLRjTruy3cXJ2cVBMqkC5i94STGV4WsxLyFfxy4SBsRH7zJuJcqf
gazIz+ibqWxPIGhounnJMChBviMBYNuLPXFyM9KS8faWbussa0qjKUJjDFGwyHad
lxGvU5dWFgdfClgdUH+NUCPLs8BZDfXqGLqak9hsFX7J+fYZR8a7K2rKhcsSysze
IQQCRTSYmNCZiwIMzevs40dMz5QUSTjscJklxwIDAQABAoIBAAyVzwMiLEpqhyNy
88N7osVTQxQKH3zp3l6eaA4SlocBgbCKeHw/t4y98uzNtEbASAYjTkHbd5ytRs/C
78Cckt75vfiQcIELXoUnLKfPfQ28q30+JyCmPcX7EZs/iqfIdL1rWFSHwEmJVkia
nik/uODA1gh4gU2EGgVIQEGXzNCv2RgTgmuyY+/4LbgEnUdMF+/umDhQd+o+nnL3
Ie0eJ7mTNPq78Dw6/21OcpE3j+yBIHGf5ZOHf5Qy+kBAytR7n+wzorkVYhbTvb4s
aWzHzmNBViQcum2SOaa/5I/HiG/Z4R2vnD53+bbjeLAsT/4OgUMdWziKOQbGbWwp
z+j/tekCgYEA8cfyOrbvkTLQHpC/1PC0GyrPGW+/5HKv34N+lCK4U+j6q5x7T7ia
kIacvW3/gPsmW0zcdty1RfgEKQXOT7sK7YN1yMwfRwHX7Ea9TW9ZPW/gnOlhwVJ0
Fx2SNESLmubi2cjEMayI9I++pDzwPMTLWNj53kgBhB/Qhw6+N74ScL0CgYEAymMm
JKVu3eNzooVIjS0wwlEpsteQZa1PI26QOyN/sr8wgbVrb/6P/dNc7Bl3YUMe0ZyR
ejyFo1bLsFemsWCZKWdgVOCQG1DqtHOPQSqkZPfYKfoMY9LWemyPk61CQDWzL3tg
L742V0iVPtYmtysKqagFFNeVML91duLWiobDwtMCgYAkl+2Kg2uI31bueVv/X5ry
zazgnbA+ZDlIK/+5bfPWB1oBJULokvkZzLXmWcKlA94PTXfEqazp9Rq0FsPd/2It
BouKI5LMTXQft6kpEiRAjzFArnX0K4WUhg49yO4UOMO20JMZLZLg6OyisPJvUB2y
ycwvn1hTZflKp6mUiDkERQKBgBUDyJEjkGh/1qD4f/kQyTBUJyVH1tmH7mC6eUV6
wSa5TXsacGZ3o1Hy4YIufsPdqVSQklaD9EhqmcncwBVI935iGpGVo8ECXOyR1z0o
BVvqlEp/iUvQN68MmLf31JpAOTPj9q/ea1wS0FRu/iQk1v2Y0bZBUF94ceT/VtGZ
frg7AoGBAI3iNhiZMQrq6j8kaEgDea+T1Ui0rGYIeD60TAuHfJ0a50Ga4rPp0u2p
G92Mk0sk2VGzZCkV6YRd/DJBXsqDAGeaEKzEnXiDBObUORDIo9SL3YhzosH191Ce
45ZXINAgovSzBcZfwStWjebUDw4v10Arc6ipkHGBPbK556S6wXGK
-----END RSA PRIVATE KEY-----
"""


#: CN=LemurTrust Unittests Class 1 CA 2018
INTERMEDIATE_CERT_STR = """\
-----BEGIN CERTIFICATE-----
MIIEGjCCAwKgAwIBAgIRAJ96dbOdrkw/lSTGiwbaagwwDQYJKoZIhvcNAQELBQAw
gaQxKjAoBgNVBAMMIUxlbXVyVHJ1c3QgVW5pdHRlc3RzIFJvb3QgQ0EgMjAxODEj
MCEGA1UECgwaTGVtdXJUcnVzdCBFbnRlcnByaXNlcyBMdGQxJjAkBgNVBAsMHVVu
aXR0ZXN0aW5nIE9wZXJhdGlvbnMgQ2VudGVyMQswCQYDVQQGEwJFRTEMMAoGA1UE
CAwDTi9BMQ4wDAYDVQQHDAVFYXJ0aDAeFw0xNzEyMzEyMjAwMDBaFw00NzEyMzEy
MjAwMDBaMIGnMS0wKwYDVQQDDCRMZW11clRydXN0IFVuaXR0ZXN0cyBDbGFzcyAx
IENBIDIwMTgxIzAhBgNVBAoMGkxlbXVyVHJ1c3QgRW50ZXJwcmlzZXMgTHRkMSYw
JAYDVQQLDB1Vbml0dGVzdGluZyBPcGVyYXRpb25zIENlbnRlcjELMAkGA1UEBhMC
RUUxDDAKBgNVBAgMA04vQTEOMAwGA1UEBwwFRWFydGgwggEiMA0GCSqGSIb3DQEB
AQUAA4IBDwAwggEKAoIBAQDR+qNdfNsLhGvgw3IgCQNakL2B9dpQtkVnvAXhdRZq
JETm/tHLkGvONWTXAwGdoiKv6+0j3I5InUsW+wzUPewcfj+PLNu4mFMq8jH/gPhT
ElKiAztPRdm8QKchvrqiaU6uEbia8ClM6uPpIi8StxE1aJRYL03p0WeMJjJPrsl6
eSSdpR4qL69GTd1n5je9OuWAcn5utXXnt/jO4vNeFRjlGp/0n3JmTDd9w4vtAyY9
UrdGgo37eBmi6mXt5J9i//NenhaiOVU81RqxZM2Jt1kkg2WSjcqcIQfBEWp9StG4
6VmHLaL+9/v2XAV3tL1VilJGj6PoFMb4gY5MXthfGSiXAgMBAAGjQjBAMA8GA1Ud
EwEB/wQFMAMBAf8wDgYDVR0PAQH/BAQDAgEGMB0GA1UdDgQWBBQstpQr0iMBVfv0
lODIsMgT9+9oezANBgkqhkiG9w0BAQsFAAOCAQEASYQbv1Qwb5zES6Gb5LEhrAcH
81ZB2uIpKd3Ki6AS4fLJVymMGkUs0RZjt39Ep4qX1zf0hn82Yh9YwRalrkgu+tzK
rp0JgegNe6+gyFRrJC0SIGA4zc3M02m/n4tdaouU2lp6jhmWruL3g25ZkgbQ8LO2
zjpSMtblR2euvR2+bI7TepklyG71qx5y6/N8x5PT+hnTlleiZeE/ji9D96MZlpWB
4kBihekWmxuptED22z/tpQtac+hPBNgt8z1uFVEYN2rKEcCE7V6Qk7icS+M4Vb7M
3D8kLyWDubs9Yy3l0EWjOXQXxEhTaKEm4gSuY/j+Y35bBVkA2Fcyuq7msiTgrw==
-----END CERTIFICATE-----
"""
INTERMEDIATE_CERT = parse_certificate(INTERMEDIATE_CERT_STR)
INTERMEDIATE_KEY = """\
-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEA0fqjXXzbC4Rr4MNyIAkDWpC9gfXaULZFZ7wF4XUWaiRE5v7R
y5BrzjVk1wMBnaIir+vtI9yOSJ1LFvsM1D3sHH4/jyzbuJhTKvIx/4D4UxJSogM7
T0XZvECnIb66omlOrhG4mvApTOrj6SIvErcRNWiUWC9N6dFnjCYyT67JenkknaUe
Ki+vRk3dZ+Y3vTrlgHJ+brV157f4zuLzXhUY5Rqf9J9yZkw3fcOL7QMmPVK3RoKN
+3gZoupl7eSfYv/zXp4WojlVPNUasWTNibdZJINlko3KnCEHwRFqfUrRuOlZhy2i
/vf79lwFd7S9VYpSRo+j6BTG+IGOTF7YXxkolwIDAQABAoIBAQC8dTuSeLEQUTWR
cVlIr043RpkPv1zF/BGm3PZaOAB6GztMJ4CcN27KkNmEsMoOdKq1QgaAnT+GpMX0
RjZpd3omyJi7JAPAVdavQNjm/RXjWRqZFlVw/LxDXbOjcc+IXQOk73rEdLBcvKT5
ZRjirzPev5IE49AF/0/0VYPqSHHEXL1HMWy5Q7KZU/C4Yp8em2l3UAh1/nLI2L+U
iw6DxBhBbdsc2vD4dd4UUQmznBfvdBOiejKs9DYIDzmTrGI2lH0VUU6PVzb9LKJ3
UtlIsjOO0FdJxXhkjsDCcF4eEMhO4qOfVJDcl0YYsPLjM0t4IVfe2C0b3Hd37hhF
b3ux0YoBAoGBAPWCPBepUdtiiMM21dLFPuy/d9/C3k8xTD23aO8zupwnGD1zz8OY
1GzsCtR9me0nIbHGxT+Unovuxh2JYQgrFpAFsG/izL8SpkDeT3hJDPY8PLrhGqMS
RBcYYgZ+1aNZTPZRKDjTG7PlwkcDt8VzQYYl87MBtDpPmdYkWYPnS+5XAoGBANrz
uSjbbTEX4a3WGsot9WzyhML11WgsdMgbqSWLMEmj2guwm4jn3XdqA+6nZAUJHgPI
Ir5l2E3zWTdw5pZ0+IkQnj/OJP2OLqS+msVMl89PHJjtPyB2LVjCg+Z/vfm2ar5S
RnQ/AoR5cSXctiL9wJbJBidfZeEpXBSeKN/SW6/BAoGAXaQ4EXpWq4wQyAzRT9xG
HP0G1wU30BLolp2vW5Vqdwb+Wuoic+OGGqmJk/T4Uhlb47gCIjcopg0D6d4tcXUl
3Pcejf5+w950JUfmHeYXGJBvRYR4qXxdFkYJlZqpF+4Gyei4o7v51Astp/KGFLza
YDV3l25t9NPJxIEG16XQM28CgYEAyT3yFYd42QKmPuznOqT7SuOs+rSRLWqO+83Q
rd08yLJ9Gvl8O11BxRv/+T6JQ8eZeshchrt9EEh22+o9RlTEitZnXSXQAezJGkrG
XkmDzttb4YNN3jxAebBvI1COABKWEc/1SasQWUp1oOM31Pl+JhkmOtIIBefJ5nlo
ADCMbQECgYBdpg/2J58JOB0NhFFUDh3+6+bzZIciT7BYZ8zYBej5DR6kd4Inkvvl
JxOlYIKSew5uyi7i2po7fBFRhKuyfpNMBYK01ObrgJNZELbqVgYvKw1HEqqtgExa
eMVHHbWm1CpGO294R+vMBv4jcuhIBOx63KZE4VaoJuaazF6TE5czDw==
-----END RSA PRIVATE KEY-----
"""


#: CN=san.example.org, issued by LemurTrust Unittests Class 1 CA 2018
SAN_CERT_CSR = """\
-----BEGIN CERTIFICATE REQUEST-----
MIICvTCCAaUCAQAweDELMAkGA1UEBhMCRUUxDDAKBgNVBAgMA04vQTEOMAwGA1UE
BwwFRWFydGgxGDAWBgNVBAoMD0RhbmllbCBTYW4gJiBjbzEXMBUGA1UECwwOS2Fy
YXRlIExlc3NvbnMxGDAWBgNVBAMMD3Nhbi5leGFtcGxlLm9yZzCCASIwDQYJKoZI
hvcNAQEBBQADggEPADCCAQoCggEBAMia9BcpypZUU9xJoknzdEp+AevQE93XSAyl
IlXji80ZlYS/T/mVWtu6hNwz2IJDBFh6nPaHT1Ud/AI4YanDMa+fF4KJxzlkKPbY
quWx4EOjTZ2sFBBCivwxlo1So8r5Hf4NZ9Ewu4AIma3zmk+dzxJTpnWbTIFJGsDG
LwJO9iu6uqf79VdYkGELCusq3dyF2j2DNDiGHoRcQYFMMhDKR6uYmCTYvwjf0+sf
6k1zk2EK1X+ZWUyjP+Nl2NB6bpL0TydF75fuplWROczceiO6BKO4YT2uNPdF4BAH
p/kQCkqnjw5FCX7PONRT4wTW/AjDkt5WOgY+AB90zQBPxvXWbUMCAwEAAaAAMA0G
CSqGSIb3DQEBCwUAA4IBAQAFYgEafwRmsqdK1i1xrLFYbNNLkzmAZyL+6gXUBVIJ
TbGVVWSNNIcEmHIX8O9X4lN52qDYWOsxH/OKPVxpXqoHm/ztczFlte76wOYg+VAS
yK8DwQRP/+n+j6J40o1cZwnilPWqHgee5zbIL7lpCVxuFDofWpskwP5PLbxibFq8
4TWynhjKKUw4+q4h4iCHG3PQhbV0ExWOyqX05QyDtJdkEwgJUWz1m9caHU2Jl7kX
5bWKOtXORpCYA7ed3WqktKQIxBD6vCVbQ+LuLZPYeWzGHYjfOejL6usD32KmNa2E
ZhDsC0fjqSX0FJKz6gOhP88bkbbapyHuGB71o2dwhCKV
-----END CERTIFICATE REQUEST-----
"""

SAN_CERT_STR = """\
-----BEGIN CERTIFICATE-----
MIIESjCCAzKgAwIBAgIRAK/y20+NLU2OgPo4KuJ8IzMwDQYJKoZIhvcNAQELBQAw
gacxLTArBgNVBAMMJExlbXVyVHJ1c3QgVW5pdHRlc3RzIENsYXNzIDEgQ0EgMjAx
ODEjMCEGA1UECgwaTGVtdXJUcnVzdCBFbnRlcnByaXNlcyBMdGQxJjAkBgNVBAsM
HVVuaXR0ZXN0aW5nIE9wZXJhdGlvbnMgQ2VudGVyMQswCQYDVQQGEwJFRTEMMAoG
A1UECAwDTi9BMQ4wDAYDVQQHDAVFYXJ0aDAeFw0xNzEyMzEyMjAwMDBaFw00NzEy
MzEyMjAwMDBaMHgxGDAWBgNVBAMMD3Nhbi5leGFtcGxlLm9yZzEYMBYGA1UECgwP
RGFuaWVsIFNhbiAmIGNvMRcwFQYDVQQLDA5LYXJhdGUgTGVzc29uczELMAkGA1UE
BhMCRUUxDDAKBgNVBAgMA04vQTEOMAwGA1UEBwwFRWFydGgwggEiMA0GCSqGSIb3
DQEBAQUAA4IBDwAwggEKAoIBAQDImvQXKcqWVFPcSaJJ83RKfgHr0BPd10gMpSJV
44vNGZWEv0/5lVrbuoTcM9iCQwRYepz2h09VHfwCOGGpwzGvnxeCicc5ZCj22Krl
seBDo02drBQQQor8MZaNUqPK+R3+DWfRMLuACJmt85pPnc8SU6Z1m0yBSRrAxi8C
TvYrurqn+/VXWJBhCwrrKt3chdo9gzQ4hh6EXEGBTDIQykermJgk2L8I39PrH+pN
c5NhCtV/mVlMoz/jZdjQem6S9E8nRe+X7qZVkTnM3HojugSjuGE9rjT3ReAQB6f5
EApKp48ORQl+zzjUU+ME1vwIw5LeVjoGPgAfdM0AT8b11m1DAgMBAAGjgZ4wgZsw
DAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDATAOBgNVHQ8BAf8EBAMC
BaAwHQYDVR0OBBYEFPMfkyOR6taOKARGUn1zyauOeOhAMEcGA1UdEQEB/wQ9MDuC
D3Nhbi5leGFtcGxlLm9yZ4IQc2FuMi5leGFtcGxlLm9yZ4IWZGFuaWVsLXNhbi5l
eGFtcGxlLm9yZzANBgkqhkiG9w0BAQsFAAOCAQEAJer6tDWlcc4gDvZEbPNpFPKY
kPqww6WtVKrsyPQyucmP7arFx+kUulkZN6GY6FyIao7zQPrHU3GqtPXt/lR++wzz
bnj+iHONNoAY5TZLycPXAW7pn8UCvdQzkyhMzCB1aRkRS/Tr0/sC0T5lYux42dQG
rTUxVG7Umr4jq8hKEElNIwxF+3kkfzztICZXJtzmVRAz4XpmXmyMcZOOXh3F7z04
kePn3ztNEJNuGK0qMnTJlL7T2a9v+Q29YhIiNqCFdoALJbzJmEj3+QgcLFJh3KjY
J4WEmpGK7FylxmZLrya12SxPMA8A60hMPeWH90xRS8SJfd57K5oCluGK14X+og==
-----END CERTIFICATE-----
"""
SAN_CERT = parse_certificate(SAN_CERT_STR)
SAN_CERT_KEY = """\
-----BEGIN RSA PRIVATE KEY-----
MIIEpQIBAAKCAQEAyJr0FynKllRT3EmiSfN0Sn4B69AT3ddIDKUiVeOLzRmVhL9P
+ZVa27qE3DPYgkMEWHqc9odPVR38AjhhqcMxr58XgonHOWQo9tiq5bHgQ6NNnawU
EEKK/DGWjVKjyvkd/g1n0TC7gAiZrfOaT53PElOmdZtMgUkawMYvAk72K7q6p/v1
V1iQYQsK6yrd3IXaPYM0OIYehFxBgUwyEMpHq5iYJNi/CN/T6x/qTXOTYQrVf5lZ
TKM/42XY0HpukvRPJ0Xvl+6mVZE5zNx6I7oEo7hhPa4090XgEAen+RAKSqePDkUJ
fs841FPjBNb8CMOS3lY6Bj4AH3TNAE/G9dZtQwIDAQABAoIBAQCxN3J7JAg8VbLf
4IzmF5ScWkUINYHXcN/Ni/SRO7u9LOTRqNDWBAOIKXZFseeK6/li0K7pew+yehKv
Q2/DsRSruTfjsiO1p64oo7AVytX76sAekm4HD0IJGSWPI3pfTUQZs24Ld6msqexZ
p+Kigx7zacKcEt27OQHRW0McHvWKGpCzc+G9o8hlYumtnTnR8kP4bYG4MpCVtZdt
ZuQr9QHEH7omQzWedLHICqGLP8WBu+lMY3bPM/gLgBSy2f8CP5MBy7xUUSTgolCc
ZZIt2H2cCStfb3y61WvDHX/bZ7B/a0lDWKHUK2f9Du5ENOWAAsaIUCPGrFwMFM1n
ANPtYIUBAoGBAP7SXzTcsHx0Iy1TMqluQxZNL7Up6dHksZFEO8zAwmg6Z3VHsXjc
3ovst1gQRWUzEhc2Soun8KbZ4ZVhApHpkxg+lLio2A4mLm7xf6ASsPwkwUCwUBYP
zSt56oopQ6BIc84ZDNe5QZY9OziRyHRc0SgVhuoK1i2t832kUii/aWmTAoGBAMmI
aA6vDlojHnyhvNCS/v+aOEjNyW7LJjDhNK6qTKZF7FeCSjTLxzJJ6XddQjLXnjVa
aF1I0dYPLQfjbeDaM6pKyg4l0pnN70qh9YgFCCIxU0dIqL2v0a057JILwlC4ZU8z
rmQEP3DROGGi50r74/SoYMBVm6r/6opYkThF+HuRAoGBAIHxrXM7hxQv9TBL2O3l
uHhK7CUqNn4+bP5zGTuUoI6eGdwIr0u+9g3MrMJPqdOtc3A601DcVy/+s7aFPdZC
kiwu3ZA9KdAtUEhrBnYOkgpCg/oE7xIRBMNC7IN//2hhCgzzYUUwx21h1C1Iyjvs
iQwzzhTTadzpc92CShNVaN/ZAoGBAKwG+fv+xdt/OtjcHpZTw8NfW0gaESW31yPG
OPgXelI8QZ/5IWqrv59XpCg1vPo1P2D/iTKHpEZ6sc+X/QUAfTWRnaQx+PE87lPg
p/uxf93gCNxCU3eHiw248g1AaGAK5r+St/u7/INKtDvzmEdTeKQwzlWfPb/br9Lk
AyHr6E1hAoGAN375PaykO+webBV9GkUwCcs1wTmDFjWKuaCVk9qvj8qavr1HOarb
ZolDBFYQSUUcOJHE9QRg7zDWTjI+Av670rXtTsREs0yMkdHhF/aRD7FJoBvO/7v/
tqdgqMHlJ0YPL/8/6sTMasepIqmAkQgFINzDskHubBBEPfHKqF0KQA4=
-----END RSA PRIVATE KEY-----
"""

WILDCARD_CERT_STR = """\
-----BEGIN CERTIFICATE-----
MIIEJDCCAwygAwIBAgIPK/QvcZhAY7VPU7Ek/nCDMA0GCSqGSIb3DQEBCwUAMIGn
MS0wKwYDVQQDDCRMZW11clRydXN0IFVuaXR0ZXN0cyBDbGFzcyAxIENBIDIwMTgx
IzAhBgNVBAoMGkxlbXVyVHJ1c3QgRW50ZXJwcmlzZXMgTHRkMSYwJAYDVQQLDB1V
bml0dGVzdGluZyBPcGVyYXRpb25zIENlbnRlcjELMAkGA1UEBhMCRUUxDDAKBgNV
BAgMA04vQTEOMAwGA1UEBwwFRWFydGgwHhcNMTcxMjMxMjIwMDAwWhcNNDcxMjMx
MjIwMDAwWjB9MRswGQYDVQQDDBIqLndpbGQuZXhhbXBsZS5vcmcxETAPBgNVBAoM
CFBsYXl0ZWNoMRkwFwYDVQQLDBBJbmZyYSBPcGVyYXRpb25zMQswCQYDVQQGEwJF
RTERMA8GA1UECAwISGFyanVtYWExEDAOBgNVBAcMB1RhbGxpbm4wggEiMA0GCSqG
SIb3DQEBAQUAA4IBDwAwggEKAoIBAQCoT8Ak5kynUzosBvP8hCnP4hGMAtgHLcHG
UBWug4BofAhxxBrZW3UteoQzNznK5jz0hy2azqnz3/9q5N/FKwHxfMY/VEHPXyYK
QsZuSdVceJ/EHL+MLx+uisIRJstV8fC5oYRfg74m07ZED7NM4EerJTxKZAy7UuSM
L65i/LEChPzjLN46GcUEuC2O03nZtFTPvN9j7vzen9/qIzs1TGQukOn4z5l2GuAx
RCEfBl3IrnvSY+npGARPJsXSymXCCP3ntzq6I6iRHuZf+QETZtiMR1TCNZRTqcc2
LxWn+W5N18yyXvUcVMfrg4jzEWKHuhwInoiH1pu/myyKrnoIi4nTAgMBAAGjdjB0
MBMGA1UdJQQMMAoGCCsGAQUFBwMBMA4GA1UdDwEB/wQEAwIFoDAMBgNVHRMBAf8E
AjAAMB0GA1UdDgQWBBRR9Q9DHJRPt69Qm8lir4iJfOmJ4TAgBgNVHREBAf8EFjAU
ghIqLndpbGQuZXhhbXBsZS5vcmcwDQYJKoZIhvcNAQELBQADggEBAMm2DiYfGLve
r/gCtYgXKkRmbuv57PmAUm52w5l4hjxssdUwq4Wn4T+K0+Sqp3IzcNhEaIqPB+bG
8rIbJLBiiDPbSUZC0DbvlXihk7FHjqmrbVFwNmkWNywLhB1qOlp0kQH+w9lDWA1p
y99P0Bxcot66scbiaag0i0AUpkRKbUG+v+VGXdPrJrWE+63ROhWQMmQNiUlZ6QGO
45tUSn//MuUpJiJVkUVR1fSbCpHQj2mHiuhShOmatmh5e1ISwVP19cX64Gr6djlY
wKJqcmw7WDjl+T+y7luJWw4UqI7s7hY6Y9RQVh61L4eV8CIma3NmTaQCSgR3tCxh
d4FCKAE8+Lw=
-----END CERTIFICATE-----
"""
WILDCARD_CERT = parse_certificate(WILDCARD_CERT_STR)
WILDCARD_CERT_KEY = """\
-----BEGIN RSA PRIVATE KEY-----
MIIEogIBAAKCAQEAqE/AJOZMp1M6LAbz/IQpz+IRjALYBy3BxlAVroOAaHwIccQa
2Vt1LXqEMzc5yuY89Ictms6p89//auTfxSsB8XzGP1RBz18mCkLGbknVXHifxBy/
jC8frorCESbLVfHwuaGEX4O+JtO2RA+zTOBHqyU8SmQMu1LkjC+uYvyxAoT84yze
OhnFBLgtjtN52bRUz7zfY+783p/f6iM7NUxkLpDp+M+ZdhrgMUQhHwZdyK570mPp
6RgETybF0splwgj957c6uiOokR7mX/kBE2bYjEdUwjWUU6nHNi8Vp/luTdfMsl71
HFTH64OI8xFih7ocCJ6Ih9abv5ssiq56CIuJ0wIDAQABAoIBAHpE3SexKbRQMK01
K9+gPyOmberRUg/8/IzNNsL9ArZkjFnhBUQrPXeZThpKnzA3i8ZzwPx571qbudf0
hl6cfJ/qbbYpxlkYHPFNSwtplZbEhgOYgsoanaBVat+81/AKfz7LB/e/I87e88SD
x1QshcPdm+vKvLkEYcU5Ci0ctpZFyjgqWAVdsaEmstSpygOsU/CExiiq20MuGXQV
lfRpYqmA0D1LicN6t0rGRNLUHWzhZS5s1ZHD8oR5bgtlcxvOSDm81ROfV1obAQRo
RyCeYG9Juo2BisyuG5RsKuzfWvP6goxXdGD2uTOj7rdMYRfnUzsz4QXkzdA+IEE9
qmv9gsECgYEA0gYoSVwF+koYL8SURMLbQPdmVBiTSxZQytSaFTl9WWYUvyFJ/R0c
xmuMjfVvnxuMUa6qHcaOt9Gjq0pAp2T10/8B+fB8XAJ0bxl6DKArxIONx7W1T8RX
SdEAWF77eRlSPAD4dfAyFIjPzCH4l98nX84sRk+GiusLGhm5MdwTS2ECgYEAzSgB
H33YIfPMW7Mh9R7QpfQRyda9Ikb8pGDegprTLp9SDLU0zCZ3aPmSN3iVrGThwcf1
olnx5g8ZK9qiLLIUctxJWeK/EO2HAii/fmasJfq4zI0xBflkX25V+BbZRqU8GDa0
dLpOcr1cIEFAqpZ182O1uXAJdGzkVv4LN5Iq9bMCgYBuYAIIG66gjRQM9pidUnJ7
wAktJQUzrvSiw/x+LwprUzSQBeSmewhGVvs1F8mjqoyh2NNadqFGhYwoVwuHbY2r
7haRzgrtJ/Uc9hyoSfz2d9SpIhu5YgwlpQszZiduyxnmovPwt1z6YbQjKa9F0WcV
+HpYMS8aDtB01RP42hnhwQKBgHDg8PU9zZyowqk1v5pZ8R1OVDdE4t2oRzu+XM9p
loaRfJinXzxwccUdhFjnDRtEin6PodRJPvKBHi1l51NGTEACeo1tWAldV6pVdz96
CIABGorZqL6LwLFNSRnuoG/hXFZKSzHqjF1PWRAaNxVlIdLf6s30Gg+oFl7S+qMB
1odHAoGAQqPWDTSBVhbMIzXXoGGjIx08WG9xuNqRyQ6fjekSporXisoej1XB+pYA
wZF7+p71qUsrVrg+j7SRm8pFsR79nvshQImzaLGOCLUsdHZfcKjfy8D7wB6T7AmF
gt+hqNmTFkpLkvaze7sBEwRWRU0/RWpZLDmx3+zvRXPw1lQmB1Q=
-----END RSA PRIVATE KEY-----
"""


INVALID_CERT_STR = """
-----BEGIN CERTIFICATE-----
MIIEFTCCAv2gAwIBAgICA+gwDQYJKoZIhvcNAQELBQAwgYwxCzAJBgNVBAYTAlVT
MQswCQYDVQQIDAJDQTEQMA4GA1UEBwwHQSBwbGFjZTEXMBUGA1UEAwwObG9uZy5s
aXZlZC5jb20xEDAOBgNVBAoMB0V4YW1wbGUxEzARBgNVBAsMCk9wZXJhdGlvbnMx
HjAcBgkqhkiG9w0BCQEWD2ppbUBleGFtcGxlLmNvbTAeFw0xNTA2MjYyMDM2NDha
Fw0xNTA2MjcyMDM2NDhaMGkxCzAJBgNVBAYTAlVTMQswCQYDVQQIEwJDQTEQMA4G
A1UEBxMHQSBwbGFjZTEQMA4GA1UEChMHRXhhbXBsZTETMBEGA1UECxMKT3BlcmF0
aW9uczEUMBIGA1UEAxMLZXhwaXJlZC5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IB
DwAwggEKAoIBAQCcSMzRxB6+UONPqYMy1Ojw3Wi8DIpt9USnSR60I8LiEuRK2ayr
0RMjLJ6sBEgy/hISEqpLgTsciDpxwaTC/WNrkT9vaMcwfiG3V0Red8zbKHQzC+Ty
cLRg9wbC3v613kaIZCQCoE7Aouru9WbVPmuRoasfztrgksWmH9infQbL4TDcmcxo
qGaMn4ajQTVAD63CKnut+CULZIMBREBVlSTLiOO7qZdTrd+vjtLWvdXVPcWLSBrd
Vpu3YnhqqTte+DMzQHwY7A2s3fu4Cg4H4npzcR+0H1H/B5z64kxqZq9FWGIcZcz7
0xXeHN9UUKPDSTgsjtIzKTaIOe9eML3jGSU7AgMBAAGjgaIwgZ8wDAYDVR0TAQH/
BAIwADAOBgNVHQ8BAf8EBAMCBaAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwEwHQYD
VR0OBBYEFKwBYaxCLxK0csmV319rbRdqDllWMEgGA1UdHwRBMD8wPaA7oDmGN2h0
dHA6Ly90ZXN0LmNsb3VkY2EuY3JsLm5ldGZsaXguY29tL2xvbmdsaXZlZENBL2Ny
bC5wZW0wDQYJKoZIhvcNAQELBQADggEBADFngqsMsGnNBWknphLDvnoWu5MTrpsD
AgN0bktv5ACKRWhi/qtCmkEf6TieecRMwpQNMpE50dko3LGGdWlZRCI8wdH/zrw2
8MnOeCBxuS1nB4muUGjbf4LIbtuwoHSESrkfmuKjGGK9JTszLL6Hb9YnoFefeg8L
T7W3s8mm5bVHhQM7J9tV6dz/sVDmpOSuzL8oZkqeKP+lWU6ytaohFFpbdzaxWipU
3+GobVe4vRqoF1kwuhQ8YbMbXWDK6zlrT9pjFABcQ/b5nveiW93JDQUbjmVccx/u
kP+oGWtHvhteUAe8Gloo5NchZJ0/BqlYRCD5aAHcmbXRsDid9mO4ADU=
-----END CERTIFICATE-----
"""
INVALID_CERT = parse_certificate(INVALID_CERT_STR)


EXTERNAL_VALID_STR = """
-----BEGIN CERTIFICATE-----
MIIFHzCCBAegAwIBAgIQGFWCciDWzbOej/TbAJN0WzANBgkqhkiG9w0BAQsFADCB
pDELMAkGA1UEBhMCVVMxHTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8w
HQYDVQQLExZGT1IgVEVTVCBQVVJQT1NFUyBPTkxZMR8wHQYDVQQLExZTeW1hbnRl
YyBUcnVzdCBOZXR3b3JrMTQwMgYDVQQDEytTeW1hbnRlYyBDbGFzcyAzIFNlY3Vy
ZSBTZXJ2ZXIgVEVTVCBDQSAtIEc0MB4XDTE1MDYyNDAwMDAwMFoXDTE1MDYyNTIz
NTk1OVowgYMxCzAJBgNVBAYTAlVTMRMwEQYDVQQIDApDQUxJRk9STklBMRIwEAYD
VQQHDAlMb3MgR2F0b3MxFjAUBgNVBAoMDU5ldGZsaXgsIEluYy4xEzARBgNVBAsM
Ck9wZXJhdGlvbnMxHjAcBgNVBAMMFXR0dHQyLm5ldGZsaXh0ZXN0Lm5ldDCCASIw
DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBALwMY/yod9YGLKLCzbbsSUBWm4ZC
DfcgbUNL3JLtZaFCaOeUPLa4YNqty+9ACXBLYPNMm+dgsRHix8N2uwtZrGazHILK
qey96eSTosPsvKFt0KLNpUl8GC/YxA69L128SJgFaaq5Dr2Mp3NP0rt0RIz5luPj
Oae0hkGOS8uS0dySlAmfOw2OsJY3gCw5UHcmpcCHpO2f7uU+tWKmgfz4U/PpQ0kz
WVJno+JhcaXIximtiLreCNF1LpraAjrcZJ+ySJwYaLaYMiJoFkdXUtKJcyqmkbA3
Splt7N4Hb8c+5aXv225uQYCh0HXQeMyBotlaIrAddP5obrtjxhXBxB4ysEcCAwEA
AaOCAWowggFmMCAGA1UdEQQZMBeCFXR0dHQyLm5ldGZsaXh0ZXN0Lm5ldDAJBgNV
HRMEAjAAMA4GA1UdDwEB/wQEAwIFoDAdBgNVHSUEFjAUBggrBgEFBQcDAQYIKwYB
BQUHAwIwYQYDVR0gBFowWDBWBgZngQwBAgIwTDAjBggrBgEFBQcCARYXaHR0cHM6
Ly9kLnN5bWNiLmNvbS9jcHMwJQYIKwYBBQUHAgIwGRoXaHR0cHM6Ly9kLnN5bWNi
LmNvbS9ycGEwHwYDVR0jBBgwFoAUNI9UtT8KH1K6nLJl7bqLCGcZ4AQwKwYDVR0f
BCQwIjAgoB6gHIYaaHR0cDovL3NzLnN5bWNiLmNvbS9zcy5jcmwwVwYIKwYBBQUH
AQEESzBJMB8GCCsGAQUFBzABhhNodHRwOi8vc3Muc3ltY2QuY29tMCYGCCsGAQUF
BzAChhpodHRwOi8vc3Muc3ltY2IuY29tL3NzLmNydDANBgkqhkiG9w0BAQsFAAOC
AQEAQuIfyBltvCZ9orqNdS6PUo2PaeUgJzkmdDwbDVd7rTwbZIwGZXZjeKseqMSb
L+r/jN6DWrScVylleiz0N/D0lSUhC609dQKuicGpy3yQaXwhfYZ6duxrW3Ii/+Vz
pFv7DnG3JPZjIXCmVhQVIv/8oaV0bfUF/1mrWRFwZiBILxa7iaycRhjusJEVRtzN
Ot/qkLluHO0wbEHnASV4P9Y5NuR/bliuFS/DeRczofNS78jJuZrGvl2AqS/19Hvm
Bs63gULVCqWygt5KEbv990m/XGuRMaXuHzHCHB4v5LRM30FiFmqCzyD8d+btzW9B
1hZ5s3rj+a6UwvpinKJoPfgkgg==
-----END CERTIFICATE-----
"""
EXTERNAL_CERT = parse_certificate(EXTERNAL_VALID_STR)


INTERNAL_CERTIFICATE_A_STR = """
-----BEGIN CERTIFICATE-----
MIIDazCCAlOgAwIBAgIBATANBgkqhkiG9w0BAQsFADB5MQswCQYDVQQGEwJVUzET
MBEGA1UECAwKQ2FsaWZvcm5pYTESMBAGA1UEBwwJTG9zIEdhdG9zMRYwFAYDVQQK
DA1OZXRmbGl4LCBJbmMuMRMwEQYDVQQLDApPcGVyYXRpb25zMRQwEgYDVQQDDAtB
Y29tbW9uTmFtZTAeFw0xNjA2MjkyMjE0NDdaFw0zNjA2MjkyMjE0NDdaMHkxCzAJ
BgNVBAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlhMRIwEAYDVQQHDAlMb3MgR2F0
b3MxFjAUBgNVBAoMDU5ldGZsaXgsIEluYy4xEzARBgNVBAsMCk9wZXJhdGlvbnMx
FDASBgNVBAMMC0Fjb21tb25OYW1lMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIB
CgKCAQEAtkyvL6EqSgYSJX11635Hb8FBG/8Wey6C2KtG7M+GXvGCsSmfNqQMeZdf
W9Avxelkstp5/K+ilVJJ2TJRelu1yVUUkQcrP7imgf7CxKQAnPz2oXQImLFbm7OS
1zKA+qwtLGrId3vVQaotUtdI+wxx0YE66pyfOhQJsVOeuYwG8CCxnAj/lXeNLA1t
n39A8FLfj9nxjvZWWm2z8qXO2IYOWEMOOel1zixhypeJoTD2cJHDKNlUnXN4q5ej
psD4ehLFXIPXsKJv5XOtNYB9UHB3moXlEOuKAquRzBOfTP+rUYyfbHmzCN4eXekp
R6vze49hlg8QdCNjVY6jHRrOuVKGuwIDAQABMA0GCSqGSIb3DQEBCwUAA4IBAQAt
rE2Ee6a0zRlJHiuP5Zr61s6ZnwIsPN5sjo3pFJ/goHeNWbq+02FUJLXROtxSMlo8
jLYpnQbm3Qoyd0KjGn9myP1vqBL6Yzf9dRI2li9XYmavxU7OK/KJtBo/Wnw3DVT5
jxYrn4YKJU9+T0hr57bWUQ7HjMNojwBcgglzPN9KOtfTfbPEUIeoRpCjeyjwBUSN
nrTDiYPV+XI4LAyDmuR7esSvm2+0h6C0dmUbVspkxBaKFEYUKIYaZbEFEBsyZGri
qDIyu9HSvu2MJ2lVxfMNsW+IYG74DOqJQsIFP+7hrfdPoMGm4GvAiHR1IuSmq+sf
L0Ew8hy0GG3nZ6uXLW7q
-----END CERTIFICATE-----
"""


INTERNAL_PRIVATE_KEY_A_STR = """
-----BEGIN RSA PRIVATE KEY-----
MIIEpQIBAAKCAQEAtkyvL6EqSgYSJX11635Hb8FBG/8Wey6C2KtG7M+GXvGCsSmf
NqQMeZdfW9Avxelkstp5/K+ilVJJ2TJRelu1yVUUkQcrP7imgf7CxKQAnPz2oXQI
mLFbm7OS1zKA+qwtLGrId3vVQaotUtdI+wxx0YE66pyfOhQJsVOeuYwG8CCxnAj/
lXeNLA1tn39A8FLfj9nxjvZWWm2z8qXO2IYOWEMOOel1zixhypeJoTD2cJHDKNlU
nXN4q5ejpsD4ehLFXIPXsKJv5XOtNYB9UHB3moXlEOuKAquRzBOfTP+rUYyfbHmz
CN4eXekpR6vze49hlg8QdCNjVY6jHRrOuVKGuwIDAQABAoIBACYPnqfwGzc3S0Se
jCctx1Zy39grixMO4+y+3eEFdwWNoP7CNOagm6YrT5KIxeCpWQfqi3uRY/2PH7IE
SnSkfzDY3aFmAMaeE82iViHeJ+6e9hNBeaX/qaO5e1gIyFsN5aSXauFfbmf2Ut4v
6qHXuE/Ijnd7WdczZc6rKcGNlck+f/QtsZhYEYbgHT3Nrt0ztlvkdrcyRIxZTeS7
7gvVWrVv6rviTobi/ZkeM9pqe5bbLuWgb/ArvI52pJwaUcz9LPGo+miank6e4gAd
cTudoREtBKVgXROhTSz33mdjjUTCDGdtILTztDSgLpJXYT0w2h1zmfV7t4tztzzQ
xW5LVCECgYEA33YG/gaZbfH6szC/heilojelrIG+n7GjsqpfMqGFofYNBAswUC3w
qZdeXxqGZEXC8mx8CufDhC50vJv353WAHaFFJcwy2QeGvHfPAZ4ZQ68o9XLeva4t
M6+ZtOiaK8u/mzxq43Jj7FbXmxxlJXY3B0uWdWpKGsPRTmSaUw0lKPECgYEA0NhG
74C6zRgHY2Eq2Qq7+NtlvpzUtVtalhiDoCEpDMhjzLUTBNy6yMsSdP8SyCy9O7Ng
rrXJdgKHvpjnJyUvB3hhEAurPIPWJArEfEHAF+V8mHY8f58xZqgHRsYsH3tWHYx4
2lzmposTES5KKV4xsYbjjyzXX+WNdaOkC4JBCmsCgYEA3j2JKL0xfMordmlmIWzG
xnWnnNCQ4EwQrVGKSlWgDPsj6MCj9Sorbs9veRBtVm6XOvkvyLzFk8GMMkTAIf+X
QmCw362daIF2vBw/0bEGGW2sQ6hR5L3EkOH08ZpgMmx6DI7jE4Ah5txbpBVydvaC
Ngw0AGSMfOABW4DshurM6VECgYEAxeH3rJ2r4gL/lSGPaOGr5At2Z1rQjRqHRarq
pQJmk+8X6PI1mCjRbspDrcm2cSc7EmNPm5sxzXhuSKE2fLfVzN06EusLkCZW9AWj
0Ry3t6zBFvEJN9+N/nf9lQjW6+mAWjUsmbLm9SzXnzLeID5ZFZ365kGVvQ6Tr8Cj
AiikGgsCgYEAlYGNwBKWClm797YVyPhmqrFX4T9Hpxc7oC3vVwd96tAbLlSrW8r5
o6ynBW1bG+qfjx9GyThgudvRtB+0vTSShrT5GftLCyMtOiYSHkGEvMOGFBuowzoz
3i841gR9+cwA0S1hy7fC0PDmTo0xC91JocwesPQ023MmECPfu6Frzog=
-----END RSA PRIVATE KEY-----
"""


CSR_STR = """
-----BEGIN CERTIFICATE REQUEST-----
MIIC1zCCAb8CAQAwczEUMBIGA1UEAwwLQUNvbW1vbk5hbWUxFTATBgNVBAoMDG9y
Z2FuaXphdGlvbjEOMAwGA1UECwwFZ3VuaXQxCzAJBgNVBAYTAlVTMRMwEQYDVQQI
DApDYWxpZm9ybmlhMRIwEAYDVQQHDAlzb21ld2hlcmUwggEiMA0GCSqGSIb3DQEB
AQUAA4IBDwAwggEKAoIBAQDNnY+Ap+V9+Eg/PAtd7bq27D7tDvbL10AysNUSazy7
gJyHfJyE3oiXm28zjFNzRQ35qhsCFpWg8M36FpdP9fIFG9sVXV/ye+YNBkZ2aTJi
RnbErZcy8qc+2MRd2JKE9g0pISp9hAEeEPLTwSoGqf5VqOaBehBqL5OKNUr7JAxV
TIH1oVU87w/6xg/WsUiyPo49WXxF/3DZNP1UOTYiffxIiARhTb9EtlXpt5iOlic3
w/vBX6qsH++XJIus2WE+ABlAVUQTCvc6bgpu4zjc8nlm3ClqkAKcxn2ubEder+Fh
hagMYGsbYG+/IWrKYN6S0BjE26tNMiOlmIebimjEdFpnAgMBAAGgHzAdBgkqhkiG
9w0BCQ4xEDAOMAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcNAQELBQADggEBAE5OKI/n
b1ZRJDL4SpjWggRjfwBdYmb96lGH0aGDoVUP9UUusLzpWLtutkgr9Hh29agSsLZF
j535NeXHf+Jc4UyR288WQVJthgAT1e5+jBNPxz4IcTnDW7ZMJLGm495XaKi6Krcg
+8Qn2+h04jBTbN2Z9+MXGak0B8ycrbDx/FYL4KgBJRvS805d43zC6L1aUfRbpZgN
QeQoBdLhFNB1kAYSWCyETwRQOeGEphBJYBPcXsQVBWbMtLpbhjRZ1uTVZEFIh8Oa
zm3Cn4Ul8DO26w9QS4fmZjmnPOZFXYMWoOR6osHzb62PWQ8FBMqXcdToBV2Q9Iw4
PiFAxlc0tVjlLqQ=
-----END CERTIFICATE REQUEST-----
"""


EC_CERT_STR = """
-----BEGIN CERTIFICATE-----
MIIDxzCCAq+gAwIBAgIIHsJeci1JWAkwDQYJKoZIhvcNAQELBQAwVDELMAkGA1UE
BhMCVVMxHjAcBgNVBAoTFUdvb2dsZSBUcnVzdCBTZXJ2aWNlczElMCMGA1UEAxMc
R29vZ2xlIEludGVybmV0IEF1dGhvcml0eSBHMzAeFw0xOTAyMTMxNTM1NTdaFw0x
OTA1MDgxNTM1MDBaMGgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlh
MRYwFAYDVQQHDA1Nb3VudGFpbiBWaWV3MRMwEQYDVQQKDApHb29nbGUgTExDMRcw
FQYDVQQDDA53d3cuZ29vZ2xlLmNvbTBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IA
BKwMlIbd4rAwf6eWoa6RrR2w0s5k1M40XOORPf96PByPmld+qhjRMLvA/xcAxdCR
XdcMfaX6EUr0Zw8CepitMB2jggFSMIIBTjATBgNVHSUEDDAKBggrBgEFBQcDATAO
BgNVHQ8BAf8EBAMCB4AwGQYDVR0RBBIwEIIOd3d3Lmdvb2dsZS5jb20waAYIKwYB
BQUHAQEEXDBaMC0GCCsGAQUFBzAChiFodHRwOi8vcGtpLmdvb2cvZ3NyMi9HVFNH
SUFHMy5jcnQwKQYIKwYBBQUHMAGGHWh0dHA6Ly9vY3NwLnBraS5nb29nL0dUU0dJ
QUczMB0GA1UdDgQWBBQLovm8GG0oG91gOGCL58YPNoAlejAMBgNVHRMBAf8EAjAA
MB8GA1UdIwQYMBaAFHfCuFCaZ3Z2sS3ChtCDoH6mfrpLMCEGA1UdIAQaMBgwDAYK
KwYBBAHWeQIFAzAIBgZngQwBAgIwMQYDVR0fBCowKDAmoCSgIoYgaHR0cDovL2Ny
bC5wa2kuZ29vZy9HVFNHSUFHMy5jcmwwDQYJKoZIhvcNAQELBQADggEBAKFbmNOA
e3pJ7UVI5EmkAMZgSDRdrsLHV6F7WluuyYCyE/HFpZjBd6y8xgGtYWcask6edwrq
zrcXNEN/GY34AYre0M+p0xAs+lKSwkrJd2sCgygmzsBFtGwjW6lhjm+rg83zPHhH
mQZ0ShUR1Kp4TvzXgxj44RXOsS5ZyDe3slGiG4aw/hl+igO8Y8JMvcv/Tpzo+V75
BkDAFmLRi08NayfeyCqK/TcRpzxKMKhS7jEHK8Pzu5P+FyFHKqIsobi+BA+psOix
5nZLhrweLdKNz387mE2lSSKzr7qeLGHSOMt+ajQtZio4YVyZqJvg4Y++J0n5+Rjw
MXp8GrvTfn1DQ+o=
-----END CERTIFICATE-----
"""
EC_CERT_EXAMPLE = parse_certificate(EC_CERT_STR)


ECDSA_PRIME256V1_CERT_STR = """
-----BEGIN CERTIFICATE-----
MIICUTCCAfYCCQCvH7H/e2nuiDAKBggqhkjOPQQDAjCBrzELMAkGA1UEBhMCVVMx
EzARBgNVBAgMCkNhbGlmb3JuaWExEjAQBgNVBAcMCUxvcyBHYXRvczEjMCEGA1UE
CgwaTGVtdXJUcnVzdCBFbnRlcnByaXNlcyBMdGQxJjAkBgNVBAsMHVVuaXR0ZXN0
aW5nIE9wZXJhdGlvbnMgQ2VudGVyMSowKAYDVQQDDCFMZW11clRydXN0IFVuaXR0
ZXN0cyBSb290IENBIDIwMTkwHhcNMTkwMjI2MTgxMTUyWhcNMjkwMjIzMTgxMTUy
WjCBrzELMAkGA1UEBhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEjAQBgNVBAcM
CUxvcyBHYXRvczEjMCEGA1UECgwaTGVtdXJUcnVzdCBFbnRlcnByaXNlcyBMdGQx
JjAkBgNVBAsMHVVuaXR0ZXN0aW5nIE9wZXJhdGlvbnMgQ2VudGVyMSowKAYDVQQD
DCFMZW11clRydXN0IFVuaXR0ZXN0cyBSb290IENBIDIwMTkwWTATBgcqhkjOPQIB
BggqhkjOPQMBBwNCAAQsnAVUtpDCFMK/k9Chynu8BWRVUBUYbGQ9Q9xeLR60J4fD
uBt48YpTqg5RMZEclVknMReXqTmqphOBo37/YVdlMAoGCCqGSM49BAMCA0kAMEYC
IQDQZ6xfBiCTHxY4GM4+zLeG1iPBUSfIJOjkFNViFZY/XAIhAJYmrkVQb/YjWCdd
Vl89McYhmV4IV7WDgUmUhkUSFXgy
-----END CERTIFICATE-----
"""
ECDSA_PRIME256V1_CERT = parse_certificate(ECDSA_PRIME256V1_CERT_STR)


ECDSA_SECP384r1_CERT_STR = """
-----BEGIN CERTIFICATE-----
MIICjjCCAhMCCQD2UadeQ7ub1jAKBggqhkjOPQQDAjCBrzELMAkGA1UEBhMCVVMx
EzARBgNVBAgMCkNhbGlmb3JuaWExEjAQBgNVBAcMCUxvcyBHYXRvczEjMCEGA1UE
CgwaTGVtdXJUcnVzdCBFbnRlcnByaXNlcyBMdGQxJjAkBgNVBAsMHVVuaXR0ZXN0
aW5nIE9wZXJhdGlvbnMgQ2VudGVyMSowKAYDVQQDDCFMZW11clRydXN0IFVuaXR0
ZXN0cyBSb290IENBIDIwMTgwHhcNMTkwMjI2MTgxODU2WhcNMjkwMjIzMTgxODU2
WjCBrzELMAkGA1UEBhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEjAQBgNVBAcM
CUxvcyBHYXRvczEjMCEGA1UECgwaTGVtdXJUcnVzdCBFbnRlcnByaXNlcyBMdGQx
JjAkBgNVBAsMHVVuaXR0ZXN0aW5nIE9wZXJhdGlvbnMgQ2VudGVyMSowKAYDVQQD
DCFMZW11clRydXN0IFVuaXR0ZXN0cyBSb290IENBIDIwMTgwdjAQBgcqhkjOPQIB
BgUrgQQAIgNiAARuKyHIRp2e6PB5UcY8L/bUdavkL5Zf3IegNKvaAsvkDenhDGAI
zwWgsk3rOo7jmpMibn7yJQn404uZovwyeKcApn8uVv8ltheeYAx+ySzzn/APxNGy
cye/nv1D9cDW628wCgYIKoZIzj0EAwIDaQAwZgIxANl1ljDH4ykNK2OaRqKOkBOW
cKk1SvtiEZDS/wytiZGCeaxYteSYF+3GE8V2W1geWAIxAI8D7DY0HU5zw+oxAlTD
Uw/TeHA6q0QV4otPvrINW3V09iXDwFSPe265fTkHSfT6hQ==
-----END CERTIFICATE-----
"""
ECDSA_SECP384r1_CERT = parse_certificate(ECDSA_SECP384r1_CERT_STR)

DSA_CERT_STR = """
-----BEGIN CERTIFICATE-----
MIIDmTCCA1YCCQD5h/cM7xYO9jALBglghkgBZQMEAwIwga8xCzAJBgNVBAYTAlVT
MRMwEQYDVQQIDApDYWxpZm9ybmlhMRIwEAYDVQQHDAlMb3MgR2F0b3MxIzAhBgNV
BAoMGkxlbXVyVHJ1c3QgRW50ZXJwcmlzZXMgTHRkMSYwJAYDVQQLDB1Vbml0dGVz
dGluZyBPcGVyYXRpb25zIENlbnRlcjEqMCgGA1UEAwwhTGVtdXJUcnVzdCBVbml0
dGVzdHMgUm9vdCBDQSAyMDE4MB4XDTE5MDIyNjE4MjUyMloXDTI5MDIyMzE4MjUy
Mlowga8xCzAJBgNVBAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlhMRIwEAYDVQQH
DAlMb3MgR2F0b3MxIzAhBgNVBAoMGkxlbXVyVHJ1c3QgRW50ZXJwcmlzZXMgTHRk
MSYwJAYDVQQLDB1Vbml0dGVzdGluZyBPcGVyYXRpb25zIENlbnRlcjEqMCgGA1UE
AwwhTGVtdXJUcnVzdCBVbml0dGVzdHMgUm9vdCBDQSAyMDE4MIIBtjCCASsGByqG
SM44BAEwggEeAoGBAO2+6wO20rn9K7RtXJ7/kCSVFzYZsY1RKvmJ6BBkMFIepBkz
2pk62tRhJgNH07GKF7pyTPRRKqt38CaPK4ERUpavx3Ok6vZ3PKq8tMac/PMKBmT1
Xfpch54KDlCdreEMJqYiCwbIyiSCR4+PCH+7xC5Uh0PIZo6otNWe3Wkk53CfAhUA
8d4YAtto6D30f7qkEa7DMAccUS8CgYAiv8r0k0aUEaeioblcCAjmhvE0v8/tD5u1
anHO4jZIIv7uOrNFIGfqcNEOBs5AQkt5Bxn6x0b/VvtZ0FSrD0j4f36pTgro6noG
/0oRt0JngxsMSfo0LV4+bY62v21A0SneNgTgY+ugdfgGWvb0+9tpsIhiY69T+7c8
Oa0S6OWSPAOBhAACgYB5wa+nJJNZPoTWFum27JlWGYLO2flg5EpWlOvcEE0o5RfB
FPnMM033kKQQEI0YpCAq9fIMKhhUMk1X4mKUBUTt+Nrn1pY2l/wt5G6AQdHI8QXz
P1ecBbHPNZtWe3iVnfOgz/Pd8tU9slcXP9z5XbZ7R/oGcF/TPRTtbLEkYZNaDDAL
BglghkgBZQMEAwIDMAAwLQIVANubSNMSLt8plN9ZV3cp4pe3lMYCAhQPLLE7rTgm
92X+hWfyz000QEpYEQ==
-----END CERTIFICATE-----
"""
DSA_CERT = parse_certificate(DSA_CERT_STR)


CERT_CHAIN_PKCS7_STR = """
-----BEGIN PKCS7-----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-----END PKCS7-----
"""

CERT_CHAIN_PKCS7_PEM = CERT_CHAIN_PKCS7_STR.encode('utf-8')

ACME_CHAIN_LONG_STR = SAN_CERT_STR + """
-----BEGIN CERTIFICATE-----
MIIFWzCCA0OgAwIBAgIQTfQrldHumzpMLrM7jRBd1jANBgkqhkiG9w0BAQsFADBm
MQswCQYDVQQGEwJVUzEzMDEGA1UEChMqKFNUQUdJTkcpIEludGVybmV0IFNlY3Vy
aXR5IFJlc2VhcmNoIEdyb3VwMSIwIAYDVQQDExkoU1RBR0lORykgUHJldGVuZCBQ
ZWFyIFgxMB4XDTIwMDkwNDAwMDAwMFoXDTI1MDkxNTE2MDAwMFowWTELMAkGA1UE
BhMCVVMxIDAeBgNVBAoTFyhTVEFHSU5HKSBMZXQncyBFbmNyeXB0MSgwJgYDVQQD
Ex8oU1RBR0lORykgQXJ0aWZpY2lhbCBBcHJpY290IFIzMIIBIjANBgkqhkiG9w0B
AQEFAAOCAQ8AMIIBCgKCAQEAu6TR8+74b46mOE1FUwBrvxzEYLck3iasmKrcQkb+
gy/z9Jy7QNIAl0B9pVKp4YU76JwxF5DOZZhi7vK7SbCkK6FbHlyU5BiDYIxbbfvO
L/jVGqdsSjNaJQTg3C3XrJja/HA4WCFEMVoT2wDZm8ABC1N+IQe7Q6FEqc8NwmTS
nmmRQm4TQvr06DP+zgFK/MNubxWWDSbSKKTH5im5j2fZfg+j/tM1bGaczFWw8/lS
nukyn5J2L+NJYnclzkXoh9nMFnyPmVbfyDPOc4Y25aTzVoeBKXa/cZ5MM+WddjdL
biWvm19f1sYn1aRaAIrkppv7kkn83vcth8XCG39qC2ZvaQIDAQABo4IBEDCCAQww
DgYDVR0PAQH/BAQDAgGGMB0GA1UdJQQWMBQGCCsGAQUFBwMCBggrBgEFBQcDATAS
BgNVHRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBTecnpI3zHDplDfn4Uj31c3S10u
ZTAfBgNVHSMEGDAWgBS182Xy/rAKkh/7PH3zRKCsYyXDFDA2BggrBgEFBQcBAQQq
MCgwJgYIKwYBBQUHMAKGGmh0dHA6Ly9zdGcteDEuaS5sZW5jci5vcmcvMCsGA1Ud
HwQkMCIwIKAeoByGGmh0dHA6Ly9zdGcteDEuYy5sZW5jci5vcmcvMCIGA1UdIAQb
MBkwCAYGZ4EMAQIBMA0GCysGAQQBgt8TAQEBMA0GCSqGSIb3DQEBCwUAA4ICAQCN
DLam9yN0EFxxn/3p+ruWO6n/9goCAM5PT6cC6fkjMs4uas6UGXJjr5j7PoTQf3C1
vuxiIGRJC6qxV7yc6U0X+w0Mj85sHI5DnQVWN5+D1er7mp13JJA0xbAbHa3Rlczn
y2Q82XKui8WHuWra0gb2KLpfboYj1Ghgkhr3gau83pC/WQ8HfkwcvSwhIYqTqxoZ
Uq8HIf3M82qS9aKOZE0CEmSyR1zZqQxJUT7emOUapkUN9poJ9zGc+FgRZvdro0XB
yphWXDaqMYph0DxW/10ig5j4xmmNDjCRmqIKsKoWA52wBTKKXK1na2ty/lW5dhtA
xkz5rVZFd4sgS4J0O+zm6d5GRkWsNJ4knotGXl8vtS3X40KXeb3A5+/3p0qaD215
Xq8oSNORfB2oI1kQuyEAJ5xvPTdfwRlyRG3lFYodrRg6poUBD/8fNTXMtzydpRgy
zUQZh/18F6B/iW6cbiRN9r2Hkh05Om+q0/6w0DdZe+8YrNpfhSObr/1eVZbKGMIY
qKmyZbBNu5ysENIK5MPc14mUeKmFjpN840VR5zunoU52lqpLDua/qIM8idk86xGW
xx2ml43DO/Ya/tVZVok0mO0TUjzJIfPqyvr455IsIut4RlCR9Iq0EDTve2/ZwCuG
hSjpTUFGSiQrR2JK2Evp+o6AETUkBCO1aw0PpQBPDQ==
-----END CERTIFICATE-----
-----BEGIN CERTIFICATE-----
MIIFVDCCBDygAwIBAgIRAO1dW8lt+99NPs1qSY3Rs8cwDQYJKoZIhvcNAQELBQAw
cTELMAkGA1UEBhMCVVMxMzAxBgNVBAoTKihTVEFHSU5HKSBJbnRlcm5ldCBTZWN1
cml0eSBSZXNlYXJjaCBHcm91cDEtMCsGA1UEAxMkKFNUQUdJTkcpIERvY3RvcmVk
IER1cmlhbiBSb290IENBIFgzMB4XDTIxMDEyMDE5MTQwM1oXDTI0MDkzMDE4MTQw
M1owZjELMAkGA1UEBhMCVVMxMzAxBgNVBAoTKihTVEFHSU5HKSBJbnRlcm5ldCBT
ZWN1cml0eSBSZXNlYXJjaCBHcm91cDEiMCAGA1UEAxMZKFNUQUdJTkcpIFByZXRl
bmQgUGVhciBYMTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBALbagEdD
Ta1QgGBWSYkyMhscZXENOBaVRTMX1hceJENgsL0Ma49D3MilI4KS38mtkmdF6cPW
nL++fgehT0FbRHZgjOEr8UAN4jH6omjrbTD++VZneTsMVaGamQmDdFl5g1gYaigk
kmx8OiCO68a4QXg4wSyn6iDipKP8utsE+x1E28SA75HOYqpdrk4HGxuULvlr03wZ
GTIf/oRt2/c+dYmDoaJhge+GOrLAEQByO7+8+vzOwpNAPEx6LW+crEEZ7eBXih6V
P19sTGy3yfqK5tPtTdXXCOQMKAp+gCj/VByhmIr+0iNDC540gtvV303WpcbwnkkL
YC0Ft2cYUyHtkstOfRcRO+K2cZozoSwVPyB8/J9RpcRK3jgnX9lujfwA/pAbP0J2
UPQFxmWFRQnFjaq6rkqbNEBgLy+kFL1NEsRbvFbKrRi5bYy2lNms2NJPZvdNQbT/
2dBZKmJqxHkxCuOQFjhJQNeO+Njm1Z1iATS/3rts2yZlqXKsxQUzN6vNbD8KnXRM
EeOXUYvbV4lqfCf8mS14WEbSiMy87GB5S9ucSV1XUrlTG5UGcMSZOBcEUpisRPEm
QWUOTWIoDQ5FOia/GI+Ki523r2ruEmbmG37EBSBXdxIdndqrjy+QVAmCebyDx9eV
EGOIpn26bW5LKerumJxa/CFBaKi4bRvmdJRLAgMBAAGjgfEwge4wDgYDVR0PAQH/
BAQDAgEGMA8GA1UdEwEB/wQFMAMBAf8wHQYDVR0OBBYEFLXzZfL+sAqSH/s8ffNE
oKxjJcMUMB8GA1UdIwQYMBaAFAhX2onHolN5DE/d4JCPdLriJ3NEMDgGCCsGAQUF
BwEBBCwwKjAoBggrBgEFBQcwAoYcaHR0cDovL3N0Zy1kc3QzLmkubGVuY3Iub3Jn
LzAtBgNVHR8EJjAkMCKgIKAehhxodHRwOi8vc3RnLWRzdDMuYy5sZW5jci5vcmcv
MCIGA1UdIAQbMBkwCAYGZ4EMAQIBMA0GCysGAQQBgt8TAQEBMA0GCSqGSIb3DQEB
CwUAA4IBAQB7tR8B0eIQSS6MhP5kuvGth+dN02DsIhr0yJtk2ehIcPIqSxRRmHGl
4u2c3QlvEpeRDp2w7eQdRTlI/WnNhY4JOofpMf2zwABgBWtAu0VooQcZZTpQruig
F/z6xYkBk3UHkjeqxzMN3d1EqGusxJoqgdTouZ5X5QTTIee9nQ3LEhWnRSXDx7Y0
ttR1BGfcdqHopO4IBqAhbkKRjF5zj7OD8cG35omywUbZtOJnftiI0nFcRaxbXo0v
oDfLD0S6+AC2R3tKpqjkNX6/91hrRFglUakyMcZU/xleqbv6+Lr3YD8PsBTub6lI
oZ2lS38fL18Aon458fbc0BPHtenfhKj5
-----END CERTIFICATE-----
"""

ACME_CHAIN_SHORT_STR = SAN_CERT_STR + """
-----BEGIN CERTIFICATE-----
MIIFWzCCA0OgAwIBAgIQTfQrldHumzpMLrM7jRBd1jANBgkqhkiG9w0BAQsFADBm
MQswCQYDVQQGEwJVUzEzMDEGA1UEChMqKFNUQUdJTkcpIEludGVybmV0IFNlY3Vy
aXR5IFJlc2VhcmNoIEdyb3VwMSIwIAYDVQQDExkoU1RBR0lORykgUHJldGVuZCBQ
ZWFyIFgxMB4XDTIwMDkwNDAwMDAwMFoXDTI1MDkxNTE2MDAwMFowWTELMAkGA1UE
BhMCVVMxIDAeBgNVBAoTFyhTVEFHSU5HKSBMZXQncyBFbmNyeXB0MSgwJgYDVQQD
Ex8oU1RBR0lORykgQXJ0aWZpY2lhbCBBcHJpY290IFIzMIIBIjANBgkqhkiG9w0B
AQEFAAOCAQ8AMIIBCgKCAQEAu6TR8+74b46mOE1FUwBrvxzEYLck3iasmKrcQkb+
gy/z9Jy7QNIAl0B9pVKp4YU76JwxF5DOZZhi7vK7SbCkK6FbHlyU5BiDYIxbbfvO
L/jVGqdsSjNaJQTg3C3XrJja/HA4WCFEMVoT2wDZm8ABC1N+IQe7Q6FEqc8NwmTS
nmmRQm4TQvr06DP+zgFK/MNubxWWDSbSKKTH5im5j2fZfg+j/tM1bGaczFWw8/lS
nukyn5J2L+NJYnclzkXoh9nMFnyPmVbfyDPOc4Y25aTzVoeBKXa/cZ5MM+WddjdL
biWvm19f1sYn1aRaAIrkppv7kkn83vcth8XCG39qC2ZvaQIDAQABo4IBEDCCAQww
DgYDVR0PAQH/BAQDAgGGMB0GA1UdJQQWMBQGCCsGAQUFBwMCBggrBgEFBQcDATAS
BgNVHRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBTecnpI3zHDplDfn4Uj31c3S10u
ZTAfBgNVHSMEGDAWgBS182Xy/rAKkh/7PH3zRKCsYyXDFDA2BggrBgEFBQcBAQQq
MCgwJgYIKwYBBQUHMAKGGmh0dHA6Ly9zdGcteDEuaS5sZW5jci5vcmcvMCsGA1Ud
HwQkMCIwIKAeoByGGmh0dHA6Ly9zdGcteDEuYy5sZW5jci5vcmcvMCIGA1UdIAQb
MBkwCAYGZ4EMAQIBMA0GCysGAQQBgt8TAQEBMA0GCSqGSIb3DQEBCwUAA4ICAQCN
DLam9yN0EFxxn/3p+ruWO6n/9goCAM5PT6cC6fkjMs4uas6UGXJjr5j7PoTQf3C1
vuxiIGRJC6qxV7yc6U0X+w0Mj85sHI5DnQVWN5+D1er7mp13JJA0xbAbHa3Rlczn
y2Q82XKui8WHuWra0gb2KLpfboYj1Ghgkhr3gau83pC/WQ8HfkwcvSwhIYqTqxoZ
Uq8HIf3M82qS9aKOZE0CEmSyR1zZqQxJUT7emOUapkUN9poJ9zGc+FgRZvdro0XB
yphWXDaqMYph0DxW/10ig5j4xmmNDjCRmqIKsKoWA52wBTKKXK1na2ty/lW5dhtA
xkz5rVZFd4sgS4J0O+zm6d5GRkWsNJ4knotGXl8vtS3X40KXeb3A5+/3p0qaD215
Xq8oSNORfB2oI1kQuyEAJ5xvPTdfwRlyRG3lFYodrRg6poUBD/8fNTXMtzydpRgy
zUQZh/18F6B/iW6cbiRN9r2Hkh05Om+q0/6w0DdZe+8YrNpfhSObr/1eVZbKGMIY
qKmyZbBNu5ysENIK5MPc14mUeKmFjpN840VR5zunoU52lqpLDua/qIM8idk86xGW
xx2ml43DO/Ya/tVZVok0mO0TUjzJIfPqyvr455IsIut4RlCR9Iq0EDTve2/ZwCuG
hSjpTUFGSiQrR2JK2Evp+o6AETUkBCO1aw0PpQBPDQ==
-----END CERTIFICATE-----
"""

IP_SAN_NO_CN_CERT_STR = """
-----BEGIN CERTIFICATE-----
MIICkzCCAXugAwIBAgIJAMkF8ROZ675BMA0GCSqGSIb3DQEBBQUAMIGnMS0wKwYD
VQQDDCRMZW11clRydXN0IFVuaXR0ZXN0cyBDbGFzcyAxIENBIDIwMTgxIzAhBgNV
BAoMGkxlbXVyVHJ1c3QgRW50ZXJwcmlzZXMgTHRkMSYwJAYDVQQLDB1Vbml0dGVz
dGluZyBPcGVyYXRpb25zIENlbnRlcjELMAkGA1UEBhMCRUUxDDAKBgNVBAgMA04v
QTEOMAwGA1UEBwwFRWFydGgwHhcNMjExMTA4MjMyNjMzWhcNMjExMTA5MjMyNjMz
WjAgMR4wHAYJKoZIhvcNAQkBFg9qb2VAZXhhbXBsZS5jb20wWTATBgcqhkjOPQIB
BggqhkjOPQMBBwNCAATrE43GpSEMQZ1Z0vuWb+oGI4gx1eTXqK65HwgIQtspx9z0
TkgugyIrfua7BB1W832uuINmqPc7yWYYWA0iRZQCoxMwETAPBgNVHREECDAGhwTA
qAcBMA0GCSqGSIb3DQEBBQUAA4IBAQCFqYYm83M+lXN0UFzTGSSJ3NVUw4WJPTu8
DQjuAOtBCMCzbSvEmZu1cpY5pApgTAGp31p9J6qTQHMh0nFyFXordoU0NPzqzqnY
i8rLuoy2ouRCdLUZFsvxid6KhWlcyMkLx5Rm0ivEj7LphEysJLIH/LHinAjxFptN
fKFOqtCbXjrdMGzZAicsRK5wq0z0/M/66MUcQCLP4Af0REK9zkC2zgYLWXwyYJGo
s1Ji7VnxGbEXhldv1FotvgKfTgWHA+7XFRBHt57VcezIvnuuz26V6igzk6rOKBjc
sBWmooFdbP7Hc7KCHOLeH4zJPcEGr1O4UGuwO9JAXqNoKIBEQwix
-----END CERTIFICATE-----
"""
