"""Reference data for D3 calculations."""

from typing import Dict

import numpy as np

from morfeus.typing import Array2D

# These values are used in the D3 dispersion model. Taken from Grimme's program.
r2_r4: Dict[int, float] = {
    1: 2.00734898,
    2: 1.56637132,
    3: 5.01986934,
    4: 3.85379032,
    5: 3.64446594,
    6: 3.10492822,
    7: 2.71175247,
    8: 2.5936168,
    9: 2.3882525,
    10: 2.21522516,
    11: 6.58585536,
    12: 5.46295967,
    13: 5.65216669,
    14: 4.88284902,
    15: 4.29727576,
    16: 4.04108902,
    17: 3.72932356,
    18: 3.44677275,
    19: 7.97762753,
    20: 7.07623947,
    21: 6.60844053,
    22: 6.28791364,
    23: 6.07728703,
    24: 5.54643096,
    25: 5.80491167,
    26: 5.58415602,
    27: 5.41374528,
    28: 5.28497229,
    29: 5.22592821,
    30: 5.09817141,
    31: 6.12149689,
    32: 5.54083734,
    33: 5.06696878,
    34: 4.87005108,
    35: 4.59089647,
    36: 4.31176304,
    37: 9.55461698,
    38: 8.67396077,
    39: 7.97210197,
    40: 7.43439917,
    41: 6.58711862,
    42: 6.19536215,
    43: 6.0151729,
    44: 5.8162341,
    45: 5.65710424,
    46: 5.52640661,
    47: 5.44263305,
    48: 5.58285373,
    49: 7.02081898,
    50: 6.46815523,
    51: 5.9808912,
    52: 5.81686657,
    53: 5.53321815,
    54: 5.25477007,
    55: 11.02204549,
    56: 10.15679528,
    57: 9.35167836,
    58: 9.06926079,
    59: 8.97241155,
    60: 8.90092807,
    61: 8.8598484,
    62: 8.81736827,
    63: 8.7931771,
    64: 7.89969626,
    65: 8.80588454,
    66: 8.42439218,
    67: 8.54289262,
    68: 8.4758337,
    69: 8.45090888,
    70: 8.47339339,
    71: 7.83525634,
    72: 8.20702843,
    73: 7.70559063,
    74: 7.32755997,
    75: 7.03887381,
    76: 6.6897872,
    77: 6.05450052,
    78: 5.88752022,
    79: 5.70661499,
    80: 5.78450695,
    81: 7.79780729,
    82: 7.26443867,
    83: 6.78151984,
    84: 6.67883169,
    85: 6.39024318,
    86: 6.09527958,
    87: 11.79156076,
    88: 11.10997644,
    89: 9.51377795,
    90: 8.67197068,
    91: 8.77140725,
    92: 8.65402716,
    93: 8.53923501,
    94: 8.85024712,
}
"""Atomic numbers as keys and r2r4 values as values."""

c6_reference_data: Dict[int, Array2D] = {
    1: np.array(
        [
            [3.0267, 0.9118, 0.9118],
            [4.7379, 0.0, 0.9118],
            [4.7379, 0.9118, 0.0],
            [7.5916, 0.0, 0.0],
        ]
    ),
    2: np.array([[1.5583, 0.0, 0.0]]),
    3: np.array(
        [
            [8.5319700e01, 9.8650000e-01, 9.8650000e-01],
            [2.8221060e02, 0.0000000e00, 9.8650000e-01],
            [2.8221060e02, 9.8650000e-01, 0.0000000e00],
            [1.1634454e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    4: np.array(
        [
            [55.1364, 1.9697, 1.9697],
            [84.4, 0.9808, 1.9697],
            [84.4, 1.9697, 0.9808],
            [113.9677, 0.0, 1.9697],
            [113.9677, 1.9697, 0.0],
            [135.8445, 0.9808, 0.9808],
            [186.3453, 0.0, 0.9808],
            [186.3453, 0.9808, 0.0],
            [257.4863, 0.0, 0.0],
        ]
    ),
    5: np.array(
        [
            [28.0315, 4.5856, 4.5856],
            [29.6804, 2.9128, 4.5856],
            [29.6804, 4.5856, 2.9128],
            [31.4436, 2.9128, 2.9128],
            [38.1398, 1.9441, 4.5856],
            [38.1398, 4.5856, 1.9441],
            [40.5192, 1.9441, 2.9128],
            [40.5192, 2.9128, 1.9441],
            [44.9888, 0.9706, 4.5856],
            [44.9888, 4.5856, 0.9706],
            [47.858, 0.9706, 2.9128],
            [47.858, 2.9128, 0.9706],
            [52.9606, 0.0, 4.5856],
            [52.9606, 4.5856, 0.0],
            [53.1128, 1.9441, 1.9441],
            [56.4116, 0.0, 2.9128],
            [56.4116, 2.9128, 0.0],
            [63.1685, 0.9706, 1.9441],
            [63.1685, 1.9441, 0.9706],
            [74.9865, 0.0, 1.9441],
            [74.9865, 1.9441, 0.0],
            [75.3686, 0.9706, 0.9706],
            [89.7429, 0.0, 0.9706],
            [89.7429, 0.9706, 0.0],
            [107.1777, 0.0, 0.0],
        ]
    ),
    6: np.array(
        [
            [18.2067, 3.9844, 3.9844],
            [21.5377, 2.9987, 3.9844],
            [21.5377, 3.9844, 2.9987],
            [22.9517, 1.9985, 3.9844],
            [22.9517, 3.9844, 1.9985],
            [25.7809, 2.9987, 2.9987],
            [27.5063, 1.9985, 2.9987],
            [27.5063, 2.9987, 1.9985],
            [27.5206, 0.9868, 3.9844],
            [27.5206, 3.9844, 0.9868],
            [29.283, 0.0, 3.9844],
            [29.283, 3.9844, 0.0],
            [29.3602, 1.9985, 1.9985],
            [33.254, 0.9868, 2.9987],
            [33.254, 2.9987, 0.9868],
            [35.4129, 0.0, 2.9987],
            [35.4129, 2.9987, 0.0],
            [35.5219, 0.9868, 1.9985],
            [35.5219, 1.9985, 0.9868],
            [37.8419, 0.0, 1.9985],
            [37.8419, 1.9985, 0.0],
            [43.2452, 0.9868, 0.9868],
            [46.0681, 0.0, 0.9868],
            [46.0681, 0.9868, 0.0],
            [49.113, 0.0, 0.0],
        ]
    ),
    7: np.array(
        [
            [15.5817, 2.9903, 2.9903],
            [17.4928, 2.0143, 2.9903],
            [17.4928, 2.9903, 2.0143],
            [18.518, 0.9944, 2.9903],
            [18.518, 2.9903, 0.9944],
            [19.6768, 2.0143, 2.0143],
            [19.7707, 0.0, 2.9903],
            [19.7707, 2.9903, 0.0],
            [20.8501, 0.9944, 2.0143],
            [20.8501, 2.0143, 0.9944],
            [22.1241, 0.9944, 0.9944],
            [22.2794, 0.0, 2.0143],
            [22.2794, 2.0143, 0.0],
            [23.6295, 0.0, 0.9944],
            [23.6295, 0.9944, 0.0],
            [25.2685, 0.0, 0.0],
        ]
    ),
    8: np.array(
        [
            [10.3708, 1.9887, 1.9887],
            [11.5009, 0.9925, 1.9887],
            [11.5009, 1.9887, 0.9925],
            [12.6277, 0.0, 1.9887],
            [12.6277, 1.9887, 0.0],
            [12.8161, 0.9925, 0.9925],
            [14.0764, 0.0, 0.9925],
            [14.0764, 0.9925, 0.0],
            [15.5059, 0.0, 0.0],
        ]
    ),
    9: np.array(
        [
            [7.1341, 0.9982, 0.9982],
            [8.2739, 0.0, 0.9982],
            [8.2739, 0.9982, 0.0],
            [9.6916, 0.0, 0.0],
        ]
    ),
    10: np.array([[6.2896, 0.0, 0.0]]),
    11: np.array(
        [
            [1.8610520e02, 9.6840000e-01, 9.6840000e-01],
            [5.0543630e02, 0.0000000e00, 9.6840000e-01],
            [5.0543630e02, 9.6840000e-01, 0.0000000e00],
            [1.6080286e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    12: np.array(
        [
            [175.5616, 1.9496, 1.9496],
            [274.159, 0.9628, 1.9496],
            [274.159, 1.9496, 0.9628],
            [334.2493, 0.0, 1.9496],
            [334.2493, 1.9496, 0.0],
            [451.2708, 0.9628, 0.9628],
            [554.5967, 0.0, 0.9628],
            [554.5967, 0.9628, 0.0],
            [683.3758, 0.0, 0.0],
        ]
    ),
    13: np.array(
        [
            [153.5945, 2.9146, 2.9146],
            [196.9411, 1.9311, 2.9146],
            [196.9411, 2.9146, 1.9311],
            [252.4953, 0.9648, 2.9146],
            [252.4953, 2.9146, 0.9648],
            [256.7497, 1.9311, 1.9311],
            [278.5639, 0.0, 2.9146],
            [278.5639, 2.9146, 0.0],
            [333.5341, 0.9648, 1.9311],
            [333.5341, 1.9311, 0.9648],
            [369.3829, 0.0, 1.9311],
            [369.3829, 1.9311, 0.0],
            [437.7575, 0.9648, 0.9648],
            [486.2365, 0.0, 0.9648],
            [486.2365, 0.9648, 0.0],
            [540.5406, 0.0, 0.0],
        ]
    ),
    14: np.array(
        [
            [149.7734, 3.8677, 3.8677],
            [181.0237, 2.9407, 3.8677],
            [181.0237, 3.8677, 2.9407],
            [185.9996, 1.9435, 3.8677],
            [185.9996, 3.8677, 1.9435],
            [199.753, 0.9507, 3.8677],
            [199.753, 3.8677, 0.9507],
            [215.8998, 0.0, 3.8677],
            [215.8998, 3.8677, 0.0],
            [221.5046, 2.9407, 2.9407],
            [227.1882, 1.9435, 2.9407],
            [227.1882, 2.9407, 1.9435],
            [233.1254, 1.9435, 1.9435],
            [244.6288, 0.9507, 2.9407],
            [244.6288, 2.9407, 0.9507],
            [250.9376, 0.9507, 1.9435],
            [250.9376, 1.9435, 0.9507],
            [265.1189, 0.0, 2.9407],
            [265.1189, 2.9407, 0.0],
            [270.265, 0.9507, 0.9507],
            [271.8844, 0.0, 1.9435],
            [271.8844, 1.9435, 0.0],
            [292.9994, 0.0, 0.9507],
            [292.9994, 0.9507, 0.0],
            [317.8574, 0.0, 0.0],
        ]
    ),
    15: np.array(
        [
            [151.6886, 2.9859, 2.9859],
            [163.0764, 2.0102, 2.9859],
            [163.0764, 2.9859, 2.0102],
            [165.4518, 0.9947, 2.9859],
            [165.4518, 2.9859, 0.9947],
            [170.4045, 0.0, 2.9859],
            [170.4045, 2.9859, 0.0],
            [175.5534, 2.0102, 2.0102],
            [178.0622, 0.9947, 2.0102],
            [178.0622, 2.0102, 0.9947],
            [180.6238, 0.9947, 0.9947],
            [183.4322, 0.0, 2.0102],
            [183.4322, 2.0102, 0.0],
            [186.0684, 0.0, 0.9947],
            [186.0684, 0.9947, 0.0],
            [191.6887, 0.0, 0.0],
        ]
    ),
    16: np.array(
        [
            [125.8109, 1.9903, 1.9903],
            [128.3764, 0.9948, 1.9903],
            [128.3764, 1.9903, 0.9948],
            [129.8413, 0.0, 1.9903],
            [129.8413, 1.9903, 0.0],
            [130.9965, 0.9948, 0.9948],
            [132.4922, 0.0, 0.9948],
            [132.4922, 0.9948, 0.0],
            [134.0066, 0.0, 0.0],
        ]
    ),
    17: np.array(
        [
            [90.3985, 0.9972, 0.9972],
            [91.3667, 0.0, 0.9972],
            [91.3667, 0.9972, 0.0],
            [92.346, 0.0, 0.0],
        ]
    ),
    18: np.array([[64.6462, 0.0, 0.0]]),
    19: np.array(
        [
            [3.3802070e02, 9.7670000e-01, 9.7670000e-01],
            [1.1122277e03, 0.0000000e00, 9.7670000e-01],
            [1.1122277e03, 9.7670000e-01, 0.0000000e00],
            [4.9835009e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    20: np.array(
        [
            [3.4333430e02, 1.9349000e00, 1.9349000e00],
            [6.6406200e02, 9.8310000e-01, 1.9349000e00],
            [6.6406200e02, 1.9349000e00, 9.8310000e-01],
            [8.3223790e02, 0.0000000e00, 1.9349000e00],
            [8.3223790e02, 1.9349000e00, 0.0000000e00],
            [1.5251891e03, 9.8310000e-01, 9.8310000e-01],
            [1.8864221e03, 0.0000000e00, 9.8310000e-01],
            [1.8864221e03, 9.8310000e-01, 0.0000000e00],
            [2.3526862e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    21: np.array(
        [
            [313.9846, 2.8999, 2.8999],
            [318.8004, 1.8627, 2.8999],
            [318.8004, 2.8999, 1.8627],
            [324.5068, 1.8627, 1.8627],
            [638.4869, 0.0, 2.8999],
            [638.4869, 2.8999, 0.0],
            [660.678, 0.0, 1.8627],
            [660.678, 1.8627, 0.0],
            [1522.4676, 0.0, 0.0],
        ]
    ),
    22: np.array(
        [
            [301.9988, 1.8299, 1.8299],
            [311.2408, 1.8299, 3.8675],
            [311.2408, 3.8675, 1.8299],
            [321.589, 3.8675, 3.8675],
            [612.2225, 0.0, 1.8299],
            [612.2225, 1.8299, 0.0],
            [621.6228, 0.0, 3.8675],
            [621.6228, 3.8675, 0.0],
            [1361.9185, 0.0, 0.0],
        ]
    ),
    23: np.array(
        [
            [269.3349, 2.911, 2.911],
            [279.2755, 1.9138, 2.911],
            [279.2755, 2.911, 1.9138],
            [290.1436, 1.9138, 1.9138],
            [521.4602, 0.0, 2.911],
            [521.4602, 2.911, 0.0],
            [548.3908, 0.0, 1.9138],
            [548.3908, 1.9138, 0.0],
            [1116.0984, 0.0, 0.0],
        ]
    ),
    24: np.array(
        [
            [153.3435, 10.6191, 10.6191],
            [206.3426, 1.8269, 10.6191],
            [206.3426, 10.6191, 1.8269],
            [285.6905, 1.8269, 1.8269],
            [298.4894, 0.0, 10.6191],
            [298.4894, 10.6191, 0.0],
            [430.2353, 0.0, 1.8269],
            [430.2353, 1.8269, 0.0],
            [690.7425, 0.0, 0.0],
        ]
    ),
    25: np.array(
        [
            [113.8463, 9.8849, 9.8849],
            [162.6844, 1.6406, 9.8849],
            [162.6844, 9.8849, 1.6406],
            [239.5554, 1.6406, 1.6406],
            [273.3937, 0.0, 9.8849],
            [273.3937, 9.8849, 0.0],
            [420.4207, 0.0, 1.6406],
            [420.4207, 1.6406, 0.0],
            [802.7484, 0.0, 0.0],
        ]
    ),
    26: np.array(
        [
            [109.5041, 9.1376, 9.1376],
            [148.3171, 1.6483, 9.1376],
            [148.3171, 9.1376, 1.6483],
            [204.9925, 1.6483, 1.6483],
            [212.1109, 0.0, 9.1376],
            [212.1109, 9.1376, 0.0],
            [303.048, 0.0, 1.6483],
            [303.048, 1.6483, 0.0],
            [491.3349, 0.0, 0.0],
        ]
    ),
    27: np.array(
        [
            [118.2863, 7.7785, 7.7785],
            [132.0085, 2.9263, 7.7785],
            [132.0085, 7.7785, 2.9263],
            [147.8057, 1.7149, 7.7785],
            [147.8057, 7.7785, 1.7149],
            [148.0657, 2.9263, 2.9263],
            [166.0911, 1.7149, 2.9263],
            [166.0911, 2.9263, 1.7149],
            [186.5462, 1.7149, 1.7149],
            [237.6994, 0.0, 7.7785],
            [237.6994, 7.7785, 0.0],
            [271.6288, 0.0, 2.9263],
            [271.6288, 2.9263, 0.0],
            [306.4968, 0.0, 1.7149],
            [306.4968, 1.7149, 0.0],
            [532.7794, 0.0, 0.0],
        ]
    ),
    28: np.array(
        [
            [128.782, 6.5458, 6.5458],
            [134.8933, 6.2918, 6.5458],
            [134.8933, 6.5458, 6.2918],
            [141.8763, 6.2918, 6.2918],
            [151.2409, 1.7937, 6.5458],
            [151.2409, 6.5458, 1.7937],
            [158.9405, 1.7937, 6.2918],
            [158.9405, 6.2918, 1.7937],
            [178.3311, 1.7937, 1.7937],
            [258.3888, 0.0, 6.5458],
            [258.3888, 6.5458, 0.0],
            [276.2327, 0.0, 6.2918],
            [276.2327, 6.2918, 0.0],
            [309.2543, 0.0, 1.7937],
            [309.2543, 1.7937, 0.0],
            [574.7436, 0.0, 0.0],
        ]
    ),
    29: np.array(
        [
            [174.9969, 0.9576, 0.9576],
            [238.1207, 0.0, 0.9576],
            [238.1207, 0.9576, 0.0],
            [337.1808, 0.0, 0.0],
        ]
    ),
    30: np.array(
        [
            [153.4528, 1.9419, 1.9419],
            [223.6622, 0.0, 1.9419],
            [223.6622, 1.9419, 0.0],
            [340.5213, 0.0, 0.0],
        ]
    ),
    31: np.array(
        [
            [172.3153, 2.9233, 2.9233],
            [209.513, 1.9315, 2.9233],
            [209.513, 2.9233, 1.9315],
            [257.8727, 1.9315, 1.9315],
            [262.9134, 0.9601, 2.9233],
            [262.9134, 2.9233, 0.9601],
            [279.4346, 0.0, 2.9233],
            [279.4346, 2.9233, 0.0],
            [327.6772, 0.9601, 1.9315],
            [327.6772, 1.9315, 0.9601],
            [349.5389, 0.0, 1.9315],
            [349.5389, 1.9315, 0.0],
            [421.6497, 0.9601, 0.9601],
            [451.4003, 0.0, 0.9601],
            [451.4003, 0.9601, 0.0],
            [483.7516, 0.0, 0.0],
        ]
    ),
    32: np.array(
        [
            [180.2912, 3.8972, 3.8972],
            [203.1922, 2.9186, 3.8972],
            [203.1922, 3.8972, 2.9186],
            [225.8139, 1.9447, 3.8972],
            [225.8139, 3.8972, 1.9447],
            [229.9498, 2.9186, 2.9186],
            [236.7867, 0.9434, 3.8972],
            [236.7867, 3.8972, 0.9434],
            [252.4591, 0.0, 3.8972],
            [252.4591, 3.8972, 0.0],
            [256.4009, 1.9447, 2.9186],
            [256.4009, 2.9186, 1.9447],
            [269.2215, 0.9434, 2.9186],
            [269.2215, 2.9186, 0.9434],
            [286.6545, 1.9447, 1.9447],
            [287.6103, 0.0, 2.9186],
            [287.6103, 2.9186, 0.0],
            [301.3102, 0.9434, 1.9447],
            [301.3102, 1.9447, 0.9434],
            [316.8521, 0.9434, 0.9434],
            [322.3974, 0.0, 1.9447],
            [322.3974, 1.9447, 0.0],
            [339.2404, 0.0, 0.9434],
            [339.2404, 0.9434, 0.0],
            [363.5474, 0.0, 0.0],
        ]
    ),
    33: np.array(
        [
            [224.2851, 2.9709, 2.9709],
            [232.5744, 1.9793, 2.9709],
            [232.5744, 2.9709, 1.9793],
            [238.6601, 0.9889, 2.9709],
            [238.6601, 2.9709, 0.9889],
            [241.2494, 1.9793, 1.9793],
            [242.6736, 0.0, 2.9709],
            [242.6736, 2.9709, 0.0],
            [247.6174, 0.9889, 1.9793],
            [247.6174, 1.9793, 0.9889],
            [251.8155, 0.0, 1.9793],
            [251.8155, 1.9793, 0.0],
            [254.1924, 0.9889, 0.9889],
            [258.5263, 0.0, 0.9889],
            [258.5263, 0.9889, 0.0],
            [262.9498, 0.0, 0.0],
        ]
    ),
    34: np.array(
        [
            [210.5231, 1.9812, 1.9812],
            [212.0915, 0.0, 1.9812],
            [212.0915, 1.9812, 0.0],
            [212.1475, 0.9901, 1.9812],
            [212.1475, 1.9812, 0.9901],
            [213.6738, 0.0, 0.0],
            [213.7298, 0.0, 0.9901],
            [213.7298, 0.9901, 0.0],
            [213.7862, 0.9901, 0.9901],
        ]
    ),
    35: np.array(
        [
            [167.1297, 0.0, 0.0],
            [168.1791, 0.0, 0.9974],
            [168.1791, 0.9974, 0.0],
            [169.2375, 0.9974, 0.9974],
        ]
    ),
    36: np.array([[130.4017, 0.0, 0.0]]),
    37: np.array(
        [
            [4.7625910e02, 9.7380000e-01, 9.7380000e-01],
            [1.4577817e03, 0.0000000e00, 9.7380000e-01],
            [1.4577817e03, 9.7380000e-01, 0.0000000e00],
            [6.1387755e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    38: np.array(
        [
            [4.9174480e02, 1.9143000e00, 1.9143000e00],
            [9.6807870e02, 9.8010000e-01, 1.9143000e00],
            [9.6807870e02, 1.9143000e00, 9.8010000e-01],
            [1.1858018e03, 0.0000000e00, 1.9143000e00],
            [1.1858018e03, 1.9143000e00, 0.0000000e00],
            [2.3126602e03, 9.8010000e-01, 9.8010000e-01],
            [2.7824577e03, 0.0000000e00, 9.8010000e-01],
            [2.7824577e03, 9.8010000e-01, 0.0000000e00],
            [3.3813672e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    39: np.array(
        [
            [4.5108450e02, 2.8903000e00, 2.8903000e00],
            [6.7945910e02, 1.9153000e00, 2.8903000e00],
            [6.7945910e02, 2.8903000e00, 1.9153000e00],
            [9.5501530e02, 0.0000000e00, 2.8903000e00],
            [9.5501530e02, 2.8903000e00, 0.0000000e00],
            [1.1322983e03, 1.9153000e00, 1.9153000e00],
            [1.6243636e03, 0.0000000e00, 1.9153000e00],
            [1.6243636e03, 1.9153000e00, 0.0000000e00],
            [2.3658925e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    40: np.array(
        [
            [458.6896, 3.9106, 3.9106],
            [507.5216, 1.9355, 3.9106],
            [507.5216, 3.9106, 1.9355],
            [568.6789, 1.9355, 1.9355],
            [857.6011, 0.0, 3.9106],
            [857.6011, 3.9106, 0.0],
            [987.8257, 0.0, 1.9355],
            [987.8257, 1.9355, 0.0],
            [1822.7181, 0.0, 0.0],
        ]
    ),
    41: np.array(
        [
            [428.8962, 2.9225, 2.9225],
            [496.1127, 1.9545, 2.9225],
            [496.1127, 2.9225, 1.9545],
            [580.5824, 1.9545, 1.9545],
            [758.7789, 0.0, 2.9225],
            [758.7789, 2.9225, 0.0],
            [906.0801, 0.0, 1.9545],
            [906.0801, 1.9545, 0.0],
            [1475.25, 0.0, 0.0],
        ]
    ),
    42: np.array(
        [
            [337.1075, 11.0556, 11.0556],
            [407.0659, 1.942, 11.0556],
            [407.0659, 11.0556, 1.942],
            [502.0073, 1.942, 1.942],
            [509.7424, 0.0, 11.0556],
            [509.7424, 11.0556, 0.0],
            [643.4721, 0.0, 1.942],
            [643.4721, 1.942, 0.0],
            [845.8972, 0.0, 0.0],
        ]
    ),
    43: np.array(
        [
            [265.8654, 9.5402, 9.5402],
            [397.0367, 1.6682, 9.5402],
            [397.0367, 9.5402, 1.6682],
            [493.743, 0.0, 9.5402],
            [493.743, 9.5402, 0.0],
            [649.7143, 1.6682, 1.6682],
            [829.2408, 0.0, 1.6682],
            [829.2408, 1.6682, 0.0],
            [1067.0169, 0.0, 0.0],
        ]
    ),
    44: np.array(
        [
            [239.0071, 8.8895, 8.8895],
            [297.6676, 1.8584, 8.8895],
            [297.6676, 8.8895, 1.8584],
            [358.484, 0.0, 8.8895],
            [358.484, 8.8895, 0.0],
            [379.2655, 1.8584, 1.8584],
            [468.5695, 0.0, 1.8584],
            [468.5695, 1.8584, 0.0],
            [598.1988, 0.0, 0.0],
        ]
    ),
    45: np.array(
        [
            [244.6586, 2.9696, 2.9696],
            [284.1821, 1.9003, 2.9696],
            [284.1821, 2.9696, 1.9003],
            [332.2425, 1.9003, 1.9003],
            [402.9498, 0.0, 2.9696],
            [402.9498, 2.9696, 0.0],
            [478.2848, 0.0, 1.9003],
            [478.2848, 1.9003, 0.0],
            [713.9427, 0.0, 0.0],
        ]
    ),
    46: np.array(
        [
            [265.8746, 5.7095, 5.7095],
            [275.9575, 1.863, 5.7095],
            [275.9575, 5.7095, 1.863],
            [286.9519, 1.863, 1.863],
            [392.3548, 0.0, 5.7095],
            [392.3548, 5.7095, 0.0],
            [411.1361, 0.0, 1.863],
            [411.1361, 1.863, 0.0],
            [608.5041, 0.0, 0.0],
        ]
    ),
    47: np.array(
        [
            [268.613, 0.9679, 0.9679],
            [333.1583, 0.0, 0.9679],
            [333.1583, 0.9679, 0.0],
            [426.745, 0.0, 0.0],
        ]
    ),
    48: np.array(
        [
            [263.4508, 1.9539, 1.9539],
            [345.0599, 0.0, 1.9539],
            [345.0599, 1.9539, 0.0],
            [468.19, 0.0, 0.0],
        ]
    ),
    49: np.array(
        [
            [310.8315, 2.9353, 2.9353],
            [362.7514, 1.9378, 2.9353],
            [362.7514, 2.9353, 1.9378],
            [427.914, 1.9378, 1.9378],
            [447.3269, 0.9633, 2.9353],
            [447.3269, 2.9353, 0.9633],
            [468.2521, 0.0, 2.9353],
            [468.2521, 2.9353, 0.0],
            [534.7248, 0.9633, 1.9378],
            [534.7248, 1.9378, 0.9633],
            [561.9006, 0.0, 1.9378],
            [561.9006, 1.9378, 0.0],
            [678.992, 0.9633, 0.9633],
            [716.7957, 0.0, 0.9633],
            [716.7957, 0.9633, 0.0],
            [757.7397, 0.0, 0.0],
        ]
    ),
    50: np.array(
        [
            [336.102, 3.9123, 3.9123],
            [370.1674, 2.9259, 3.9123],
            [370.1674, 3.9123, 2.9259],
            [409.1453, 2.9259, 2.9259],
            [409.2008, 1.9505, 3.9123],
            [409.2008, 3.9123, 1.9505],
            [422.595, 0.9514, 3.9123],
            [422.595, 3.9123, 0.9514],
            [451.7578, 0.0, 3.9123],
            [451.7578, 3.9123, 0.0],
            [453.9195, 1.9505, 2.9259],
            [453.9195, 2.9259, 1.9505],
            [469.3751, 0.9514, 2.9259],
            [469.3751, 2.9259, 0.9514],
            [502.9891, 0.0, 2.9259],
            [502.9891, 2.9259, 0.0],
            [505.4148, 1.9505, 1.9505],
            [523.291, 0.9514, 1.9505],
            [523.291, 1.9505, 0.9514],
            [542.0439, 0.9514, 0.9514],
            [562.1276, 0.0, 1.9505],
            [562.1276, 1.9505, 0.0],
            [582.7693, 0.0, 0.9514],
            [582.7693, 0.9514, 0.0],
            [627.5677, 0.0, 0.0],
        ]
    ),
    51: np.array(
        [
            [417.379, 2.9315, 2.9315],
            [432.3357, 1.9523, 2.9315],
            [432.3357, 2.9315, 1.9523],
            [444.177, 0.9749, 2.9315],
            [444.177, 2.9315, 0.9749],
            [448.0573, 1.9523, 1.9523],
            [452.9187, 0.0, 2.9315],
            [452.9187, 2.9315, 0.0],
            [460.515, 0.9749, 1.9523],
            [460.515, 1.9523, 0.9749],
            [469.7255, 0.0, 1.9523],
            [469.7255, 1.9523, 0.0],
            [473.4703, 0.9749, 0.9749],
            [483.0596, 0.0, 0.9749],
            [483.0596, 0.9749, 0.0],
            [492.9379, 0.0, 0.0],
        ]
    ),
    52: np.array(
        [
            [418.5789, 1.9639, 1.9639],
            [421.8178, 0.9811, 1.9639],
            [421.8178, 1.9639, 0.9811],
            [422.0187, 0.0, 1.9639],
            [422.0187, 1.9639, 0.0],
            [425.1015, 0.9811, 0.9811],
            [425.3152, 0.0, 0.9811],
            [425.3152, 0.9811, 0.0],
            [425.5355, 0.0, 0.0],
        ]
    ),
    53: np.array(
        [
            [351.9667, 0.0, 0.0],
            [355.2175, 0.0, 0.9968],
            [355.2175, 0.9968, 0.0],
            [358.4995, 0.9968, 0.9968],
        ]
    ),
    54: np.array([[290.2223, 0.0, 0.0]]),
    55: np.array(
        [
            [6.4909240e02, 9.9090000e-01, 9.9090000e-01],
            [1.9759738e03, 0.0000000e00, 9.9090000e-01],
            [1.9759738e03, 9.9090000e-01, 0.0000000e00],
            [9.3307294e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    56: np.array(
        [
            [7.1654750e02, 1.8467000e00, 1.8467000e00],
            [1.5113430e03, 9.7970000e-01, 1.8467000e00],
            [1.5113430e03, 1.8467000e00, 9.7970000e-01],
            [1.8160675e03, 0.0000000e00, 1.8467000e00],
            [1.8160675e03, 1.8467000e00, 0.0000000e00],
            [4.1717842e03, 9.7970000e-01, 9.7970000e-01],
            [4.8451776e03, 0.0000000e00, 9.7970000e-01],
            [4.8451776e03, 9.7970000e-01, 0.0000000e00],
            [5.7269887e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    57: np.array(
        [
            [7.6602020e02, 2.9175000e00, 2.9175000e00],
            [1.1269136e03, 1.9373000e00, 2.9175000e00],
            [1.1269136e03, 2.9175000e00, 1.9373000e00],
            [1.6035452e03, 0.0000000e00, 2.9175000e00],
            [1.6035452e03, 2.9175000e00, 0.0000000e00],
            [1.8500003e03, 1.9373000e00, 1.9373000e00],
            [2.6909591e03, 0.0000000e00, 1.9373000e00],
            [2.6909591e03, 1.9373000e00, 0.0000000e00],
            [3.9906172e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    58: np.array([[688.0353, 2.7991, 2.7991]]),
    59: np.array(
        [
            [6.4702500e02, 2.9425000e00, 2.9425000e00],
            [1.4767389e03, 0.0000000e00, 2.9425000e00],
            [1.4767389e03, 2.9425000e00, 0.0000000e00],
            [4.3422386e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    60: np.array(
        [
            [6.2052060e02, 2.9455000e00, 2.9455000e00],
            [1.3875284e03, 0.0000000e00, 2.9455000e00],
            [1.3875284e03, 2.9455000e00, 0.0000000e00],
            [3.9244211e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    61: np.array(
        [
            [5.8174350e02, 2.9413000e00, 2.9413000e00],
            [1.3066123e03, 0.0000000e00, 2.9413000e00],
            [1.3066123e03, 2.9413000e00, 0.0000000e00],
            [3.7109375e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    62: np.array(
        [
            [5.8778910e02, 2.9300000e00, 2.9300000e00],
            [1.2924475e03, 0.0000000e00, 2.9300000e00],
            [1.2924475e03, 2.9300000e00, 0.0000000e00],
            [3.5220508e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    63: np.array(
        [
            [5.4902810e02, 1.8286000e00, 1.8286000e00],
            [1.2495054e03, 0.0000000e00, 1.8286000e00],
            [1.2495054e03, 1.8286000e00, 0.0000000e00],
            [3.3583122e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    64: np.array(
        [
            [595.3628, 2.8732, 2.8732],
            [999.0617, 0.0, 2.8732],
            [999.0617, 2.8732, 0.0],
            [1891.6719, 0.0, 0.0],
        ]
    ),
    65: np.array(
        [
            [524.9195, 2.9086, 2.9086],
            [1053.6587, 0.0, 2.9086],
            [1053.6587, 2.9086, 0.0],
            [2851.6677, 0.0, 0.0],
        ]
    ),
    66: np.array(
        [
            [544.4356, 2.8965, 2.8965],
            [1051.8609, 0.0, 2.8965],
            [1051.8609, 2.8965, 0.0],
            [2617.331, 0.0, 0.0],
        ]
    ),
    67: np.array(
        [
            [472.795, 2.9242, 2.9242],
            [1011.4677, 0.0, 2.9242],
            [1011.4677, 2.9242, 0.0],
            [2664.1668, 0.0, 0.0],
        ]
    ),
    68: np.array(
        [
            [446.2314, 2.9282, 2.9282],
            [959.5536, 0.0, 2.9282],
            [959.5536, 2.9282, 0.0],
            [2545.1713, 0.0, 0.0],
        ]
    ),
    69: np.array(
        [
            [440.5228, 2.9246, 2.9246],
            [936.4457, 0.0, 2.9246],
            [936.4457, 2.9246, 0.0],
            [2437.4539, 0.0, 0.0],
        ]
    ),
    70: np.array(
        [
            [487.0753, 2.8482, 2.8482],
            [982.2266, 0.0, 2.8482],
            [982.2266, 2.8482, 0.0],
            [2390.1227, 0.0, 0.0],
        ]
    ),
    71: np.array(
        [
            [411.5665, 2.9219, 2.9219],
            [751.4308, 0.0, 2.9219],
            [751.4308, 2.9219, 0.0],
            [1597.4796, 0.0, 0.0],
        ]
    ),
    72: np.array(
        [
            [420.9766, 3.884, 3.884],
            [559.5455, 1.9254, 3.884],
            [559.5455, 3.884, 1.9254],
            [740.2521, 0.0, 3.884],
            [740.2521, 3.884, 0.0],
            [780.3872, 1.9254, 1.9254],
            [1054.3108, 0.0, 1.9254],
            [1054.3108, 1.9254, 0.0],
            [1441.2394, 0.0, 0.0],
        ]
    ),
    73: np.array(
        [
            [458.7677, 2.8988, 2.8988],
            [546.1603, 1.9459, 2.8988],
            [546.1603, 2.8988, 1.9459],
            [659.7501, 1.9459, 1.9459],
            [712.5324, 0.0, 2.8988],
            [712.5324, 2.8988, 0.0],
            [871.1517, 0.0, 1.9459],
            [871.1517, 1.9459, 0.0],
            [1163.8241, 0.0, 0.0],
        ]
    ),
    74: np.array(
        [
            [398.0973, 10.9153, 10.9153],
            [461.6567, 1.9292, 10.9153],
            [461.6567, 10.9153, 1.9292],
            [543.4036, 1.9292, 1.9292],
            [553.3851, 0.0, 10.9153],
            [553.3851, 10.9153, 0.0],
            [660.7636, 0.0, 1.9292],
            [660.7636, 1.9292, 0.0],
            [814.3622, 0.0, 0.0],
        ]
    ),
    75: np.array(
        [
            [332.1448, 9.8054, 9.8054],
            [420.7023, 1.8104, 9.8054],
            [420.7023, 9.8054, 1.8104],
            [508.5233, 0.0, 9.8054],
            [508.5233, 9.8054, 0.0],
            [561.4338, 1.8104, 1.8104],
            [684.5655, 0.0, 1.8104],
            [684.5655, 1.8104, 0.0],
            [836.331, 0.0, 0.0],
        ]
    ),
    76: np.array(
        [
            [297.8338, 9.1527, 9.1527],
            [353.5038, 1.8858, 9.1527],
            [353.5038, 9.1527, 1.8858],
            [426.1903, 1.8858, 1.8858],
            [435.2044, 0.0, 9.1527],
            [435.2044, 9.1527, 0.0],
            [533.2555, 0.0, 1.8858],
            [533.2555, 1.8858, 0.0],
            [678.5278, 0.0, 0.0],
        ]
    ),
    77: np.array(
        [
            [305.5258, 2.9424, 2.9424],
            [339.573, 1.8648, 2.9424],
            [339.573, 2.9424, 1.8648],
            [378.4939, 1.8648, 1.8648],
            [410.589, 0.0, 2.9424],
            [410.589, 2.9424, 0.0],
            [460.1533, 0.0, 1.8648],
            [460.1533, 1.8648, 0.0],
            [566.066, 0.0, 0.0],
        ]
    ),
    78: np.array(
        [
            [336.8727, 6.6669, 6.6669],
            [341.4644, 1.9188, 6.6669],
            [341.4644, 6.6669, 1.9188],
            [346.361, 1.9188, 1.9188],
            [361.3108, 0.0, 6.6669],
            [361.3108, 6.6669, 0.0],
            [367.1687, 0.0, 1.9188],
            [367.1687, 1.9188, 0.0],
            [391.1448, 0.0, 0.0],
        ]
    ),
    79: np.array(
        [
            [317.1561, 0.9846, 0.9846],
            [329.1138, 0.0, 0.9846],
            [329.1138, 0.9846, 0.0],
            [342.3526, 0.0, 0.0],
        ]
    ),
    80: np.array(
        [
            [298.8616, 1.9896, 1.9896],
            [327.8549, 0.0, 1.9896],
            [327.8549, 1.9896, 0.0],
            [362.0755, 0.0, 0.0],
        ]
    ),
    81: np.array(
        [
            [389.6069, 2.942, 2.942],
            [434.5543, 1.9302, 2.942],
            [434.5543, 2.942, 1.9302],
            [487.7418, 1.9302, 1.9302],
            [536.104, 0.9267, 2.942],
            [536.104, 2.942, 0.9267],
            [536.7156, 0.0, 2.942],
            [536.7156, 2.942, 0.0],
            [609.0789, 0.9267, 1.9302],
            [609.0789, 1.9302, 0.9267],
            [611.1733, 0.0, 1.9302],
            [611.1733, 1.9302, 0.0],
            [778.6188, 0.9267, 0.9267],
            [784.9496, 0.0, 0.9267],
            [784.9496, 0.9267, 0.0],
            [792.2378, 0.0, 0.0],
        ]
    ),
    82: np.array(
        [
            [455.2854, 3.9098, 3.9098],
            [487.974, 2.9081, 3.9098],
            [487.974, 3.9098, 2.9081],
            [496.662, 1.9356, 3.9098],
            [496.662, 3.9098, 1.9356],
            [524.184, 2.9081, 2.9081],
            [532.8947, 0.9383, 3.9098],
            [532.8947, 3.9098, 0.9383],
            [534.0827, 1.9356, 2.9081],
            [534.0827, 2.9081, 1.9356],
            [544.4563, 1.9356, 1.9356],
            [571.0696, 0.0, 3.9098],
            [571.0696, 3.9098, 0.0],
            [574.442, 0.9383, 2.9081],
            [574.442, 2.9081, 0.9383],
            [586.2856, 0.9383, 1.9356],
            [586.2856, 1.9356, 0.9383],
            [617.181, 0.0, 2.9081],
            [617.181, 2.9081, 0.0],
            [630.6968, 0.0, 1.9356],
            [630.6968, 1.9356, 0.0],
            [632.9991, 0.9383, 0.9383],
            [682.8545, 0.0, 0.9383],
            [682.8545, 0.9383, 0.0],
            [738.8156, 0.0, 0.0],
        ]
    ),
    83: np.array(
        [
            [533.881, 2.95, 2.95],
            [549.5487, 1.9655, 2.95],
            [549.5487, 2.95, 1.9655],
            [562.5087, 0.982, 2.95],
            [562.5087, 2.95, 0.982],
            [566.0497, 1.9655, 1.9655],
            [572.8443, 0.0, 2.95],
            [572.8443, 2.95, 0.0],
            [579.7329, 0.982, 1.9655],
            [579.7329, 1.9655, 0.982],
            [590.6919, 0.0, 1.9655],
            [590.6919, 1.9655, 0.0],
            [594.0457, 0.982, 0.982],
            [605.5499, 0.0, 0.982],
            [605.5499, 0.982, 0.0],
            [617.5296, 0.0, 0.0],
        ]
    ),
    84: np.array(
        [
            [558.9712, 1.9639, 1.9639],
            [560.6983, 0.0, 1.9639],
            [560.6983, 1.9639, 0.0],
            [561.0805, 0.9815, 1.9639],
            [561.0805, 1.9639, 0.9815],
            [562.6011, 0.0, 0.0],
            [562.9049, 0.0, 0.9815],
            [562.9049, 0.9815, 0.0],
            [563.2463, 0.9815, 0.9815],
        ]
    ),
    85: np.array(
        [
            [483.6536, 0.0, 0.0],
            [489.5133, 0.0, 0.9954],
            [489.5133, 0.9954, 0.0],
            [495.4474, 0.9954, 0.9954],
        ]
    ),
    86: np.array([[412.8275, 0.0, 0.0]]),
    87: np.array(
        [
            [8.2204820e02, 9.7050000e-01, 9.7050000e-01],
            [2.0671766e03, 0.0000000e00, 9.7050000e-01],
            [2.0671766e03, 9.7050000e-01, 0.0000000e00],
            [7.3147398e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    88: np.array(
        [
            [8.6755840e02, 1.8075000e00, 1.8075000e00],
            [1.6402094e03, 9.6620000e-01, 1.8075000e00],
            [1.6402094e03, 1.8075000e00, 9.6620000e-01],
            [1.9615822e03, 0.0000000e00, 1.8075000e00],
            [1.9615822e03, 1.8075000e00, 0.0000000e00],
            [3.7875943e03, 9.6620000e-01, 9.6620000e-01],
            [4.4613947e03, 0.0000000e00, 9.6620000e-01],
            [4.4613947e03, 9.6620000e-01, 0.0000000e00],
            [5.3054399e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    89: np.array(
        [
            [9.0604480e02, 2.9070000e00, 2.9070000e00],
            [1.7226039e03, 0.0000000e00, 2.9070000e00],
            [1.7226039e03, 2.9070000e00, 0.0000000e00],
            [3.7996565e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    90: np.array(
        [
            [1224.5933, 2.8844, 2.8844],
            [1835.9296, 0.0, 2.8844],
            [1835.9296, 2.8844, 0.0],
            [2847.2704, 0.0, 0.0],
        ]
    ),
    91: np.array(
        [
            [9.8037580e02, 2.8738000e00, 2.8738000e00],
            [1.6281996e03, 0.0000000e00, 2.8738000e00],
            [1.6281996e03, 2.8738000e00, 0.0000000e00],
            [2.9089206e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    92: np.array(
        [
            [753.8617, 2.8878, 2.8878],
            [1352.0713, 0.0, 2.8878],
            [1352.0713, 2.8878, 0.0],
            [2721.5209, 0.0, 0.0],
        ]
    ),
    93: np.array(
        [
            [6.9816440e02, 2.9095000e00, 2.9095000e00],
            [1.3314737e03, 0.0000000e00, 2.9095000e00],
            [1.3314737e03, 2.9095000e00, 0.0000000e00],
            [3.0329760e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
    94: np.array(
        [
            [1.0512468e03, 1.9209000e00, 1.9209000e00],
            [1.6834151e03, 0.0000000e00, 1.9209000e00],
            [1.6834151e03, 1.9209000e00, 0.0000000e00],
            [2.8152366e03, 0.0000000e00, 0.0000000e00],
        ]
    ),
}
"""Reference data for C6 coefficients for Grimme's D3."""
