"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcEndpointServiceDomainName = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const lib_1 = require("../lib");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * A Private DNS configuration for a VPC endpoint service.
 *
 * @stability stable
 */
class VpcEndpointServiceDomainName extends core_2.Construct {
    // The way this class works is by using three custom resources and a TxtRecord in conjunction
    // The first custom resource tells the VPC endpoint service to use the given DNS name
    // The VPC endpoint service will then say:
    // "ok, create a TXT record using these two values to prove you own the domain"
    // The second custom resource retrieves these two values from the service
    // The TxtRecord is created from these two values
    // The third custom resource tells the VPC Endpoint Service to verify the domain ownership
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const serviceUniqueId = core_1.Names.nodeUniqueId(props.endpointService.node);
        const serviceId = props.endpointService.vpcEndpointServiceId;
        const privateDnsName = props.domainName;
        // Make sure a user doesn't accidentally add multiple domains
        this.validateProps(props);
        VpcEndpointServiceDomainName.endpointServicesMap[serviceUniqueId] = privateDnsName;
        VpcEndpointServiceDomainName.endpointServices.push(props.endpointService);
        // Enable Private DNS on the endpoint service and retrieve the AWS-generated configuration
        const privateDnsConfiguration = this.getPrivateDnsConfiguration(serviceUniqueId, serviceId, privateDnsName);
        // Tell AWS to verify that this account owns the domain attached to the service
        this.verifyPrivateDnsConfiguration(privateDnsConfiguration, props.publicHostedZone);
        // Finally, don't do any of the above before the endpoint service is created
        this.node.addDependency(props.endpointService);
    }
    validateProps(props) {
        const serviceUniqueId = core_1.Names.nodeUniqueId(props.endpointService.node);
        if (serviceUniqueId in VpcEndpointServiceDomainName.endpointServicesMap) {
            const endpoint = VpcEndpointServiceDomainName.endpointServicesMap[serviceUniqueId];
            throw new Error(`Cannot create a VpcEndpointServiceDomainName for service ${serviceUniqueId}, another VpcEndpointServiceDomainName (${endpoint}) is already associated with it`);
        }
    }
    /**
     * Sets up Custom Resources to make AWS calls to set up Private DNS on an endpoint service,
     * returning the values to use in a TxtRecord, which AWS uses to verify domain ownership.
     */
    getPrivateDnsConfiguration(serviceUniqueId, serviceId, privateDnsName) {
        // The custom resource which tells AWS to enable Private DNS on the given service, using the given domain name
        // AWS will generate a name/value pair for use in a TxtRecord, which is used to verify domain ownership.
        const enablePrivateDnsAction = {
            service: 'EC2',
            action: 'modifyVpcEndpointServiceConfiguration',
            parameters: {
                ServiceId: serviceId,
                PrivateDnsName: privateDnsName,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(serviceUniqueId),
        };
        const removePrivateDnsAction = {
            service: 'EC2',
            action: 'modifyVpcEndpointServiceConfiguration',
            parameters: {
                ServiceId: serviceId,
                RemovePrivateDnsName: true,
            },
        };
        const enable = new custom_resources_1.AwsCustomResource(this, 'EnableDns', {
            onCreate: enablePrivateDnsAction,
            onUpdate: enablePrivateDnsAction,
            onDelete: removePrivateDnsAction,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [
                    core_1.Fn.join(':', [
                        'arn',
                        core_1.Stack.of(this).partition,
                        'ec2',
                        core_1.Stack.of(this).region,
                        core_1.Stack.of(this).account,
                        core_1.Fn.join('/', [
                            'vpc-endpoint-service',
                            serviceId,
                        ]),
                    ]),
                ],
            }),
        });
        // Look up the name/value pair if the domain changes, or the service changes,
        // which would cause the values to be different. If the unique ID changes,
        // the resource may be entirely recreated, so we will need to look it up again.
        const lookup = hashcode(core_1.Names.uniqueId(this) + serviceUniqueId + privateDnsName);
        // Create the custom resource to look up the name/value pair generated by AWS
        // after the previous API call
        const retrieveNameValuePairAction = {
            service: 'EC2',
            action: 'describeVpcEndpointServiceConfigurations',
            parameters: {
                ServiceIds: [serviceId],
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(lookup),
        };
        const getNames = new custom_resources_1.AwsCustomResource(this, 'GetNames', {
            onCreate: retrieveNameValuePairAction,
            onUpdate: retrieveNameValuePairAction,
            // describeVpcEndpointServiceConfigurations can't take an ARN for granular permissions
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        // We only want to call and get the name/value pair after we've told AWS to enable Private DNS
        // If we call before then, we'll get an empty pair of values.
        getNames.node.addDependency(enable);
        // Get the references to the name/value pair associated with the endpoint service
        const name = getNames.getResponseField('ServiceConfigurations.0.PrivateDnsNameConfiguration.Name');
        const value = getNames.getResponseField('ServiceConfigurations.0.PrivateDnsNameConfiguration.Value');
        return { name, value, serviceId };
    }
    /**
     * Creates a Route53 entry and a Custom Resource which explicitly tells AWS to verify ownership
     * of the domain name attached to an endpoint service.
     */
    verifyPrivateDnsConfiguration(config, publicHostedZone) {
        // Create the TXT record in the provided hosted zone
        const verificationRecord = new lib_1.TxtRecord(this, 'DnsVerificationRecord', {
            recordName: config.name,
            values: [config.value],
            zone: publicHostedZone,
        });
        // Tell the endpoint service to verify the domain ownership
        const startVerificationAction = {
            service: 'EC2',
            action: 'startVpcEndpointServicePrivateDnsVerification',
            parameters: {
                ServiceId: config.serviceId,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(core_1.Fn.join(':', [config.name, config.value])),
        };
        const startVerification = new custom_resources_1.AwsCustomResource(this, 'StartVerification', {
            onCreate: startVerificationAction,
            onUpdate: startVerificationAction,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [
                    core_1.Fn.join(':', [
                        'arn',
                        core_1.Stack.of(this).partition,
                        'ec2',
                        core_1.Stack.of(this).region,
                        core_1.Stack.of(this).account,
                        core_1.Fn.join('/', [
                            'vpc-endpoint-service',
                            config.serviceId,
                        ]),
                    ]),
                ],
            }),
        });
        // Only verify after the record has been created
        startVerification.node.addDependency(verificationRecord);
    }
}
exports.VpcEndpointServiceDomainName = VpcEndpointServiceDomainName;
_a = JSII_RTTI_SYMBOL_1;
VpcEndpointServiceDomainName[_a] = { fqn: "@aws-cdk/aws-route53.VpcEndpointServiceDomainName", version: "1.108.1" };
// Track all domain names created, so someone doesn't accidentally associate two domains with a single service
VpcEndpointServiceDomainName.endpointServices = [];
// Track all domain names created, so someone doesn't accidentally associate two domains with a single service
VpcEndpointServiceDomainName.endpointServicesMap = {};
/**
 * Hash a string
 */
function hashcode(s) {
    const hash = crypto.createHash('md5');
    hash.update(s);
    return hash.digest('hex');
}
;
//# sourceMappingURL=data:application/json;base64,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