from typing import List, Union

import pydantic

from artefacts.mixins import Artifact
from artefacts.mixins import ContextReader
from artefacts.shared import Metadata


# https://schemas.getdbt.com/dbt/run-results/v3/index.html#results
class Result(ContextReader, pydantic.BaseModel):
    status: str
    timing: List[dict]
    thread_id: str
    execution_time: float
    adapter_response: dict
    message: Union[None, str]
    failures: Union[None, int]
    unique_id: str

    def __str__(self):
        return f"<RunResult {self.unique_id}>"

    def __repr__(self):
        return str(self)


# https://schemas.getdbt.com/dbt/run-results/v3/index.html
class RunResults(Artifact, ContextReader, pydantic.BaseModel):
    """
    Run results are artifacts generated by `dbt build`, `dbt run`, `dbt test`, etc.

    Examples:

        >>> from artefacts import Project
        >>> project = Project(target='./target')
        >>> print(project.run_results)
        <RunResults v3>

    Attributes:

        metadata (Metadata):
            The [`Metadata`][artefacts.shared.Metadata] from the build.

        results (List[Result]):
            A list of the results of each node executed during the build.

        elapsed_time (float):
            The elapsed time of the build.

        args (dict):
            Any additional args passed to dbt during the build.

    """

    class Config:
        arbitrary_types_allowed = True

    metadata: Metadata
    results: List[Result]
    elapsed_time: float
    args: dict
