"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ViolationCheck = void 0;
const path = require("path");
class ViolationCheck {
    constructor(ruleCheck, templatePath, rulePath) {
        this.ruleCheck = ruleCheck;
        this.templatePath = templatePath;
        this.rulePath = rulePath;
        this.violatingResources = new Map();
        this.violation = {};
    }
    /**
     * A single guard rule can contain multiple "checks" that are run against a resource
     * or against multiple resources. So for example you might get something like:
     * {
     *   name: 's3-public-buckets-prohibited',
     *   messages: {
     *     custom_message: 'Buckets should not be public',
     *   },
     *   checks: [
     *     {
     *       traversed: {
     *         to: {
     *           path: '/Resources/MyCustomL3ConstructBucket8C61BCA7/Properties/PublicAccessBlockConfiguration/BlockPublicPolicy'
     *         }
     *       }
     *     },
     *     {
     *       traversed: {
     *         to: {
     *           path: '/Resources/MyCustomL3ConstructBucket8C61BCA7/Properties/PublicAccessBlockConfiguration/BlockPublicAcls'
     *         }
     *       }
     *     }
     *   ]
     * }
     *
     * We want to display this to the user as a single violation since there is a single
     * custom_message. This method sets up some inheritance and constructs a single violation per
     * resource+message.
     */
    setViolatingResources(check) {
        // pull the description from the custom message or from the error message if available
        this.violation.description = this.violation.description || check.messages?.custom_message || check.messages?.error_message;
        // The fix will only appear in a custom_message because it would be a user
        // generated message
        this.violation.fix = this.violation.fix ?? check.messages?.custom_message ?? 'N/A';
        const location = check.traversed.to.path;
        const resourceName = location.split('/')[2];
        const violatingResource = this.violatingResources.get(this.violation.fix);
        const result = {
            locations: [location],
        };
        if (violatingResource) {
            const resource = violatingResource.resource.get(resourceName);
            if (resource) {
                resource.locations.push(location);
            }
            else {
                violatingResource.resource.set(resourceName, result);
            }
        }
        else {
            this.violatingResources.set(this.violation.fix, {
                resource: new Map([[resourceName, result]]),
                violation: {
                    description: this.violation.description ?? '',
                    fix: this.violation.fix,
                },
            });
        }
    }
    /**
     * Process a Guard result check and return a plugin violation
     * We are establishing a bit of a convention with the messages where we expect
     * the custom_message field to contain a string formatted like this
     * (based on the Control Tower rules)
     *
     *     [FIX]: Do something...
     *     [XXX]: description of the rule
     *
     * If it does contain a structure like that then we try and parse out the
     * fix and description fields, otherwise we just take the custom_message as
     * is and use it for both.
     */
    processCheck() {
        this.ruleCheck.checks.forEach(check => {
            if (check.messages?.custom_message) {
                const message = check.messages.custom_message.split('\n').filter(m => m.trim() !== '');
                message.forEach(m => {
                    const mes = m.trim();
                    if (mes.startsWith('[FIX]')) {
                        this.violation.fix = mes;
                    }
                    else {
                        this.violation.description = mes;
                    }
                });
            }
            this.setViolatingResources(check);
        });
        return Array.from(this.violatingResources.values()).map(violation => {
            return {
                description: violation.violation.description,
                fix: violation.violation.fix,
                ruleMetadata: {
                    DocumentationUrl: this.generateRuleDocUrl(),
                },
                ruleName: this.ruleCheck.name,
                violatingResources: Array.from(violation.resource.entries()).map(([key, value]) => {
                    return {
                        locations: value.locations,
                        resourceLogicalId: key,
                        templatePath: this.templatePath,
                    };
                }),
            };
        });
    }
    generateRuleDocUrl() {
        const serviceName = path.basename(path.dirname(this.rulePath));
        const ruleName = path.basename(this.rulePath, '.guard');
        const root = 'https://docs.aws.amazon.com/controltower/latest/userguide';
        return `${root}/${serviceName}-rules.html#${ruleName}-description`;
    }
}
exports.ViolationCheck = ViolationCheck;
//# sourceMappingURL=data:application/json;base64,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