"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
class TestApp {
    constructor() {
        this.app = new cdk.App();
        const account = '123456789012';
        const region = 'bermuda-triangle';
        this.app.node.setContext(`availability-zones:${account}:${region}`, [`${region}-1a`]);
        this.stack = new cdk.Stack(this.app, 'MyStack', { env: { account, region } });
    }
    synthesizeTemplate() {
        return this.app.synthesizeStack(this.stack.name);
    }
}
module.exports = {
    'default properties': {
        'public hosted zone'(test) {
            const app = new TestApp();
            new lib_1.PublicHostedZone(app.stack, 'HostedZone', { zoneName: 'test.public' });
            assert_1.expect(app.synthesizeTemplate()).to(assert_1.exactlyMatchTemplate({
                Resources: {
                    HostedZoneDB99F866: {
                        Type: "AWS::Route53::HostedZone",
                        Properties: {
                            Name: "test.public."
                        }
                    }
                }
            }));
            test.done();
        },
        'private hosted zone'(test) {
            const app = new TestApp();
            const vpcNetwork = new ec2.VpcNetwork(app.stack, 'VPC');
            new lib_1.PrivateHostedZone(app.stack, 'HostedZone', { zoneName: 'test.private', vpc: vpcNetwork });
            assert_1.expect(app.synthesizeTemplate()).to(assert_1.beASupersetOfTemplate({
                Resources: {
                    HostedZoneDB99F866: {
                        Type: "AWS::Route53::HostedZone",
                        Properties: {
                            Name: "test.private.",
                            VPCs: [{
                                    VPCId: { Ref: 'VPCB9E5F0B4' },
                                    VPCRegion: 'bermuda-triangle'
                                }]
                        }
                    }
                }
            }));
            test.done();
        },
        'when specifying multiple VPCs'(test) {
            const app = new TestApp();
            const vpcNetworkA = new ec2.VpcNetwork(app.stack, 'VPC1');
            const vpcNetworkB = new ec2.VpcNetwork(app.stack, 'VPC2');
            new lib_1.PrivateHostedZone(app.stack, 'HostedZone', { zoneName: 'test.private', vpc: vpcNetworkA })
                .addVpc(vpcNetworkB);
            assert_1.expect(app.synthesizeTemplate()).to(assert_1.beASupersetOfTemplate({
                Resources: {
                    HostedZoneDB99F866: {
                        Type: "AWS::Route53::HostedZone",
                        Properties: {
                            Name: "test.private.",
                            VPCs: [{
                                    VPCId: { Ref: 'VPC17DE2CF87' },
                                    VPCRegion: 'bermuda-triangle'
                                },
                                {
                                    VPCId: { Ref: 'VPC2C1F0E711' },
                                    VPCRegion: 'bermuda-triangle'
                                }]
                        }
                    }
                }
            }));
            test.done();
        }
    },
    'exporting and importing works'(test) {
        const stack1 = new cdk.Stack();
        const stack2 = new cdk.Stack();
        const zone = new lib_1.PublicHostedZone(stack1, 'Zone', {
            zoneName: 'cdk.local',
        });
        const zoneRef = zone.export();
        const importedZone = lib_1.HostedZone.import(stack2, 'Imported', zoneRef);
        new lib_1.TxtRecord(importedZone, 'Record', {
            zone: importedZone,
            recordName: 'lookHere',
            recordValue: 'SeeThere'
        });
        assert_1.expect(stack1).to(assert_1.exactlyMatchTemplate({
            Resources: {
                ZoneA5DE4B68: {
                    Type: "AWS::Route53::HostedZone",
                    Properties: {
                        Name: "cdk.local."
                    }
                }
            },
            Outputs: {
                ZoneHostedZoneId413B8768: {
                    Value: { Ref: "ZoneA5DE4B68" },
                    Export: { Name: "Stack:ZoneHostedZoneId413B8768" }
                }
            }
        }));
        assert_1.expect(stack2).to(assert_1.haveResource("AWS::Route53::RecordSet", {
            HostedZoneId: { "Fn::ImportValue": "Stack:ZoneHostedZoneId413B8768" },
            Name: "lookHere.cdk.local.",
            ResourceRecords: ["\"SeeThere\""],
            Type: "TXT"
        }));
        test.done();
    },
    'adds period to name if not provided'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.HostedZone(stack, 'MyHostedZone', {
            zoneName: 'zonename'
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::HostedZone', {
            Name: 'zonename.'
        }));
        test.done();
    },
    'fails if zone name ends with a trailing dot'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new lib_1.HostedZone(stack, 'MyHostedZone', { zoneName: 'zonename.' }), /zone name must not end with a trailing dot/);
        test.done();
    },
    'a hosted zone can be assiciated with a VPC either upon creation or using "addVpc"'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc1 = new ec2.VpcNetwork(stack, 'VPC1');
        const vpc2 = new ec2.VpcNetwork(stack, 'VPC2');
        const vpc3 = new ec2.VpcNetwork(stack, 'VPC3');
        // WHEN
        const zone = new lib_1.HostedZone(stack, 'MyHostedZone', {
            zoneName: 'zonename',
            vpcs: [vpc1, vpc2]
        });
        zone.addVpc(vpc3);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::HostedZone', {
            VPCs: [
                {
                    VPCId: {
                        Ref: "VPC17DE2CF87"
                    },
                    VPCRegion: {
                        Ref: "AWS::Region"
                    }
                },
                {
                    VPCId: {
                        Ref: "VPC2C1F0E711"
                    },
                    VPCRegion: {
                        Ref: "AWS::Region"
                    }
                },
                {
                    VPCId: {
                        Ref: "VPC3CB5FCDA8"
                    },
                    VPCRegion: {
                        Ref: "AWS::Region"
                    }
                }
            ]
        }));
        test.done();
    },
    'public zone cannot be associated with a vpc (runtime error)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const zone = new lib_1.PublicHostedZone(stack, 'MyHostedZone', { zoneName: 'zonename' });
        const vpc = new ec2.VpcNetwork(stack, 'VPC');
        // THEN
        test.throws(() => zone.addVpc(vpc), /Cannot associate public hosted zones with a VPC/);
        test.done();
    },
    'setting up zone delegation'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const zone = new lib_1.PublicHostedZone(stack, 'TopZone', { zoneName: 'top.test' });
        const delegate = new lib_1.PublicHostedZone(stack, 'SubZone', { zoneName: 'sub.top.test' });
        // WHEN
        zone.addDelegation(delegate, { ttl: 1337 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Type: 'NS',
            Name: 'sub.top.test.',
            HostedZoneId: zone.node.resolve(zone.hostedZoneId),
            ResourceRecords: zone.node.resolve(delegate.hostedZoneNameServers),
            TTL: '1337',
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5yb3V0ZTUzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC5yb3V0ZTUzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSw0Q0FBb0c7QUFDcEcsd0NBQXlDO0FBQ3pDLG9DQUFxQztBQUVyQyxnQ0FBb0Y7QUFxTnBGLE1BQU0sT0FBTztJQUlYO1FBRmlCLFFBQUcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUduQyxNQUFNLE9BQU8sR0FBRyxjQUFjLENBQUM7UUFDL0IsTUFBTSxNQUFNLEdBQUcsa0JBQWtCLENBQUM7UUFDbEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLHNCQUFzQixPQUFPLElBQUksTUFBTSxFQUFFLEVBQ2hFLENBQUMsR0FBRyxNQUFNLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDcEIsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ2hGLENBQUM7SUFFTSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsR0FBRyxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ25ELENBQUM7Q0FDRjtBQWxPRCxpQkFBUztJQUNQLG9CQUFvQixFQUFFO1FBQ3BCLG9CQUFvQixDQUFDLElBQVU7WUFDN0IsTUFBTSxHQUFHLEdBQUcsSUFBSSxPQUFPLEVBQUUsQ0FBQztZQUMxQixJQUFJLHNCQUFnQixDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLEVBQUUsUUFBUSxFQUFFLGFBQWEsRUFBRSxDQUFDLENBQUM7WUFDM0UsZUFBTSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLDZCQUFvQixDQUFDO2dCQUN2RCxTQUFTLEVBQUU7b0JBQ1Qsa0JBQWtCLEVBQUU7d0JBQ2xCLElBQUksRUFBRSwwQkFBMEI7d0JBQ2hDLFVBQVUsRUFBRTs0QkFDVixJQUFJLEVBQUUsY0FBYzt5QkFDckI7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCxxQkFBcUIsQ0FBQyxJQUFVO1lBQzlCLE1BQU0sR0FBRyxHQUFHLElBQUksT0FBTyxFQUFFLENBQUM7WUFDMUIsTUFBTSxVQUFVLEdBQUcsSUFBSSxHQUFHLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDeEQsSUFBSSx1QkFBaUIsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRSxFQUFFLFFBQVEsRUFBRSxjQUFjLEVBQUUsR0FBRyxFQUFFLFVBQVUsRUFBRSxDQUFDLENBQUM7WUFDOUYsZUFBTSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLDhCQUFxQixDQUFDO2dCQUN4RCxTQUFTLEVBQUU7b0JBQ1Qsa0JBQWtCLEVBQUU7d0JBQ2xCLElBQUksRUFBRSwwQkFBMEI7d0JBQ2hDLFVBQVUsRUFBRTs0QkFDVixJQUFJLEVBQUUsZUFBZTs0QkFDckIsSUFBSSxFQUFFLENBQUM7b0NBQ0wsS0FBSyxFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsRUFBRTtvQ0FDN0IsU0FBUyxFQUFFLGtCQUFrQjtpQ0FDOUIsQ0FBQzt5QkFDSDtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBQ0osSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUNELCtCQUErQixDQUFDLElBQVU7WUFDeEMsTUFBTSxHQUFHLEdBQUcsSUFBSSxPQUFPLEVBQUUsQ0FBQztZQUMxQixNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztZQUMxRCxNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztZQUMxRCxJQUFJLHVCQUFpQixDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLEVBQUUsUUFBUSxFQUFFLGNBQWMsRUFBRSxHQUFHLEVBQUUsV0FBVyxFQUFFLENBQUM7aUJBQzNGLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUN2QixlQUFNLENBQUMsR0FBRyxDQUFDLGtCQUFrQixFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsOEJBQXFCLENBQUM7Z0JBQ3hELFNBQVMsRUFBRTtvQkFDVCxrQkFBa0IsRUFBRTt3QkFDbEIsSUFBSSxFQUFFLDBCQUEwQjt3QkFDaEMsVUFBVSxFQUFFOzRCQUNWLElBQUksRUFBRSxlQUFlOzRCQUNyQixJQUFJLEVBQUUsQ0FBQztvQ0FDTCxLQUFLLEVBQUUsRUFBRSxHQUFHLEVBQUUsY0FBYyxFQUFFO29DQUM5QixTQUFTLEVBQUUsa0JBQWtCO2lDQUM5QjtnQ0FDRDtvQ0FDRSxLQUFLLEVBQUUsRUFBRSxHQUFHLEVBQUUsY0FBYyxFQUFFO29DQUM5QixTQUFTLEVBQUUsa0JBQWtCO2lDQUM5QixDQUFDO3lCQUNIO3FCQUNGO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7SUFFRCwrQkFBK0IsQ0FBQyxJQUFVO1FBQ3hDLE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQy9CLE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRS9CLE1BQU0sSUFBSSxHQUFHLElBQUksc0JBQWdCLENBQUMsTUFBTSxFQUFFLE1BQU0sRUFBRTtZQUNoRCxRQUFRLEVBQUUsV0FBVztTQUN0QixDQUFDLENBQUM7UUFFSCxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsTUFBTSxFQUFFLENBQUM7UUFDOUIsTUFBTSxZQUFZLEdBQUcsZ0JBQVUsQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLFVBQVUsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUNwRSxJQUFJLGVBQVMsQ0FBQyxZQUFtQixFQUFFLFFBQVEsRUFBRTtZQUMzQyxJQUFJLEVBQUUsWUFBWTtZQUNsQixVQUFVLEVBQUUsVUFBVTtZQUN0QixXQUFXLEVBQUUsVUFBVTtTQUN4QixDQUFDLENBQUM7UUFFSCxlQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxDQUFDLDZCQUFvQixDQUFDO1lBQ3JDLFNBQVMsRUFBRTtnQkFDVCxZQUFZLEVBQUU7b0JBQ1osSUFBSSxFQUFFLDBCQUEwQjtvQkFDaEMsVUFBVSxFQUFFO3dCQUNWLElBQUksRUFBRSxZQUFZO3FCQUNuQjtpQkFDRjthQUNGO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLHdCQUF3QixFQUFFO29CQUN4QixLQUFLLEVBQUUsRUFBRSxHQUFHLEVBQUUsY0FBYyxFQUFFO29CQUM5QixNQUFNLEVBQUUsRUFBRSxJQUFJLEVBQUUsZ0NBQWdDLEVBQUU7aUJBQ25EO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLGVBQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRTtZQUN4RCxZQUFZLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxnQ0FBZ0MsRUFBRTtZQUNyRSxJQUFJLEVBQUUscUJBQXFCO1lBQzNCLGVBQWUsRUFBRSxDQUFFLGNBQWMsQ0FBRTtZQUNuQyxJQUFJLEVBQUUsS0FBSztTQUNaLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHFDQUFxQyxDQUFDLElBQVU7UUFDOUMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE9BQU87UUFDUCxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtZQUNwQyxRQUFRLEVBQUUsVUFBVTtTQUNyQixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLDBCQUEwQixFQUFFO1lBQ3hELElBQUksRUFBRSxXQUFXO1NBQ2xCLENBQUMsQ0FBQyxDQUFDO1FBQ0osSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDZDQUE2QyxDQUFDLElBQVU7UUFDdEQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE9BQU87UUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFLEVBQUUsUUFBUSxFQUFFLFdBQVcsRUFBRSxDQUFDLEVBQUUsNENBQTRDLENBQUMsQ0FBQztRQUNsSSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsbUZBQW1GLENBQUMsSUFBVTtRQUM1RixRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUMvQyxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQy9DLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFL0MsT0FBTztRQUNQLE1BQU0sSUFBSSxHQUFHLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO1lBQ2pELFFBQVEsRUFBRSxVQUFVO1lBQ3BCLElBQUksRUFBRSxDQUFFLElBQUksRUFBRSxJQUFJLENBQUU7U0FDckIsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUVsQixPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLDBCQUEwQixFQUFFO1lBQ3hELElBQUksRUFBRTtnQkFDSjtvQkFDRSxLQUFLLEVBQUU7d0JBQ0wsR0FBRyxFQUFFLGNBQWM7cUJBQ3BCO29CQUNELFNBQVMsRUFBRTt3QkFDVCxHQUFHLEVBQUUsYUFBYTtxQkFDbkI7aUJBQ0Y7Z0JBQ0Q7b0JBQ0UsS0FBSyxFQUFFO3dCQUNMLEdBQUcsRUFBRSxjQUFjO3FCQUNwQjtvQkFDRCxTQUFTLEVBQUU7d0JBQ1QsR0FBRyxFQUFFLGFBQWE7cUJBQ25CO2lCQUNGO2dCQUNEO29CQUNFLEtBQUssRUFBRTt3QkFDTCxHQUFHLEVBQUUsY0FBYztxQkFDcEI7b0JBQ0QsU0FBUyxFQUFFO3dCQUNULEdBQUcsRUFBRSxhQUFhO3FCQUNuQjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNkRBQTZELENBQUMsSUFBVTtRQUN0RSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxzQkFBZ0IsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFLEVBQUUsUUFBUSxFQUFFLFVBQVUsRUFBRSxDQUFDLENBQUM7UUFDbkYsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztRQUU3QyxPQUFPO1FBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxFQUFFLGlEQUFpRCxDQUFDLENBQUM7UUFDdkYsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDRCQUE0QixDQUFDLElBQVU7UUFDckMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sSUFBSSxHQUFHLElBQUksc0JBQWdCLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRSxFQUFFLFFBQVEsRUFBRSxVQUFVLEVBQUUsQ0FBQyxDQUFDO1FBQzlFLE1BQU0sUUFBUSxHQUFHLElBQUksc0JBQWdCLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRSxFQUFFLFFBQVEsRUFBRSxjQUFjLEVBQUUsQ0FBQyxDQUFDO1FBRXRGLE9BQU87UUFDUCxJQUFJLENBQUMsYUFBYSxDQUFDLFFBQVEsRUFBRSxFQUFFLEdBQUcsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBRTVDLE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMseUJBQXlCLEVBQUU7WUFDdkQsSUFBSSxFQUFFLElBQUk7WUFDVixJQUFJLEVBQUUsZUFBZTtZQUNyQixZQUFZLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQztZQUNsRCxlQUFlLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLHFCQUFxQixDQUFDO1lBQ2xFLEdBQUcsRUFBRSxNQUFNO1NBQ1osQ0FBQyxDQUFDLENBQUM7UUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGJlQVN1cGVyc2V0T2ZUZW1wbGF0ZSwgZXhhY3RseU1hdGNoVGVtcGxhdGUsIGV4cGVjdCwgaGF2ZVJlc291cmNlIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCBlYzIgPSByZXF1aXJlKCdAYXdzLWNkay9hd3MtZWMyJyk7XG5pbXBvcnQgY2RrID0gcmVxdWlyZSgnQGF3cy1jZGsvY2RrJyk7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0IHsgSG9zdGVkWm9uZSwgUHJpdmF0ZUhvc3RlZFpvbmUsIFB1YmxpY0hvc3RlZFpvbmUsIFR4dFJlY29yZCB9IGZyb20gJy4uL2xpYic7XG5cbmV4cG9ydCA9IHtcbiAgJ2RlZmF1bHQgcHJvcGVydGllcyc6IHtcbiAgICAncHVibGljIGhvc3RlZCB6b25lJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBhcHAgPSBuZXcgVGVzdEFwcCgpO1xuICAgICAgbmV3IFB1YmxpY0hvc3RlZFpvbmUoYXBwLnN0YWNrLCAnSG9zdGVkWm9uZScsIHsgem9uZU5hbWU6ICd0ZXN0LnB1YmxpYycgfSk7XG4gICAgICBleHBlY3QoYXBwLnN5bnRoZXNpemVUZW1wbGF0ZSgpKS50byhleGFjdGx5TWF0Y2hUZW1wbGF0ZSh7XG4gICAgICAgIFJlc291cmNlczoge1xuICAgICAgICAgIEhvc3RlZFpvbmVEQjk5Rjg2Njoge1xuICAgICAgICAgICAgVHlwZTogXCJBV1M6OlJvdXRlNTM6Okhvc3RlZFpvbmVcIixcbiAgICAgICAgICAgIFByb3BlcnRpZXM6IHtcbiAgICAgICAgICAgICAgTmFtZTogXCJ0ZXN0LnB1YmxpYy5cIlxuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfSkpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICAncHJpdmF0ZSBob3N0ZWQgem9uZScodGVzdDogVGVzdCkge1xuICAgICAgY29uc3QgYXBwID0gbmV3IFRlc3RBcHAoKTtcbiAgICAgIGNvbnN0IHZwY05ldHdvcmsgPSBuZXcgZWMyLlZwY05ldHdvcmsoYXBwLnN0YWNrLCAnVlBDJyk7XG4gICAgICBuZXcgUHJpdmF0ZUhvc3RlZFpvbmUoYXBwLnN0YWNrLCAnSG9zdGVkWm9uZScsIHsgem9uZU5hbWU6ICd0ZXN0LnByaXZhdGUnLCB2cGM6IHZwY05ldHdvcmsgfSk7XG4gICAgICBleHBlY3QoYXBwLnN5bnRoZXNpemVUZW1wbGF0ZSgpKS50byhiZUFTdXBlcnNldE9mVGVtcGxhdGUoe1xuICAgICAgICBSZXNvdXJjZXM6IHtcbiAgICAgICAgICBIb3N0ZWRab25lREI5OUY4NjY6IHtcbiAgICAgICAgICAgIFR5cGU6IFwiQVdTOjpSb3V0ZTUzOjpIb3N0ZWRab25lXCIsXG4gICAgICAgICAgICBQcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgICAgIE5hbWU6IFwidGVzdC5wcml2YXRlLlwiLFxuICAgICAgICAgICAgICBWUENzOiBbe1xuICAgICAgICAgICAgICAgIFZQQ0lkOiB7IFJlZjogJ1ZQQ0I5RTVGMEI0JyB9LFxuICAgICAgICAgICAgICAgIFZQQ1JlZ2lvbjogJ2Jlcm11ZGEtdHJpYW5nbGUnXG4gICAgICAgICAgICAgIH1dXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9KSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgICd3aGVuIHNwZWNpZnlpbmcgbXVsdGlwbGUgVlBDcycodGVzdDogVGVzdCkge1xuICAgICAgY29uc3QgYXBwID0gbmV3IFRlc3RBcHAoKTtcbiAgICAgIGNvbnN0IHZwY05ldHdvcmtBID0gbmV3IGVjMi5WcGNOZXR3b3JrKGFwcC5zdGFjaywgJ1ZQQzEnKTtcbiAgICAgIGNvbnN0IHZwY05ldHdvcmtCID0gbmV3IGVjMi5WcGNOZXR3b3JrKGFwcC5zdGFjaywgJ1ZQQzInKTtcbiAgICAgIG5ldyBQcml2YXRlSG9zdGVkWm9uZShhcHAuc3RhY2ssICdIb3N0ZWRab25lJywgeyB6b25lTmFtZTogJ3Rlc3QucHJpdmF0ZScsIHZwYzogdnBjTmV0d29ya0EgfSlcbiAgICAgICAgLmFkZFZwYyh2cGNOZXR3b3JrQik7XG4gICAgICBleHBlY3QoYXBwLnN5bnRoZXNpemVUZW1wbGF0ZSgpKS50byhiZUFTdXBlcnNldE9mVGVtcGxhdGUoe1xuICAgICAgICBSZXNvdXJjZXM6IHtcbiAgICAgICAgICBIb3N0ZWRab25lREI5OUY4NjY6IHtcbiAgICAgICAgICAgIFR5cGU6IFwiQVdTOjpSb3V0ZTUzOjpIb3N0ZWRab25lXCIsXG4gICAgICAgICAgICBQcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgICAgIE5hbWU6IFwidGVzdC5wcml2YXRlLlwiLFxuICAgICAgICAgICAgICBWUENzOiBbe1xuICAgICAgICAgICAgICAgIFZQQ0lkOiB7IFJlZjogJ1ZQQzE3REUyQ0Y4NycgfSxcbiAgICAgICAgICAgICAgICBWUENSZWdpb246ICdiZXJtdWRhLXRyaWFuZ2xlJ1xuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgVlBDSWQ6IHsgUmVmOiAnVlBDMkMxRjBFNzExJyB9LFxuICAgICAgICAgICAgICAgIFZQQ1JlZ2lvbjogJ2Jlcm11ZGEtdHJpYW5nbGUnXG4gICAgICAgICAgICAgIH1dXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9KSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9XG4gIH0sXG5cbiAgJ2V4cG9ydGluZyBhbmQgaW1wb3J0aW5nIHdvcmtzJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sxID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHN0YWNrMiA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIGNvbnN0IHpvbmUgPSBuZXcgUHVibGljSG9zdGVkWm9uZShzdGFjazEsICdab25lJywge1xuICAgICAgem9uZU5hbWU6ICdjZGsubG9jYWwnLFxuICAgIH0pO1xuXG4gICAgY29uc3Qgem9uZVJlZiA9IHpvbmUuZXhwb3J0KCk7XG4gICAgY29uc3QgaW1wb3J0ZWRab25lID0gSG9zdGVkWm9uZS5pbXBvcnQoc3RhY2syLCAnSW1wb3J0ZWQnLCB6b25lUmVmKTtcbiAgICBuZXcgVHh0UmVjb3JkKGltcG9ydGVkWm9uZSBhcyBhbnksICdSZWNvcmQnLCB7XG4gICAgICB6b25lOiBpbXBvcnRlZFpvbmUsXG4gICAgICByZWNvcmROYW1lOiAnbG9va0hlcmUnLFxuICAgICAgcmVjb3JkVmFsdWU6ICdTZWVUaGVyZSdcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjazEpLnRvKGV4YWN0bHlNYXRjaFRlbXBsYXRlKHtcbiAgICAgIFJlc291cmNlczoge1xuICAgICAgICBab25lQTVERTRCNjg6IHtcbiAgICAgICAgICBUeXBlOiBcIkFXUzo6Um91dGU1Mzo6SG9zdGVkWm9uZVwiLFxuICAgICAgICAgIFByb3BlcnRpZXM6IHtcbiAgICAgICAgICAgIE5hbWU6IFwiY2RrLmxvY2FsLlwiXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgT3V0cHV0czoge1xuICAgICAgICBab25lSG9zdGVkWm9uZUlkNDEzQjg3Njg6IHtcbiAgICAgICAgICBWYWx1ZTogeyBSZWY6IFwiWm9uZUE1REU0QjY4XCIgfSxcbiAgICAgICAgICBFeHBvcnQ6IHsgTmFtZTogXCJTdGFjazpab25lSG9zdGVkWm9uZUlkNDEzQjg3NjhcIiB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9KSk7XG5cbiAgICBleHBlY3Qoc3RhY2syKS50byhoYXZlUmVzb3VyY2UoXCJBV1M6OlJvdXRlNTM6OlJlY29yZFNldFwiLCB7XG4gICAgICBIb3N0ZWRab25lSWQ6IHsgXCJGbjo6SW1wb3J0VmFsdWVcIjogXCJTdGFjazpab25lSG9zdGVkWm9uZUlkNDEzQjg3NjhcIiB9LFxuICAgICAgTmFtZTogXCJsb29rSGVyZS5jZGsubG9jYWwuXCIsXG4gICAgICBSZXNvdXJjZVJlY29yZHM6IFsgXCJcXFwiU2VlVGhlcmVcXFwiXCIgXSxcbiAgICAgIFR5cGU6IFwiVFhUXCJcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYWRkcyBwZXJpb2QgdG8gbmFtZSBpZiBub3QgcHJvdmlkZWQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBIb3N0ZWRab25lKHN0YWNrLCAnTXlIb3N0ZWRab25lJywge1xuICAgICAgem9uZU5hbWU6ICd6b25lbmFtZSdcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpSb3V0ZTUzOjpIb3N0ZWRab25lJywge1xuICAgICAgTmFtZTogJ3pvbmVuYW1lLidcbiAgICB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2ZhaWxzIGlmIHpvbmUgbmFtZSBlbmRzIHdpdGggYSB0cmFpbGluZyBkb3QnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QudGhyb3dzKCgpID0+IG5ldyBIb3N0ZWRab25lKHN0YWNrLCAnTXlIb3N0ZWRab25lJywgeyB6b25lTmFtZTogJ3pvbmVuYW1lLicgfSksIC96b25lIG5hbWUgbXVzdCBub3QgZW5kIHdpdGggYSB0cmFpbGluZyBkb3QvKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYSBob3N0ZWQgem9uZSBjYW4gYmUgYXNzaWNpYXRlZCB3aXRoIGEgVlBDIGVpdGhlciB1cG9uIGNyZWF0aW9uIG9yIHVzaW5nIFwiYWRkVnBjXCInKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHZwYzEgPSBuZXcgZWMyLlZwY05ldHdvcmsoc3RhY2ssICdWUEMxJyk7XG4gICAgY29uc3QgdnBjMiA9IG5ldyBlYzIuVnBjTmV0d29yayhzdGFjaywgJ1ZQQzInKTtcbiAgICBjb25zdCB2cGMzID0gbmV3IGVjMi5WcGNOZXR3b3JrKHN0YWNrLCAnVlBDMycpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHpvbmUgPSBuZXcgSG9zdGVkWm9uZShzdGFjaywgJ015SG9zdGVkWm9uZScsIHtcbiAgICAgIHpvbmVOYW1lOiAnem9uZW5hbWUnLFxuICAgICAgdnBjczogWyB2cGMxLCB2cGMyIF1cbiAgICB9KTtcbiAgICB6b25lLmFkZFZwYyh2cGMzKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpSb3V0ZTUzOjpIb3N0ZWRab25lJywge1xuICAgICAgVlBDczogW1xuICAgICAgICB7XG4gICAgICAgICAgVlBDSWQ6IHtcbiAgICAgICAgICAgIFJlZjogXCJWUEMxN0RFMkNGODdcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgVlBDUmVnaW9uOiB7XG4gICAgICAgICAgICBSZWY6IFwiQVdTOjpSZWdpb25cIlxuICAgICAgICAgIH1cbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIFZQQ0lkOiB7XG4gICAgICAgICAgICBSZWY6IFwiVlBDMkMxRjBFNzExXCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFZQQ1JlZ2lvbjoge1xuICAgICAgICAgICAgUmVmOiBcIkFXUzo6UmVnaW9uXCJcbiAgICAgICAgICB9XG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBWUENJZDoge1xuICAgICAgICAgICAgUmVmOiBcIlZQQzNDQjVGQ0RBOFwiXG4gICAgICAgICAgfSxcbiAgICAgICAgICBWUENSZWdpb246IHtcbiAgICAgICAgICAgIFJlZjogXCJBV1M6OlJlZ2lvblwiXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICBdXG4gICAgfSkpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdwdWJsaWMgem9uZSBjYW5ub3QgYmUgYXNzb2NpYXRlZCB3aXRoIGEgdnBjIChydW50aW1lIGVycm9yKScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3Qgem9uZSA9IG5ldyBQdWJsaWNIb3N0ZWRab25lKHN0YWNrLCAnTXlIb3N0ZWRab25lJywgeyB6b25lTmFtZTogJ3pvbmVuYW1lJyB9KTtcbiAgICBjb25zdCB2cGMgPSBuZXcgZWMyLlZwY05ldHdvcmsoc3RhY2ssICdWUEMnKTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LnRocm93cygoKSA9PiB6b25lLmFkZFZwYyh2cGMpLCAvQ2Fubm90IGFzc29jaWF0ZSBwdWJsaWMgaG9zdGVkIHpvbmVzIHdpdGggYSBWUEMvKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc2V0dGluZyB1cCB6b25lIGRlbGVnYXRpb24nKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHpvbmUgPSBuZXcgUHVibGljSG9zdGVkWm9uZShzdGFjaywgJ1RvcFpvbmUnLCB7IHpvbmVOYW1lOiAndG9wLnRlc3QnIH0pO1xuICAgIGNvbnN0IGRlbGVnYXRlID0gbmV3IFB1YmxpY0hvc3RlZFpvbmUoc3RhY2ssICdTdWJab25lJywgeyB6b25lTmFtZTogJ3N1Yi50b3AudGVzdCcgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgem9uZS5hZGREZWxlZ2F0aW9uKGRlbGVnYXRlLCB7IHR0bDogMTMzNyB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpSb3V0ZTUzOjpSZWNvcmRTZXQnLCB7XG4gICAgICBUeXBlOiAnTlMnLFxuICAgICAgTmFtZTogJ3N1Yi50b3AudGVzdC4nLFxuICAgICAgSG9zdGVkWm9uZUlkOiB6b25lLm5vZGUucmVzb2x2ZSh6b25lLmhvc3RlZFpvbmVJZCksXG4gICAgICBSZXNvdXJjZVJlY29yZHM6IHpvbmUubm9kZS5yZXNvbHZlKGRlbGVnYXRlLmhvc3RlZFpvbmVOYW1lU2VydmVycyksXG4gICAgICBUVEw6ICcxMzM3JyxcbiAgICB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG59O1xuXG5jbGFzcyBUZXN0QXBwIHtcbiAgcHVibGljIHJlYWRvbmx5IHN0YWNrOiBjZGsuU3RhY2s7XG4gIHByaXZhdGUgcmVhZG9ubHkgYXBwID0gbmV3IGNkay5BcHAoKTtcblxuICBjb25zdHJ1Y3RvcigpIHtcbiAgICBjb25zdCBhY2NvdW50ID0gJzEyMzQ1Njc4OTAxMic7XG4gICAgY29uc3QgcmVnaW9uID0gJ2Jlcm11ZGEtdHJpYW5nbGUnO1xuICAgIHRoaXMuYXBwLm5vZGUuc2V0Q29udGV4dChgYXZhaWxhYmlsaXR5LXpvbmVzOiR7YWNjb3VudH06JHtyZWdpb259YCxcbiAgICAgIFtgJHtyZWdpb259LTFhYF0pO1xuICAgIHRoaXMuc3RhY2sgPSBuZXcgY2RrLlN0YWNrKHRoaXMuYXBwLCAnTXlTdGFjaycsIHsgZW52OiB7IGFjY291bnQsIHJlZ2lvbiB9IH0pO1xuICB9XG5cbiAgcHVibGljIHN5bnRoZXNpemVUZW1wbGF0ZSgpIHtcbiAgICByZXR1cm4gdGhpcy5hcHAuc3ludGhlc2l6ZVN0YWNrKHRoaXMuc3RhY2submFtZSk7XG4gIH1cbn1cbiJdfQ==