import argparse
from ..mixin import TemplateRendererMixin
from ..__version__ import get_version
from argparse import ArgumentParser


class Auto(TemplateRendererMixin):
    """Ajf file generater with auto-fragmentation.

    ::

      python -m fmojinja.abinitmp auto  $ python -m fmojinja.abinitmp auto -h

    ::

        usage: __main__.py auto [-h] [--method METHOD] [--memory MEMORY] -p PDB [-g GEOM] [-c CHARGE] [-lc LIGAND_CHARGE] [--cpf-version CPF_VERSION] [-amino FRAG_SIZE_AMINOACID]
                        [-base FRAG_SIZE_NUCLEOTIDE] [--rsolv RSOLV] [-np NPROC_PER_FRAG] [-b BASIS_SET] [--auto-frag-only]
        optional arguments:
          -h, --help            show this help message and exit
          --method METHOD
          --memory MEMORY       Default=6000, max memory size per process.
          -p PDB, --pdb PDB
          -g GEOM, --geom GEOM
          -c CHARGE, --charge CHARGE
          -lc LIGAND_CHARGE, --ligand-charge LIGAND_CHARGE
          --cpf-version CPF_VERSION
          -amino FRAG_SIZE_AMINOACID, --frag-size-aminoacid FRAG_SIZE_AMINOACID
          -base FRAG_SIZE_NUCLEOTIDE, --frag-size-nucleotide FRAG_SIZE_NUCLEOTIDE
          --rsolv RSOLV
          -np NPROC_PER_FRAG, --nproc-per-frag NPROC_PER_FRAG
          -b BASIS_SET, --basis-set BASIS_SET
          --auto-frag-only


    Example
    -------

    ::

        $   -p your.pdb

    ::

        ! Generated by fmojinja version 0.0.20210727
        &CNTRL
          Method='MP2'
          Memory=6000
          ReadGeom='your.pdb'
          WriteGeom='your.cpf'
          Charge=0
        /
        &FMOCNTRL
          FMO='ON'
          NBody=2
          AutoFrag='ON'
          FragSizeResidue=1
          FragSizeAminoacid='+amino'
          FragSizeNucleotide='+base'
          LigandCharge='None'
          Rsolv='Na=0.0,Br=0.0,Cl=0.0,Mg=0.0'
          NP=1
        /
        &BASIS
          BasisSet='6-31G*'
        /



    """

    @classmethod
    def template(cls) -> str:
        return f"! Generated by fmojinja version {get_version()}" + """ 
&CNTRL
  Method='{{ method }}'
  Memory={{ memory }}
  ReadGeom='{{ pdb }}'
  WriteGeom='{{ geom if geom else pdb.replace(".pdb", ".cpf") }}'
  Charge={{ charge if charge else 0 }} 
  {%- if cpf_version %}
  CpfVer={{ cpf_version }} {% endif %}
  {%- if auto_frag_only %}
  Nprint=0 {% endif %}
/
&FMOCNTRL
  FMO='ON'
  NBody=2
  AutoFrag='ON'
  FragSizeResidue=1
  FragSizeAminoacid='{{ frag_size_aminoacid }}'
  FragSizeNucleotide='{{ frag_size_nucleotide }}'
  LigandCharge='{{ ligand_charge }}'
  Rsolv='{{ rsolv }}'
  NP={{ nproc_per_frag }}
/
&BASIS
  BasisSet='{{ basis_set }}'
/

"""

    @classmethod
    def set_arguments(cls, p: ArgumentParser) -> ArgumentParser:
        p = super(Auto, cls).set_arguments(p)
        p.add_argument("--method", default="MP2")
        p.add_argument("--memory", type=int, default="6000", help='Default=6000, max memory size per process.')
        p.add_argument("-p", "--pdb", required=True)
        p.add_argument("-g", "--geom", type=str)
        p.add_argument("-c", "--charge", type=int, default=0)
        p.add_argument("-lc", "--ligand-charge", type=int)
        p.add_argument("--cpf-version")
        p.add_argument("-amino", "--frag-size-aminoacid", default="+amino")
        p.add_argument("-base", "--frag-size-nucleotide", default="+base")
        p.add_argument("--rsolv", default="Na=0.0,Br=0.0,Cl=0.0,Mg=0.0")
        p.add_argument("-np", "--nproc-per-frag", type=int, default=1)
        p.add_argument("-b", "--basis-set", default="6-31G*")
        p.add_argument("--auto-frag-only", action="store_true")
        return p