from ..mixin import TemplateRendererMixin
from ..__version__ import get_version
from argparse import ArgumentParser


class PdbReformer(TemplateRendererMixin):
    """The awk script generator to reformat pdb files for abinitmp.

        Following things will be done by the awk script for abinitmp execution.

        * 1. Left-justify residue names
            * e.g. " NA" -> "NA "
        * 2. Rename chain by given sequence number.

    Example
    -------
    * Set chain residue 1-100 as A, residue 101-200 as B and residue 201- as C.
    ::

        $ python -m fmojinja.awk pdb_reformer -c 101 201

    ::

        # Generated by fmojinja version 0.0.20210727
        BEGIN { i = 1; tag[1] = "A"; tag[2] = "B"; tag[3] = "C" }
        # Left-justify
        /^ATOM|^HETATM/ && substr($0, 18, 2) == "  " { $0 = substr($0, 0, 17) substr($0, 20, 1) "  "  substr($0, 21) }
        /^ATOM|^HETATM/ && substr($0, 18, 1) == " " { $0 = substr($0, 0, 17) substr($0, 19, 2) " "  substr($0, 21) }
        # ToUpper Resname
        /^ATOM|^HETATM/ { $0 = substr($0, 0, 17) toupper(substr($0, 18, 3)) substr($0, 21) }
        # Reformat chain names
        !start_seq_101 && int(substr($0, 23, 4)) == 101 { start_seq_101 = 1; i = i + 1 }
        !start_seq_201 && int(substr($0, 23, 4)) == 201 { start_seq_201 = 1; i = i + 1 }
        # Print
        { print substr($0, 0, 21) tag[i] substr($0, 23) }


    * Another example

    ::

        $ python -m fmojinja.awk pdb_reformer -c 380 | awk -f - from.pdb > to.pdb


    from.pdb::

        --
        ATOM   6099  C   ILE   379      26.623  49.171  22.144  0.8343  1.7000       C
        ATOM   6100  O   ILE   379      27.821  48.816  22.165 -0.8190  1.5000       O
        ATOM   6101  OXT ILE   379      26.211  49.995  21.297 -0.8190  1.5000       O
        ATOM   6102 HO5'  A5   380      43.944  29.387  55.113  0.4295  0.8000       H
        ATOM   6103  O5'  A5   380      43.762  29.777  55.999 -0.6223  1.5000       O
        --

    to.pdb::

        --
        ATOM   6099  C   ILE A 379      26.623  49.171  22.144  0.8343  1.7000       C
        ATOM   6100  O   ILE A 379      27.821  48.816  22.165 -0.8190  1.5000       O
        ATOM   6101  OXT ILE A 379      26.211  49.995  21.297 -0.8190  1.5000       O
        ATOM   6102 HO5' A5  B 380      43.944  29.387  55.113  0.4295  0.8000       H
        ATOM   6103  O5' A5  B 380      43.762  29.777  55.999 -0.6223  1.5000       O
        --

    Note
    ----
        see also: fmojinja.awk.makefile.pdb_reformer
    """
    
    @classmethod
    def template(cls) -> str:
        return f"# Generated by fmojinja version {get_version()}" + """
BEGIN { i = 1
{%- set letters = ("ABCDEFGHIJKLMNOPQRSTUVWXYZ" | list) %}
{%- for _ in chain_starts %}; tag[{{ loop.index }}] = "{{ letters[loop.index0] }}"{% endfor -%}
; tag[{{ (chain_starts | length) + 1}}] = "{{ letters[(chain_starts | length)] }}" }
# Left-justify
/^ATOM|^HETATM/ && substr($0, 18, 2) == "  " { $0 = substr($0, 0, 17) substr($0, 20, 1) "  "  substr($0, 21) }
/^ATOM|^HETATM/ && substr($0, 18, 1) == " " { $0 = substr($0, 0, 17) substr($0, 19, 2) " "  substr($0, 21) }
# ToUpper Resname
/^ATOM|^HETATM/ { $0 = substr($0, 0, 17) toupper(substr($0, 18, 3)) substr($0, 21) }
# Reformat chain names
{% for seq_id in chain_starts -%}
!start_seq_{{ seq_id }} && int(substr($0, 23, 4)) == {{ seq_id }} { start_seq_{{ seq_id }} = 1; i = i + 1 }
{% endfor %}# Print
{ print substr($0, 0, 21) tag[i] substr($0, 23) }


"""

    @classmethod
    def set_arguments(cls, p: ArgumentParser) -> ArgumentParser:
        p = super(PdbReformer, cls).set_arguments(p)
        p.add_argument("-c", "--chain-starts", nargs="*", default=[])
        return p