from ...__version__ import get_version
from ...mixin import TemplateRendererMixin
from jinja2 import Environment
from argparse import ArgumentParser
from ...jinja_filters import broadcast_n
from pathlib import Path


class StepPrep(TemplateRendererMixin):

    @classmethod
    def template(cls) -> str:
        return "{{title}}" + f"# Generated by fmojinja version {get_version()}" + """
MD_ENGINE = sander # e.g. sander, pmemd, pmemd.cuda, mpi -n 8 pmemd.MPI
PREFIX := {{ prefix }}
NSTEPS := {{ jobs | length }}
INPUTS :={% for _ in jobs %} $(PREFIX){{ loop.index }}.mdin{% endfor %}

define wrap_num_w_tag
$(shell awk 'BEGIN{\
d[1]="a";d[2]="b";d[3]="c";d[4]="d";d[5]="e";d[6]="f";d[7]="g";d[8]="h";d[9]="i";\
print d[length($(1))] "." int($(1))}')
endef


.PHONY: gen
gen: $(PREFIX).prmtop $(INPUTS)
$(PREFIX).prmtop:
\tcp {{ prmtop }} $@
$(PREFIX)0.restrt:
\tcp {{ inpcrd }} $@
{% for i in range(1, (jobs | length) + 1) %}
$(PREFIX){{ i }}.mdin:
\tpython -m fmojinja.sander {{ jobs | broadcast(i) }}\
{%- if titles %} -t {{ titles | broadcast(i) }} {%- else %} -t {{ i }}_{{ jobs | broadcast(i) }}{% endif %}\
{%- if restraint_mask %} -rm "{{ restraint_mask | broadcast(i) }}" -rw {{ restraint_wt | broadcast(i) }}{% endif %}\
{%- if (jobs | broadcast(i)) == "min" %} -mc {{ min_maxcyc }} -drms {{ min_drms }}{% else %}\
{{ " -gl {}".format(gamma_ln) if gamma_ln else "" }}\
{{ " -temp0 {}".format(temperature) if temperature else "" }}\
{{ " -pres0 {}".format(pressure) if pressure else "" }}\
{{ " -nt {}".format(sampling_time | broadcast(i)) if sampling_time else "" }}\
{{ " -nstlim {}".format(nsteps_limit | broadcast(i)) if nsteps_limit else "" }}\
{{ " -dt {}".format(delta_time) if delta_time else "" }}{% endif %}\
{{ " -cut {}".format(cut_off) if cut_off else "" }}\
{{ " -ig {}".format(seed) if seed else "" }} > $@
{%- endfor %}


.PHONY: run
run: $(PREFIX)_finish.restrt
$(PREFIX)_finish.restrt: gen $(PREFIX)$(NSTEPS).restrt
\tcp $(PREFIX)$(NSTEPS).restrt $@

define sander_expr 
$(PREFIX)$(1).restrt: $(PREFIX)$(2).restrt
\t$(MD_ENGINE) -O \\
\t-i $(PREFIX)$(1).mdin \\
\t-o $(PREFIX)$(1).mdout \\
\t-p $(PREFIX).prmtop \\
\t-c $(PREFIX)$(2).restrt \\
\t-ref $(PREFIX)0.restrt \\
\t-r $(PREFIX)$(1).restrt \\
\t-x $(PREFIX)$(1).mdcrd \\
\t-inf $(PREFIX)$(1).mdinfo
endef
$(foreach i,$(shell seq $(NSTEPS) -1 1),$(eval $(call sander_expr,$(i),$(shell expr $(i) - 1))))


.PHONY: link2ps
link2ps:
{%- set accum_time = namespace(value=0) %}
{%- for job in jobs %}
{%- if job != "min" %}
{%- set accum_time.value = accum_time.value + ((( nsteps_limit | broadcast(loop.index)) * delta_time) | int) %}
\tln $(PREFIX){{ loop.index }}.restrt $(PREFIX).$(call wrap_num_w_tag, {{ accum_time.value }}).restrt
\tln $(PREFIX){{ loop.index }}.mdcrd $(PREFIX).$(call wrap_num_w_tag, {{ accum_time.value }}).mdcrd
{%- endif %}{%- endfor %}


.PHONY: clean
clean:
\trm $(PREFIX)*.* 

"""

    @classmethod
    def render(cls, **kwargs) -> str:
        env = Environment()
        env.filters["broadcast"] = broadcast_n(len(kwargs.get("jobs")))
        return env.from_string(cls.template()).render(**kwargs)

    @classmethod
    def set_arguments(cls, p: ArgumentParser) -> ArgumentParser:
        p = super(StepPrep, cls).set_arguments(p)
        p.add_argument("-P", "--prefix", default="prep")

        p.add_argument("-p", "--prmtop", type=Path, required=True)
        p.add_argument("-c", "--inpcrd", type=Path, required=True)
        p.add_argument("-j", "--jobs", required=True, nargs="+", help="number of steps")
        p.add_argument("-t", "--titles", nargs="+", help="title for the input.")
        p.add_argument("-mc", "--min-maxcyc", default=10000)
        p.add_argument("-drms", "--min-drms", default=1e-4)
        p.add_argument("-nt", "--sampling-time", type=int, nargs="+")
        p.add_argument("-nstlim", "--nsteps-limit", type=int, nargs="+")
        p.add_argument("-dt", "--delta-time", type=float, default=0.0005)

        p.add_argument("-cut", "--cut-off", default=12.0)
        p.add_argument("-rm", "--restraint-mask", nargs="+", help="restraint mask. e.g. '!@H=' ")
        p.add_argument("-rw", "--restraint-wt", nargs="+", help="the weight (kcal/mol angstrom) for the positional restraints")
        p.add_argument("-ig", "--seed", default=-1)
        return p
