# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['globus_action_provider_tools',
 'globus_action_provider_tools.flask',
 'globus_action_provider_tools.testing']

package_data = \
{'': ['*']}

install_requires = \
['arrow>=0.15.5,<0.16.0',
 'dogpile.cache>=0.7.1,<0.8.0',
 'globus-sdk>=1.9,<2.0',
 'jsonschema>=3,<4',
 'pybase62>=0.4.0,<0.5.0',
 'pyyaml>=5.3.1,<6.0.0']

extras_require = \
{':python_version >= "3.6" and python_version < "3.7"': ['dataclasses>=0.7,<0.8']}

entry_points = \
{'console_scripts': ['whattimeisit-provider = '
                     'examples.flask.whattimeisitrightnow.app.app:main']}

setup_kwargs = {
    'name': 'globus-action-provider-tools',
    'version': '0.10.0',
    'description': 'Tools to help developers build services that implement the Action Provider specification.',
    'long_description': 'Action Provider Tools Introduction\n==================================\n\nThis is an experimental toolkit to help developers build Action Providers for\nuse in Globus Automate including for invocation via Globus Flows.\n\nAs this is experimental, no support is implied or provided for any sort of use\nof this package. It is published for ease of distribution among those planning\nto use it for its intended, experimental, purpose.\n\nThis README is intended to introduce the concepts, requirements and overview of\nthis toolkit to be used when implementing an Action for use with Globus\nAutomate. The three main sections are:\n\n1) an introduction to the interface required of all Actions, \n2) A description of how to prepare your service to use Globus Auth \n3) A description of the content of this toolkit and how it can be used in a\n   Python implementation.\n\nThis overview does not provide any guidance on how to host or operate the Action\nso that it is accessible via the internet.\n\n\nActions Overview\n================\nThe fundamental purpose of the Globus Automate platform is to tie together\nmultiple operations or units of work into a coordinated orchestration. We refer\nto each of these operations as an *Action*. In particular, the *Flows* service\nprovides a means of coordinating multiple actions across potentially long\nperiods of time to perform some aggregate function larger than any single Action\nprovides. The *Triggers* service ties *Events*, or occurrences within a managed\nenvironment, to Actions such that each occurrence of the Event automatically\ninvokes the Action associated with it.\n\nIn both the Flows and the Triggers cases, the Actions require a uniform\ninterface for invocation, monitoring and possibly termination so that new\nActions may be introduced without requiring customization or re-implementation\nof the invoking services. We refer to the service endpoints which can be invoked\nin this manner as *Action Providers* and the uniform interface for interacting\nwith the Action Providers as the *Action Provider Interface*. We provide here an\noverview of the *Action Provider Interface* as a guide for use when implementing\nan *Action Provider*. \n\nThe Action Provider Interface\n-----------------------------\n\nThe Action Provider Interface is a RESTful model for starting, monitoring,\ncanceling and removing state  associated with the invocation of an Action.\nFollowing the REST resource life-cycle pattern, each Action invocation returns\nan identifier representing the invocation (an *Action Instance*). This\nidentifier is used to monitor the progress of the Action Instance via further\nREST calls until its completion, or it may be used to request cancellation of\nthe instance.\n\nBecause the interface is intended to support arbitrary Action types, we\nrecognize that some Action instances may be long-running (asynchronous) such as\nthe execution of a computational job. Other Actions may be short-running\n(synchronous), able to return their final result directly in response to their\ninvocation request as is the case in typical RESTful models. The Action\nLife-cycle described below specifically supports these execution modes as well\nas handling failures and Actions which may be, temporarily, unable to make\nprogress.\n\nAction Life-cycle\n^^^^^^^^^^^^^^^^^\n\nThe Life-cycle of an Action defines the set of states that the Action may be in,\nand how it can  transition among the states. The states are defined as follows:\n\n*  ``ACTIVE``: The Action is executing and is making progress toward completion\n\n* | ``INACTIVE``: The Action is paused in its execution and is not making\n    progress toward completion. Out-of-band (i.e. not via the Action Provider\n    Interface) measures may be required to allow the Action to proceed.\n\n* | ``SUCCEEDED``: The Action reached a completion state which was considered\n    "normal" or not due to failure or other unrecoverable error. \n\n* | ``FAILED``: The Action is in a completion state which is "not normal" such as\n    due to an error condition which is not considered recoverable in any manner. \n\n* | ``RELEASED``: The Action Provider has removed the record of the existence of\n    the Action.  Further attempts to interact with the Action will be errors as if\n    the Action had never existed. All resources associated with the Action may have\n    been deleted or removed. This is not a true state in the sense that the state\n    can be observed, but ultimately all Actions will be released and unavailable for\n    further operations. Any subsequent references to the Action, e.g. via the API\n    methods described below, will behave as if the Action never existed.\n\nUpon initial creation of an Action (see operations below), the Action may be in\nany of the first four states. If it is in an ``ACTIVE`` or ``INACTIVE`` state,\nthe Action is considered "asynchronous" and further queries to get the state of\nthe Action may return updated information. If the Action is in a the\n``SUCCEEDED`` or ``FAILED`` states, the Action is synchronous, all information\nabout the Action is returned on the creation operation and no changes to the\nstate are possible.\n\nAn asynchronous Action may change state between ``ACTIVE`` and ``INACTIVE``\nduring its life time, and may update further details about its progress while\nin either of these states. When a completed state of ``SUCCEEDED`` or ``FAILED``\nis reached, the Action state cannot be updated further. The Action Provider is,\nhowever, required to maintain this final state for some period of time so that\nthe client of the Action may retrieve the completion state. Upon completion, the\nclient may request that the Action be "released" or the Action Provider may do\nso on its own after the required time-out occurs. To save server resources, it\nis preferred that the client release the Action when it has reliably retrieved\nand processed the final state.\n\nAction Provider Interface and Document Types\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\nThe primary purpose of the Action Provider Interface is to securely support and\nreport Actions progressing through the life-cycle described above. The document\ntypes supporting this are the initial Action invocation *Action Request*\ndocument, and the *Action Status* document which contains the life-cycle status\ndescribed above along with additional detailed status information specific to\nthe type of Action being executed.\n\n.. note:: \n    Below, we describe URL paths where operations can be performed. We assume that\n    all of these share a common "Base URL" which we don\'t name in this document. The\n    Base URL may be at any place in the URL path namespace desired by the Action\n    Provider, and so may be used in conjunction with any other service URLs it may\n    support.\n\n.. note:: \n    For brevity and clear presentation, in the descriptions of document types in\n    the following  sections, we present the key concepts, but do not enumerate\n    every option or field on the documents. Refer to the toolkit components,\n    including the OpenAPI format specification (as described in the toolkit\n    section), for a complete definition.\n\nStarting an Action: The Action Request Document and the ``POST /run`` Method\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\nStarting an Action is performed by making a REST ``POST`` request to the path\n``/run`` containing an Action Request document. The request document contains\nthe following fields:\n\n* | ``request_id``: A client-generated Identifier for this request. A user may\n    re-invoke the ``/run`` method with the same ``request_id`` any number of times,\n    but the Action must only be initiated once. In this way, the user may re-issue\n    the request in case it cannot be determined if a request was successfully\n    initiated for example due to network failure.\n\n* | ``body``: An Action Provider-specific object which provides the input for\n    the Action to be performed. The ``body`` must conform to the input\n    specification for the Action Provider being invoked, and thus the client must\n    understand the requirements of the Action Provider when providing the value of\n    the ``body``. Thus, the Action Provider must provide documentation on the format\n    for the ``body`` property.\n\n* | ``manage_by`` and ``monitor_by``: Each of these is a list of principal\n    values in `URN format <https://docs.globus.org/api/search/#principal_Urns>`_,\n    and they allow the user invoking the Action to delegate some capability over the\n    Action to other principals. ``manage_by`` defines the principals who are allowed\n    to attempt to change the execution of the Action (see operations ``/cancel`` and\n    ``/release`` below) while it is running. ``monitor_by`` defines principals which\n    are allowed to see the state of the Action before its state has been destroyed\n    in a release operation. In both cases, the Globus Auth identity associated with\n    the ``/run`` operation is implicitly part of both the ``manage_by`` and\n    ``monitor_by`` sets. That is, the invoking user need not include their own\n    identity into these lists.\n\nAny request to the ``/run`` method which contains an Action Request which\nadheres to the input schema will return an Action Status document as described\nin the next section. \n\nMonitoring and Completing an Action: The Action Status Document and Management Methods\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\nAll information about an Action is contained in the Action Status document which\nis returned on almost all operations related to an Action (the exception is the\nlog operation which is optional and is described briefly below). Notable fields\nof the Action Status document include:\n\n* | ``action_id``: The unique identifier for this particular action. The\n    ``action_id`` may be any string, and it should be treated as an opaque value\n    (that is, having no semantic or implied meaning) by the client. The client will\n    first learn of an Action\'s ``action_id`` in the Action Status returned by the\n    ``/run`` method.\n\n* | ``status`` and ``display_status``: These provide the description of the\n    state of the Action. ``status`` is the specific life-cycle value as described\n    above. ``display_status`` is an optional field the Action Provider may supply\n    which gives a short text description of the status using language which is\n    specific to the Action.\n     \n* | ``details``: The Action Provider-specific state, particularly the completion\n    state, of the Action are returned in the ``details`` field. In the completion\n    states, the ``details`` can be considered the "result" or the "return value" of\n    the Action. It is the successful return value for a ``SUCCEEDED`` status,  and\n    it is the error result for the ``FAILED`` status. The exact content in\n    ``details`` is always specific to the Action Provider, so must be documented by\n    the Action Provider to describe its interpretation to clients.\n\n* | ``monitor_by`` and ``manage_by``: Same as in the Action Request.\n\n* | ``start_time`` and ``completion_time``: Represent the time the Action was\n    first received by the  ``/run`` operation and the time the Action Provider\n    determined that the Action reached a completed state (``SUCCEEDED`` or\n    ``FAILED``) respectively. Action Providers are not required to continuously\n    monitor the progress of Actions, so the ``completion_time`` noted may be\n    different than the executed Action\'s actual completion time.  These values\n    **may** be the same in the case of a synchronous operation, but\n    ``completion_time`` must never be before ``start_time``.\n\n* | ``release_after``: As stated above, Action state is automatically removed\n    from the Action Provider after some time interval once it reaches a completion\n    state. The ``release_after`` is a time duration, in seconds, which states how\n    long after completion the Action will automatically be released. A typical value\n    would be 30-days, but Action Providers may define their own policy which is to\n    be exposed in the Action Status.\n\nIn addition to the ``/run`` method described above, the Action Status is the\n"universal" return value from operations on an Action. We describe the\noperations on Actions next. Each uses the ``action_id`` as part of the URL path\nmuch like other RESTful resources do with their ids, and none of them require\nany input body. \n\n* | ``GET /<action_id>/status``: This is a read-only operation for retrieving\n    the most recent state of the Action. It is commonly used to poll an Action\'s\n    state while awaiting it entering a completion state. Use of this API call\n    requires that the user authenticate with a principal value which is in the\n    ``monitor_by`` list established when the Action was started.\n\n* | ``POST /<action_id>/cancel``: Cancellation provides an advisory or hint to\n    the Action Provider that the user does not want the Action to continue\n    execution. The Action Provider is not required to ensure immediate completion or\n    that the cancel operation truly causes the Action to terminate in any manner\n    other than it would have without the cancel request. Thus, the Action Status\n    returned from the cancel operation may contain a non-completion state. If the\n    Action is already in a completed state, the Action Provider may treat the\n    request much as a ``/status`` request to simply return the current status. Use\n    of this API call requires that the user authenticates with a principal value\n    which is in the ``manage_by`` list established when the Action was started. \n\n* | ``POST /<action_id>/release``: As described in the section on life-cycle,\n    the very last step of the life-cycle is for the Action state to be removed from\n    the Action Provider. A user can specify that it has retrieved the final state or\n    is no longer interested in the state using the ``/release`` operation which\n    returns the final state. If the Action is not already in a completion state,\n    ``/release`` will return an error as this operation does not attempt to stop\n    execution (that is what ``/cancel`` does). The Action Status document returned\n    from ``/release`` will be the last record of the Action present at the Action\n    Provider. After the call to ``/release`` the ``action_id`` is no longer valid,\n    and use in any other calls will return an error, most likely an HTTP status 404\n    indicating the Action was not found.\n\nDetailed Execution History: logging\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\nSome Actions, particularly those that are long running, may have associated with\nthem a list or log of activities or sub-events which occurred during the\nAction\'s life. This detailed log is typically larger, more complex, or more\nfine-grain than the snapshot of the status returned by the ``/status`` method.\nNot all Action Providers or Actions are suitable for logging, so support is\nconsidered optional and will be advertised by the Action Provider in its\ndescription (see below). The request to retrieve the log takes the form ``GET\n/<action_id>/log?<filters,pagination>``. The filters and pagination query\nparameters are used to limit (e.g. based on start time) which log records to\nretrieve and the pagination parameter is used to scroll through a long set of\nlog records across multiple requests. Each record in the log contains the\nfollowing properties:\n\n* | ``time``: A timestamp representing the time this log record occurred.\n\n* | ``code``: A short Action Provider-specific description of the type of the log record.\n\n* | ``description``: A textual description of the purpose, cause, or information\n    on the log record.\n\n* | ``details`` (optional): An object providing additional and structured Action\n    Provider-specific representation of the log record.\n\n\nAction Provider Introspection (``GET /``)\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\nThe Automate platform is intended to help users both find and make use of the\nvariety of Action Providers which may be available on the network. The primary\nmeans of accomplishing this assistance is by making Action Providers, the\nservices which implement the Action Provider Interface, self-describing via an\n*Introspection* interface. Accessing the introspection method is performed\nsimply via a ``GET /``. That is, the HTTP ``GET`` method on the Base URL. The\nreturned JSON document contains the following fields:\n\n* | ``api_version``: A version string defining the version of the Action\n    Provider Interface supported by the Action Provider. The version described in\n    this document and currently the only version available will have value\n    ``"1.0"``.\n\n* | ``title``, ``subtitle``, ``description``, ``keywords``: Each of these\n    provide human-readable text which helps a user discover the purpose of the\n    Action Provider.\n\n* | ``visible_to`` and ``runnable_by``: Access to the action provider is limited\n    by and published through these properties. Each contains a list of principal\n    values in URN format. ``visible_to`` controls who can retrieve the information\n    via introspection (this operation) and may contain the string ``"public"``\n    indicating that all users, even those who present no credentials, may access the\n    information. The ``runnable_by`` property enumerates who can use the ``/run``\n    method to start an Action at this provider. It allows the string\n    ``"all_authenticated_users"`` indicating that any user who presents valid\n    credentials via a Bearer token may start an Action at the provider.\n\n* | ``synchronous`` and ``log_supported``: These are boolean values which simply\n    describe capabilities and modes for the Action Provider. If ``synchronous`` is\n    true, a user calling ``/run`` can assume that the returned status will always be\n    in a completed (``SUCCEEDED`` or ``FAILED``) state and there will never be a\n    need to poll using the ``/status`` method (use of ``/release`` is still\n    permitted and encouraged to remove the status from the Action Provider). As\n    indicated in the discussion of the ``/log`` method, support for it is optional,\n    and the ``log_supported`` flag provides an indication to users whether they can\n    make use of ``/log`` for fine grained monitoring of an Action.\n\n* | ``input_schema``: The ``input_schema`` value provides a complete schema\n    description for the ``body`` property of the Action Request understood by this\n    Action Provider. The schema is provided in `JSON Schema\n    <https://json-schema.org/>`_ format.\n\n\nSetting Up an Action Provider in Globus Auth\n============================================\n\nThe Action Provider Interface makes use of and is bound closely with\nauthentication via the `Globus Auth <https://globus.org/>`_ system. To\nauthenticate RESTful requests using Globus Auth, a service must register as a\n"resource server". This is a multi-step process involving use of both the Globus\nAuth developer portal, and the Globus Auth API for configuring various access\ncontrol states. To help with this process, we provide a step-by-step guide to\nusing Globus Auth for this purpose:\n\n1. Register a new App on `<https://developers.globus.org>`_ using a browser.\nAfter insuring that you are logged in to the developer portal in a browser at\nthis URL, perform the following steps:\n\n   - Click Add another project\n\n     - | Provide a name, contact email and select which of your own Globus Auth\n        linked identities are permitted to administer the project. You will be\n        required to login with this identity in future interactions with the Globus\n        Developer Portal to manipulate the resource server.\n\n   - Find your new, empty project, and select Add drop down and "new app"\n\n     - | Provide a name for the specific app within the project. This will be a\n        common name displayed to  users when they make use of the Action Provider.\n\n     - | When creating a resource server, the other fields on the app creation\n        page are not used.\n\n       - | "Redirects" is not used, but a value must be provided. You can use a\n            URL associated with your service or a placeholder value like\n            "https://localhost".\n\n       - | "Scopes" are not relevant and make no difference, so this field\n            should be left blank. The "Privacy Policy" and "Terms and Conditions"\n            may be displayed to users making use of your action provider, but they\n            are not required.\n\n   - | Make note of the "Client Id" in the expanded description of your app. This\n        value will be used elsewhere in the creation of the service and is often\n        referenced as ``client_id``.\n\n   - | In the section "Client Secrets" click "Create a new secret"\n\n     - | Provide a name which is meaningful to you. It will not be displayed to\n            other users. \n\n     - | Make note of the generated secret. Like the ``client_id`` this will be\n            used later in development. Be sure **not to lose it** as it can only be\n            displayed once. However, new client secrets can be created and old ones\n            deleted at any time should the need for a replacement secret arise.\n\n     - | Set the client_id and client_secret on your command line to follow\n            along with the rest of this guide.\n\n            .. code-block:: BASH    \n\n                export CLIENT_ID=<client_id>\n                export CLIENT_SECRET=<client_secret>\n\n2. Use the Globus Auth REST API to introspect your Action Provider Resource\nServer and create required Scopes.\n\n     .. note:: \n        In the examples below, we will use the command line tool ``curl`` to\n        perform the HTTP operations as it is very widely available. We also use\n        the command line tool ``jq`` to format the ``curl`` command\'s json\n        responses. However, other tools and clients exist for interacting with\n        REST and HTTP services, so you may need to translate the ``curl`` and\n        ``jq`` commands to your preferred tools.\n\n   - | Introspect the Globus Auth client to see the same settings you setup in\n        the developer portal. Notice we exported the ``<client_id>`` and\n        ``<client_secret>`` values generated during your registration on the Globus\n        Developer Portal into environment variables. \n    \n        .. code-block:: BASH\n\n            curl -s --user $CLIENT_ID:$CLIENT_SECRET \\ \n                https://auth.globus.org/v2/api/clients/$CLIENT_ID | jq\n        \n   - | A successful return from this command is a JSON representation of the\n        Globus Auth client similar to:\n\n        .. code-block:: JSON\n\n            {\n            "client": {\n                "scopes": [],\n                "redirect_uris": [\n                "https://localhost"\n                ],\n                "name": "My Action Provider",\n                "links": {\n                "privacy_policy": null,\n                "terms_and_conditions": null\n                },\n                "grant_types": [\n                "authorization_code",\n                "client_credentials",\n                "refresh_token",\n                "urn:globus:auth:grant_type:dependent_token"\n                ],\n                "fqdns": [],\n                "visibility": "private",\n                "project": "a47b9014-9250-4e21-9de5-b4aac81d464b",\n                "required_idp": null,\n                "preselect_idp": null,\n                "id": "8e98ba5a-21a9-4bef-ab6a-0fcdbed36405",\n                "public_client": false,\n                "parent_client": null\n              }\n            }\n\n\n   - | Of note is the field ``scopes``. ``scopes`` are created to identify\n        operations on the Action Provider. Typically, an Action Provide defines just\n        one scope and it is provided to users in the Action Provider\'s introspection\n        (``GET /``) information in the field ``globus_auth_scope``.\n\n     - Creating a Scope:  \n        - | Creation of a scope is required as the scope will be used in\n            authenticating REST calls on the Action Provider.\n        \n        - | Start by creating a "scope definition" JSON document in the\n                following format replacing the ``name``, ``description`` and optionally\n                the ``scope_suffix``\n\n            .. code-block:: JSON\n\n                {\n                    "scope": {\n                        "name": "Action Provider Operations",\n                        "description": "All Operations on My Action Provider",\n                        "scope_suffix": "action_all",\n                        "dependent_scopes": [{\n                            "optional": false,\n                            "requires_refresh_token": true,\n                            "scope": "73320ffe-4cb4-4b25-a0a3-83d53d59ce4f"\n                         }],             \n                        "advertised": true,\n                        "allow_refresh_tokens": true\n                    }\n                }\n\n\n        - | The ``name`` and ``description`` fields are purely informative and\n            will be presented to other users who use the Globus Auth API to lookup\n            the scope. The ``scope_suffix`` will be placed at the end of the\n            generated "scope string" which is a URL identifier for the scope. It\n            provides the context for the operations this scope covers among all\n            operations your service provides. For Action Providers, we commonly use\n            ``action_all`` to indicate all operations defined by the Action Provider\n            API, but any string is acceptable.\n\n        - | ``dependent_scopes`` define scopes of other Globus Auth resource\n            servers that your Action Provider will invoke to perform its work. For\n            example, if your Action Provider uses Globus Transfer to first move some\n            data to compute upon, the scope for the Globus Transfer service would be\n            placed in the ``dependent_scopes`` list. In the most common case, as\n            shown in the example, the scope for the `Globus Groups API\n            <https://docs.globus.org/api/groups/>`_ (with UUID\n            ``73320ffe-4cb4-4b25-a0a3-83d53d59ce4f``) should be listed. This allows\n            your Action Provider to determine what groups a user calling the\n            provider belongs to and can therefore enforce policies, such as\n            ``runnable_by`` or ``monitor_by`` based on group membership. If this\n            scope is not listed as a dependent scope, the Action Provider Tools\n            library will not be able to, and will therefore not attempt to, retrieve\n            a user\'s groups and so no policies based on Groups may be used. We\n            encourage you to consult the `Globus Auth Documentation\n            <https://docs.globus.org/api/auth/>`__ for more information on creation\n            and management of Scopes for more advanced scenarios such as other\n            dependent Globus Auth based services such as Globus Transfer.\n            \n            Note: Scopes supplied in the dependent_scopes array must be\n            identified by their UUID. The snippet below demonstrates how to\n            lookup a scope\'s UUID based on its uniquely idenfitfying FQDN\n\n                .. code-block:: BASH\n                \n                    # Target FQDN is https://auth.globus.org/scopes/actions.globus.org/transfer/transfer\n                    export SCOPE_STRING=https://auth.globus.org/scopes/actions.globus.org/transfer/transfer\n                    curl -s -u "$CLIENT_ID:$CLIENT_SECRET" \\\n                        "https://auth.globus.org/v2/api/scopes?scope_strings=$SCOPE_STRING" | jq ".scopes[0].id"\n\n        \n        - | The ``advertised`` property indicates whether the scope will be\n            visible to all users who do scope look ups on Globus Auth. You may\n            select either ``true`` or ``false`` for this depending on your own\n            policy. ``allow_refresh_tokens`` should generally be set to ``true``,\n            indicating that a client of the Action Provider who has authenticated\n            the user via Globus Auth is allowed to refresh that authentication\n            without further interactions from the user. Especially in the case where\n            an Action may be long running and is monitored by an automated system\n            like Globus Flows, it is important that token refresh is permitted.\n\n        - | With the scope creation JSON document complete, use the following\n            REST interaction to create the scope in Globus Auth via the ``curl``\n            command\n\n            .. code-block:: BASH\n\n                curl -s --user "$CLIENT_ID:$CLIENT_SECRET" -H \\\n                    \'Content-Type: application/json\' \\\n                    -XPOST https://auth.globus.org/v2/api/clients/$CLIENT_ID/scopes \\\n                    -d \'<Insert Scope creation document from above>\' | jq\n\n        - | This should return the definition of the new scope matching the\n            values provided in your scope creation document. As an example:\n\n            .. code-block:: JSON\n\n                {\n                    "scopes": [\n                        {\n                            "dependent_scopes": [{\n                                "optional": false,\n                                "requires_refresh_token": true,\n                                "scope": "73320ffe-4cb4-4b25-a0a3-83d53d59ce4f"\n                             }],\n                            "description": "<your description>",\n                            "allows_refresh_token": true,\n                            "client": "<client_id>",\n                            "advertised": true,\n                            "scope_string": "https://auth.globus.org/scopes/<client_id>/action_all",\n                            "id": "<A UUID for this scope>",\n                            "name": "<your scope name>"\n                        }\n                    ]\n                }\n\n        - | The returned ``scope_string``, which always takes the form of a URL,\n            will be the value exposed to users who wish to authenticate with Globus\n            Auth to use your Action Provider. It will be part of the Action Provider\n            description document, returned on the Action Provider Introspection\n            operation (``GET /``) with the key ``globus_auth_scope``.\n\n        - | Note that the returned value is an *array* of scopes. That is, more\n            than one scope definition may be generated from the single scope\n            creation request. This happens in the uncommon case where an FQDN has\n            been registered for your ``client_id`` (refer to the `Globus Auth\n            Documentation <https://docs.globus.org/api/auth/>`_ for information on\n            FQDN registration if you desire it, though it is not recommended). In\n            this case, a similar scope definition will also be generated, but the\n            ``scope_string`` will contain the FQDN value(s). The ``scope_string``\n            values may be used interchangeably both by users requesting\n            authentication to the Action Provider and in the ``globus_auth_scope``\n            value of the Action Provider Description. \n\n        - | Check that the created scope(s) are correctly associated with the\n            Action Provider:\n\n            .. code-block:: BASH\n\n                curl -s --user $CLIENT_ID:$CLIENT_SECRET \\\n                    https://auth.globus.org/v2/api/clients/$CLIENT_ID | jq\n\n3. Once your app and its scope(s) have been created and verified, remove your\ncredentials from your command line environment.  Be sure to take note of the\nclient ID and its associated client secret for use other places in the toolkit.\n\n            .. code-block:: BASH\n\n                unset CLIENT_ID CLIENT_SECRET\n                \nUsing the Toolkit\n==================\n\nThis toolkit provides the following components:\n\n1. Authentication helpers that make it easier to validate Globus Auth tokens and\ndetermine if a given request should be authorized\n\n2. An `OpenAPI v3 specification <http://spec.openapis.org/oas/v3.0.2>`_ and\nassociated helpers that can be used to validate incoming requests and verify\nthe responses your Action Provider generates. This document also defines the\ninterface which must be supported by your REST API to have it function as an\nAction Provider.\n\n3. Simple bindings for the document types "Action Request" and "Action Status"\nto Python Dataclass representations and a helper JsonEncoder for serializing and\ndeserializing these structures to/from JSON.\n\n4. Helper methods for binding the REST API calls defined by the Action Interface\nto a Flask application. These helpers will perform the Authentication and\nValidation steps (as provided by components 1 and 2) and communicate with an\nAction Provider implementation using the structures defined in 3. For those\nusers building an Action Provider using Flask, this provides a simplified method\nof getting the REST API implemented and removing common requirements so the\nfocus can be on the logic of the Action provided.\n\n\nInstallation\n------------\n\nInstallation is via PyPi using, for example:\n\n.. code-block:: BASH\n\n    pip install globus-action-provider-tools\n\n\nAuthentication\n---------------\n\nThe authentication helpers can be used in your action provider as follows:\n\n.. code-block:: python\n\n    from globus_action_provider_tools.authentication import TokenChecker\n    \n    # You will need to register a client and scope(s) in Globus Auth\n    # Then initialize a TokenChecker instance for your provider:\n    checker = TokenChecker(\n        client_id=\'YOUR_CLIENT_ID\',\n        client_secret=\'YOUR_CLIENT_SECRET\',\n        expected_scopes=[\'https://auth.globus.org/scopes/YOUR_SCOPES_HERE\'],\n    )\n\n\nWhen a request comes in, use your TokenChecker to validate the access token from\nthe HTTP Authorization header.\n\n.. code-block:: python\n\n    access_token = request.headers[\'Authorization\'].replace(\'Bearer \', \'\')\n    auth_state = checker.check_token(access_token)\n\n\nThe AuthState has several properties and methods that will make it easier for\nyou to decide whether or not to allow a request to proceed:\n\n.. code-block:: python\n\n    # This user\'s Globus identities:\n    auth_state.identities\n    # frozenset({\'urn:globus:auth:identity:9d437146-f150-42c2-be88-9d625d9e7cf9\',\n    #           \'urn:globus:auth:identity:c38f015b-8ad9-4004-9160-754b309b5b33\',\n    #           \'urn:globus:auth:identity:ffb5652b-d418-4849-9b57-556656706970\'})\n    \n    # Groups this user is a member of:\n    auth_state.groups\n    # frozenset({\'urn:globus:groups:id:606dbaa9-3d57-44b8-a33e-422a9de0c712\',\n    #           \'urn:globus:groups:id:d2ff42bc-c708-460f-9e9b-b535c3776bdd\'})\n\n.. note::\n    The ``groups`` property will only have values if the Groups API scope\n    is defined as a dependent scope as described in the previous section.\n\nYou\'ll notice that both groups and identities are represented as strings that\nunambiguously signal what type of entity they represent. This makes it easy to\nmerge the two sets without conflict, for situations where you\'d like to work\nwith a single set containing all authentications:\n\n\n.. code-block:: python\n\n    all_principals = auth_state.identities.union(auth_state.groups)\n\n\nThe AuthState object also offers a helper method, ``check_authorization()`` that\nis designed to help you test whether a request should be authorized:\n\n.. code-block:: python\n\n    resource_allows = [\'urn:globus:auth:identity:c38f015b-8ad9-4004-9160-754b309b5b33\']\n    auth_state.check_authorization(resource_allows)\n    # True\n\n\nThis method also accepts two special string values, ``\'public\'`` and\n``\'all_authenticated_users\'``, together with keyword arguments that enable their use:\n\n.. code-block:: python\n\n    resource_allows = [\'public\']\n    auth_state.check_authorization(resource_allows, allow_public=True)\n    # True\n\n    resource_allows = [\'all_authenticated_users\']\n    auth_state.check_authorization(resource_allows, allow_all_authenticated_users=True)\n    # True\n\n\nCaching\n^^^^^^^\n\nTo avoid excessively taxing Globus Auth, the ``AuthState`` will, by default,\ncache identities and group memberships for 30 seconds.\n\nThe cache is initialized when you first instantiate your ``TokenChecker()``.\nYou should only need to create one TokenChecker instance for your application,\nand then you can re-use it to check each new token. In the event that you do\nneed more than one TokenChecker, be aware that all TokenChecker instances in an\napp share the same underlying cache. \n\nIt is possible to customize a TokenChecker by supplying a custom configuration\nwhich gets passed on to the dogpile cache backend.  Each new instance of a\nTokenChecker with a custom configuration will drop the cache and recreate it\nwith the desired settings.  Since all TokenCheckers share the same underlying\ncache, subsequent attempts to configure the cache will overwrite the previous\ncache\'s settings and therefore only the last applied configuration will persist.\n\n.. code-block:: python\n\n    from globus_action_provider_tools.authentication import TokenChecker\n\n    # Create TokenChecker with default settings\n    my_token_checker = TokenChecker(\n        client_id=CLIENT_ID,\n        client_secret=CLIENT_SECRET,\n        expected_scopes=EXPECTED_SCOPES,\n    )\n\n    # Creating a TokenChecker with a custom config will drop the previous cache and\n    # create it with the new settings. Both TokenCheckers will use this new cache\n    new_token_checker = TokenChecker(\n            client_id=config["client_id"],\n            client_secret=config["client_secret"],\n            expected_scopes=config["expected_scopes"],\n            cache_config={\n                "backend": "dogpile.cache.pylibmc",\n                "timeout": "60",\n                "url": ["127.0.0.1"],\n            },\n        )\n\nValidation\n----------\n\nThere is an OpenAPI v3 specification for the Action Provider API available as\ndescribed above. From this specification, we derive schemas that can be used to\ntest incoming and outgoing messages. These schemas may be used to validate input\ndocuments and output documents within the service as follows.\n\n.. code-block:: python\n\n    from globus_action_provider_tools.validation import (\n        request_validator,\n        response_validator,  \n        ValidationRequest,\n    )\n    \n    # Validating a request\n    request = ValidationRequest(provider_doc_type=\'ActionRequest\', \n        request_data={"input_data":""})\n    result = request_validator.validate(request)\n    \n    # Or a response:\n    response = ValidationRequest(provider_doc_type=\'ActionStatus\', \n        request_data={"output_data":""})\n    result = response_validator.validate(response)\n    \n    # get list of errors\n    errors = result.errors\n    \n    # or get a single string summarizing all errors\n    err = result.error_msg\n\n\nThe request and response validation functions both take a ``ValidationRequest``\nstructure which has the name of the document type to be validated against and\nthe data to be validated. At present, the document types supported are\n``ActionRequest`` and ``ActionStatus`` documents as defined above.\n\n.. note:: \n    There are additional validation helpers available for applications written\n    using the Flask framework. Those are described below in the section\n    describing the entire set of Flask helpers.\n\nData Types\n----------\n\nThe toolkit provides some simple bindings for the document types defined by the\nAction Provider Interface to type-annotated Python3 `Dataclasses\n<https://docs.python.org/3/library/dataclasses.html>`_. This can provide a\nconvenient way to manipulate these document types within an Action\nProvider implementation. We also provider an ActionProviderJsonEncoder which can\nbe used with the built-in Python json package to properly encode these data\ntypes into JSON.\n\n.. code-block:: python\n\n    from globus_action_provider_tools.data_types import (\n        ActionProviderJsonEncoder,\n        ActionStatus,\n        ActionStatusValue,\n    )\n\n    status = ActionStatus(\n        status=ActionStatusValue.SUCCEEDED,\n        creator_id=caller_id,\n        monitor_by=request.monitor_by,\n        manage_by=request.manage_by,\n        start_time=str(datetime.datetime.now().isoformat()),\n        completion_time=str(datetime.datetime.now().isoformat()),\n        release_after=60 * 60 * 24 * 30,  # 30-days in seconds\n        display_status=ActionStatusValue.SUCCEEDED.name,\n        details=result_details,\n    )\n    json_string = json.dumps(action_status, cls=ActionProviderJsonEncoder)\n\n\nFlask Helper\n------------\n\nAs Action Providers are HTTP-servers, a common approach to building them is to\nuse the `Flask <https://palletsprojects.com/p/flask/>`_ framework. To aid in\ndeveloping Flask-based Action Providers, helper methods are provided which\nencapsulate much of the other functionality in the framework: authentication,\nvalidation and serialization for easy use in a Flask-based application. Rather\nthan defining each of the Action Provider Interface routes in the Flask\napplication, helpers are provided which declare the necessary routes to Flask,\nperform the serialization, validation and authentication on the request, and\npass only those requests which have satisfied these conditions on to a\nuser-defined implementation of the routes.\n\nTo use the helpers, you must define functions corresponding to the various\nmethods of the Action Provider interface (``run``, ``status``, ``release``,\n``cancel``), and must provide the Action Provider introspection information in\nan instance of the ``ActionProviderDescription`` dataclass defined in\nthe tookit\'s ``data_types`` package. The application must also provide a Flask\n``blueprint`` object to which the toolkit can attach the new routes. It is\nrecommended that the ``blueprint`` be created with a ``url_prefix`` so that the\nAction Provider Interface routes are rooted at a distinct root path in the\napplication\'s URL namespace.\n\nA brief example of setting up the flask helper is provided immediately below. A\nmore complete example showing implementation of all the required functions is\nprovided in the *examples/watchasay* directory. It is appropriate to use the\nexample as a starting point for any new Action Providers which are developed.\n\n.. code-block:: python\n                \n    from globus_action_provider_tools.data_types import (\n        ActionProviderDescription,\n        ActionRequest,\n        ActionStatus,\n        ActionStatusValue,\n    )\n    from globus_action_provider_tools.flask import (\n        ActionStatusReturn,\n        add_action_routes_to_blueprint,\n    )\n\n    action_blueprint = Blueprint("action", __name__, url_prefix="/action")\n    \n    provider_description = ActionProviderDescription(\n        globus_auth_scope="<scope created in Globus Auth>",\n        title="My Action Provider",\n        admin_contact="support@example.com",\n        synchronous=True,\n        input_schema={}, # JSON Schema representation of the input on the request\n        log_supported=False\n    )\n\n    add_action_routes_to_blueprint(\n        action_blueprint,\n        CLIENT_ID,\n        CLIENT_SECRET,\n        CLIENT_NAME,\n        provider_description,\n        action_run,\n        action_status,\n        action_cancel,\n        action_release,\n    )\n\n\nIn this example, the values ``CLIENT_ID``, ``CLIENT_SECRET`` and ``CLIENT_NAME``\nare as defined in Globus Auth as described above (where ``CLIENT_NAME`` is\nalmost always passed as ``None`` except in the uncommon, legacy case where a\nparticular name has been associated with a Globus Auth client). The values\n``action_run``, ``action_status``, ``action_cancel`` and ``action_release`` are\nall **functions** which will be called by the framework when the corresponding\nHTTP requests are called. Where appropriate, these functions are implemented in\nterms of the toolkit\'s data types so the need for JSON serialization and\ndeserialization is greatly reduced from the application code. The framework will\nalso provide validation of input ``ActionRequest`` data to the ``/run`` method\nprior to invoking the ``action_run`` function. As long as the return value from\nthe various functions is of type ``ActionStatus``, the framework will also\ninsure that the returned JSON data conforms to the Action Provider Interface.\nThe **watchasay** example in the ``examples/`` directory demonstrates how these\nfunctions can be implemented.',
    'author': 'Jim Pruyne',
    'author_email': 'pruyne@globus.org',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
