#!/usr/bin/python

# Copyright (c) 2015 IBM
# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

DOCUMENTATION = '''
---
module: compute_flavor_info
short_description: Retrieve information about one or more flavors
author: "David Shrewsbury (@Shrews)"
description:
    - Retrieve information about available OpenStack instance flavors. By default,
      information about ALL flavors are retrieved. Filters can be applied to get
      information for only matching flavors. For example, you can filter on the
      amount of RAM available to the flavor, or the number of virtual CPUs
      available to the flavor, or both. When specifying multiple filters,
      *ALL* filters must match on a flavor before that flavor is returned as
      a fact.
    - This module was called C(openstack.cloud.compute_flavor_facts) before Ansible 2.9, returning C(ansible_facts).
      Note that the M(openstack.cloud.compute_flavor_info) module no longer returns C(ansible_facts)!
notes:
    - The result contains a list of unsorted flavors.
options:
   name:
     description:
       - A flavor name. Cannot be used with I(ram) or I(vcpus) or I(ephemeral).
     type: str
   ram:
     description:
       - "A string used for filtering flavors based on the amount of RAM
         (in MB) desired. This string accepts the following special values:
         'MIN' (return flavors with the minimum amount of RAM), and 'MAX'
         (return flavors with the maximum amount of RAM)."

       - "A specific amount of RAM may also be specified. Any flavors with this
         exact amount of RAM will be returned."

       - "A range of acceptable RAM may be given using a special syntax. Simply
         prefix the amount of RAM with one of these acceptable range values:
         '<', '>', '<=', '>='. These values represent less than, greater than,
         less than or equal to, and greater than or equal to, respectively."
     type: str
   vcpus:
     description:
       - A string used for filtering flavors based on the number of virtual
         CPUs desired. Format is the same as the I(ram) parameter.
     type: str
   limit:
     description:
       - Limits the number of flavors returned. All matching flavors are
         returned by default.
     type: int
   ephemeral:
     description:
       - A string used for filtering flavors based on the amount of ephemeral
         storage. Format is the same as the I(ram) parameter
     type: str
requirements:
    - "python >= 3.6"
    - "openstacksdk"

extends_documentation_fragment:
- openstack.cloud.openstack
'''

EXAMPLES = '''
# Gather information about all available flavors
- openstack.cloud.compute_flavor_info:
    cloud: mycloud
  register: result

- debug:
    msg: "{{ result.openstack_flavors }}"

# Gather information for the flavor named "xlarge-flavor"
- openstack.cloud.compute_flavor_info:
    cloud: mycloud
    name: "xlarge-flavor"

# Get all flavors that have exactly 512 MB of RAM.
- openstack.cloud.compute_flavor_info:
    cloud: mycloud
    ram: "512"

# Get all flavors that have 1024 MB or more of RAM.
- openstack.cloud.compute_flavor_info:
    cloud: mycloud
    ram: ">=1024"

# Get a single flavor that has the minimum amount of RAM. Using the 'limit'
# option will guarantee only a single flavor is returned.
- openstack.cloud.compute_flavor_info:
    cloud: mycloud
    ram: "MIN"
    limit: 1

# Get all flavors with 1024 MB of RAM or more, AND exactly 2 virtual CPUs.
- openstack.cloud.compute_flavor_info:
    cloud: mycloud
    ram: ">=1024"
    vcpus: "2"

# Get all flavors with 1024 MB of RAM or more, exactly 2 virtual CPUs, and
# less than 30gb of ephemeral storage.
- openstack.cloud.compute_flavor_info:
    cloud: mycloud
    ram: ">=1024"
    vcpus: "2"
    ephemeral: "<30"
'''


RETURN = '''
openstack_flavors:
    description: Dictionary describing the flavors.
    returned: On success.
    type: complex
    contains:
        id:
            description: Flavor ID.
            returned: success
            type: str
            sample: "515256b8-7027-4d73-aa54-4e30a4a4a339"
        name:
            description: Flavor name.
            returned: success
            type: str
            sample: "tiny"
        disk:
            description: Size of local disk, in GB.
            returned: success
            type: int
            sample: 10
        ephemeral:
            description: Ephemeral space size, in GB.
            returned: success
            type: int
            sample: 10
        ram:
            description: Amount of memory, in MB.
            returned: success
            type: int
            sample: 1024
        swap:
            description: Swap space size, in MB.
            returned: success
            type: int
            sample: 100
        vcpus:
            description: Number of virtual CPUs.
            returned: success
            type: int
            sample: 2
        is_public:
            description: Make flavor accessible to the public.
            returned: success
            type: bool
            sample: true
'''


from ansible.module_utils.basic import AnsibleModule
from ansible_collections.openstack.cloud.plugins.module_utils.openstack import (openstack_full_argument_spec,
                                                                                openstack_module_kwargs,
                                                                                openstack_cloud_from_module)


def main():
    argument_spec = openstack_full_argument_spec(
        name=dict(required=False, default=None),
        ram=dict(required=False, default=None),
        vcpus=dict(required=False, default=None),
        limit=dict(required=False, default=None, type='int'),
        ephemeral=dict(required=False, default=None),
    )
    module_kwargs = openstack_module_kwargs(
        mutually_exclusive=[
            ['name', 'ram'],
            ['name', 'vcpus'],
            ['name', 'ephemeral']
        ]
    )
    module = AnsibleModule(argument_spec, **module_kwargs)
    is_old_facts = module._name == 'openstack.cloud.compute_flavor_facts'
    if is_old_facts:
        module.deprecate("The 'openstack.cloud.compute_flavor_facts' module has been renamed to 'openstack.cloud.compute_flavor_info', "
                         "and the renamed one no longer returns ansible_facts", version='2.13')

    name = module.params['name']
    vcpus = module.params['vcpus']
    ram = module.params['ram']
    ephemeral = module.params['ephemeral']
    limit = module.params['limit']

    filters = {}
    if vcpus:
        filters['vcpus'] = vcpus
    if ram:
        filters['ram'] = ram
    if ephemeral:
        filters['ephemeral'] = ephemeral

    sdk, cloud = openstack_cloud_from_module(module)
    try:
        if name:
            flavors = cloud.search_flavors(filters={'name': name})

        else:
            flavors = cloud.list_flavors()
            if filters:
                flavors = cloud.range_search(flavors, filters)

        if limit is not None:
            flavors = flavors[:limit]

        if is_old_facts:
            module.exit_json(changed=False,
                             ansible_facts=dict(openstack_flavors=flavors))
        else:
            module.exit_json(changed=False,
                             openstack_flavors=flavors)

    except sdk.exceptions.OpenStackCloudException as e:
        module.fail_json(msg=str(e))


if __name__ == '__main__':
    main()
