"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TagManager = void 0;
const cfn_resource_1 = require("./cfn-resource");
/**
 * Standard tags are a list of { key, value } objects
 */
class StandardFormatter {
    parseTags(cfnPropertyTags, priority) {
        if (!Array.isArray(cfnPropertyTags)) {
            throw new Error(`Invalid tag input expected array of {key, value} have ${JSON.stringify(cfnPropertyTags)}`);
        }
        const tags = [];
        for (const tag of cfnPropertyTags) {
            if (tag.key === undefined || tag.value === undefined) {
                throw new Error(`Invalid tag input expected {key, value} have ${JSON.stringify(tag)}`);
            }
            // using interp to ensure Token is now string
            tags.push({
                key: `${tag.key}`,
                value: `${tag.value}`,
                priority,
            });
        }
        return tags;
    }
    formatTags(tags) {
        const cfnTags = [];
        for (const tag of tags) {
            cfnTags.push({
                key: tag.key,
                value: tag.value,
            });
        }
        return cfnTags.length === 0 ? undefined : cfnTags;
    }
}
/**
 * ASG tags are a list of { key, value, propagateAtLaunch } objects
 */
class AsgFormatter {
    parseTags(cfnPropertyTags, priority) {
        const tags = [];
        if (!Array.isArray(cfnPropertyTags)) {
            throw new Error(`Invalid tag input expected array of {key, value, propagateAtLaunch} have ${JSON.stringify(cfnPropertyTags)}`);
        }
        for (const tag of cfnPropertyTags) {
            if (tag.key === undefined ||
                tag.value === undefined ||
                tag.propagateAtLaunch === undefined) {
                throw new Error(`Invalid tag input expected {key, value, propagateAtLaunch} have ${JSON.stringify(tag)}`);
            }
            // using interp to ensure Token is now string
            tags.push({
                key: `${tag.key}`,
                value: `${tag.value}`,
                priority,
                applyToLaunchedInstances: !!tag.propagateAtLaunch,
            });
        }
        return tags;
    }
    formatTags(tags) {
        const cfnTags = [];
        for (const tag of tags) {
            cfnTags.push({
                key: tag.key,
                value: tag.value,
                propagateAtLaunch: tag.applyToLaunchedInstances !== false,
            });
        }
        return cfnTags.length === 0 ? undefined : cfnTags;
    }
}
/**
 * Some CloudFormation constructs use a { key: value } map for tags
 */
class MapFormatter {
    parseTags(cfnPropertyTags, priority) {
        const tags = [];
        if (Array.isArray(cfnPropertyTags) || typeof (cfnPropertyTags) !== 'object') {
            throw new Error(`Invalid tag input expected map of {key: value} have ${JSON.stringify(cfnPropertyTags)}`);
        }
        for (const [key, value] of Object.entries(cfnPropertyTags)) {
            tags.push({
                key,
                value: `${value}`,
                priority,
            });
        }
        return tags;
    }
    formatTags(tags) {
        const cfnTags = {};
        for (const tag of tags) {
            cfnTags[`${tag.key}`] = `${tag.value}`;
        }
        return Object.keys(cfnTags).length === 0 ? undefined : cfnTags;
    }
}
/**
 * StackTags are of the format { Key: key, Value: value }
 */
class KeyValueFormatter {
    parseTags(keyValueTags, priority) {
        const tags = [];
        for (const key in keyValueTags) {
            if (keyValueTags.hasOwnProperty(key)) {
                const value = keyValueTags[key];
                tags.push({
                    key,
                    value,
                    priority,
                });
            }
        }
        return tags;
    }
    formatTags(unformattedTags) {
        const tags = [];
        unformattedTags.forEach(tag => {
            tags.push({
                Key: tag.key,
                Value: tag.value,
            });
        });
        return tags;
    }
}
class NoFormat {
    parseTags(_cfnPropertyTags) {
        return [];
    }
    formatTags(_tags) {
        return undefined;
    }
}
const TAG_FORMATTERS = {
    [cfn_resource_1.TagType.AUTOSCALING_GROUP]: new AsgFormatter(),
    [cfn_resource_1.TagType.STANDARD]: new StandardFormatter(),
    [cfn_resource_1.TagType.MAP]: new MapFormatter(),
    [cfn_resource_1.TagType.KEY_VALUE]: new KeyValueFormatter(),
    [cfn_resource_1.TagType.NOT_TAGGABLE]: new NoFormat(),
};
/**
 * TagManager facilitates a common implementation of tagging for Constructs.
 */
class TagManager {
    constructor(tagType, resourceTypeName, tagStructure, options = {}) {
        this.tags = new Map();
        this.priorities = new Map();
        this.initialTagPriority = 50;
        this.resourceTypeName = resourceTypeName;
        this.tagFormatter = TAG_FORMATTERS[tagType];
        if (tagStructure !== undefined) {
            this._setTag(...this.tagFormatter.parseTags(tagStructure, this.initialTagPriority));
        }
        this.tagPropertyName = options.tagPropertyName || 'tags';
    }
    /**
     * Check whether the given construct is Taggable
     */
    static isTaggable(construct) {
        return construct.tags !== undefined;
    }
    /**
     * Adds the specified tag to the array of tags
     *
     */
    setTag(key, value, priority = 0, applyToLaunchedInstances = true) {
        // This method mostly exists because we don't want to expose the 'Tag' type used (it will be confusing
        // to users).
        this._setTag({ key, value, priority, applyToLaunchedInstances });
    }
    /**
     * Removes the specified tag from the array if it exists
     *
     * @param key The tag to remove
     * @param priority The priority of the remove operation
     */
    removeTag(key, priority) {
        if (priority >= (this.priorities.get(key) || 0)) {
            this.tags.delete(key);
            this.priorities.set(key, priority);
        }
    }
    /**
     * Renders tags into the proper format based on TagType
     */
    renderTags() {
        const sortedTags = Array.from(this.tags.values()).sort((a, b) => a.key.localeCompare(b.key));
        return this.tagFormatter.formatTags(sortedTags);
    }
    /**
     * Determine if the aspect applies here
     *
     * Looks at the include and exclude resourceTypeName arrays to determine if
     * the aspect applies here
     */
    applyTagAspectHere(include, exclude) {
        if (exclude && exclude.length > 0 && exclude.indexOf(this.resourceTypeName) !== -1) {
            return false;
        }
        if (include && include.length > 0 && include.indexOf(this.resourceTypeName) === -1) {
            return false;
        }
        return true;
    }
    /**
     * Returns true if there are any tags defined
     */
    hasTags() {
        return this.tags.size > 0;
    }
    _setTag(...tags) {
        for (const tag of tags) {
            if (tag.priority >= (this.priorities.get(tag.key) || 0)) {
                this.tags.set(tag.key, tag);
                this.priorities.set(tag.key, tag.priority);
            }
        }
    }
}
exports.TagManager = TagManager;
//# sourceMappingURL=data:application/json;base64,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