"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.evaluateCFN = void 0;
/**
 * Simple function to evaluate CloudFormation intrinsics.
 *
 * Note that this function is not production quality, it exists to support tests.
 */
const cloudformation_lang_1 = require("../lib/private/cloudformation-lang");
function evaluateCFN(object, context = {}) {
    const intrinsics = {
        'Fn::Join'(separator, args) {
            return args.map(evaluate).join(separator);
        },
        'Ref'(logicalId) {
            if (!(logicalId in context)) {
                throw new Error(`Trying to evaluate Ref of '${logicalId}' but not in context!`);
            }
            return context[logicalId];
        },
        'Fn::GetAtt'(logicalId, attributeName) {
            const key = `${logicalId}.${attributeName}`;
            if (!(key in context)) {
                throw new Error(`Trying to evaluate Fn::GetAtt of '${logicalId}.${attributeName}' but not in context!`);
            }
            return context[key];
        },
        'Fn::Sub'(argument) {
            let template;
            let placeholders;
            if (Array.isArray(argument)) {
                template = argument[0];
                placeholders = evaluate(argument[1]);
            }
            else {
                template = argument;
                placeholders = context;
            }
            if (typeof template !== 'string') {
                throw new Error('The first argument to {Fn::Sub} must be a string literal (cannot be the result of an expression)');
            }
            return template.replace(/\$\{([a-zA-Z0-9.:-]*)\}/g, (_, key) => {
                if (key in placeholders) {
                    return placeholders[key];
                }
                throw new Error(`Unknown placeholder in Fn::Sub: ${key}`);
            });
        },
    };
    return evaluate(object);
    function evaluate(obj) {
        if (Array.isArray(obj)) {
            return obj.map(evaluate);
        }
        if (typeof obj === 'object') {
            const keys = Object.keys(obj);
            if (keys.length === 1 && (cloudformation_lang_1.isNameOfCloudFormationIntrinsic(keys[0]) || keys[0] === 'Ref')) {
                return evaluateIntrinsic(keys[0], obj[keys[0]]);
            }
            const ret = {};
            for (const key of keys) {
                ret[key] = evaluateCFN(obj[key]);
            }
            return ret;
        }
        return obj;
    }
    function evaluateIntrinsic(name, args) {
        if (!(name in intrinsics)) {
            throw new Error(`Intrinsic ${name} not supported here`);
        }
        if (!Array.isArray(args)) {
            args = [args];
        }
        return intrinsics[name].apply(intrinsics, args);
    }
}
exports.evaluateCFN = evaluateCFN;
//# sourceMappingURL=data:application/json;base64,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