"use strict";
const nodeunit = require("nodeunit");
const lib_1 = require("../lib");
function floatEqual(test, actual, expected) {
    test.ok(
    // Floats are subject to rounding errors up to Number.ESPILON
    actual >= expected - Number.EPSILON && actual <= expected + Number.EPSILON, `${actual} == ${expected}`);
}
module.exports = nodeunit.testCase({
    'negative amount'(test) {
        test.throws(() => lib_1.Duration.seconds(-1), /negative/);
        test.done();
    },
    'unresolved amount'(test) {
        const stack = new lib_1.Stack();
        const lazyDuration = lib_1.Duration.seconds(lib_1.Token.asNumber({ resolve: () => 1337 }));
        test.equals(stack.resolve(lazyDuration.toSeconds()), 1337);
        test.throws(() => stack.resolve(lazyDuration.toMinutes()), /Unable to perform time unit conversion on un-resolved token/);
        test.done();
    },
    'Duration in seconds'(test) {
        const duration = lib_1.Duration.seconds(300);
        test.equal(duration.toSeconds(), 300);
        test.equal(duration.toMinutes(), 5);
        test.throws(() => duration.toDays(), /'300 seconds' cannot be converted into a whole number of days/);
        floatEqual(test, duration.toDays({ integral: false }), 300 / 86400);
        test.equal(lib_1.Duration.seconds(60 * 60 * 24).toDays(), 1);
        test.done();
    },
    'Duration in minutes'(test) {
        const duration = lib_1.Duration.minutes(5);
        test.equal(duration.toSeconds(), 300);
        test.equal(duration.toMinutes(), 5);
        test.throws(() => duration.toDays(), /'5 minutes' cannot be converted into a whole number of days/);
        floatEqual(test, duration.toDays({ integral: false }), 300 / 86400);
        test.equal(lib_1.Duration.minutes(60 * 24).toDays(), 1);
        test.done();
    },
    'Duration in hours'(test) {
        const duration = lib_1.Duration.hours(5);
        test.equal(duration.toSeconds(), 18000);
        test.equal(duration.toMinutes(), 300);
        test.throws(() => duration.toDays(), /'5 hours' cannot be converted into a whole number of days/);
        floatEqual(test, duration.toDays({ integral: false }), 5 / 24);
        test.equal(lib_1.Duration.hours(24).toDays(), 1);
        test.done();
    },
    'seconds to milliseconds'(test) {
        const duration = lib_1.Duration.seconds(5);
        test.equal(duration.toMilliseconds(), 5000);
        test.done();
    },
    'Duration in days'(test) {
        const duration = lib_1.Duration.days(1);
        test.equal(duration.toSeconds(), 86400);
        test.equal(duration.toMinutes(), 1440);
        test.equal(duration.toDays(), 1);
        test.done();
    },
    'toISOString'(test) {
        test.equal(lib_1.Duration.millis(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.seconds(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.minutes(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.hours(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.days(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.millis(5).toISOString(), 'PT0.005S');
        test.equal(lib_1.Duration.seconds(5).toISOString(), 'PT5S');
        test.equal(lib_1.Duration.minutes(5).toISOString(), 'PT5M');
        test.equal(lib_1.Duration.hours(5).toISOString(), 'PT5H');
        test.equal(lib_1.Duration.days(5).toISOString(), 'PT5D');
        test.equal(lib_1.Duration.seconds(1 + 60 * (1 + 60 * (1 + 24))).toISOString(), 'PT1D1H1M1S');
        test.done();
    },
    'toIsoString'(test) {
        test.equal(lib_1.Duration.millis(0).toIsoString(), 'PT0S');
        test.equal(lib_1.Duration.seconds(0).toIsoString(), 'PT0S');
        test.equal(lib_1.Duration.minutes(0).toIsoString(), 'PT0S');
        test.equal(lib_1.Duration.hours(0).toIsoString(), 'PT0S');
        test.equal(lib_1.Duration.days(0).toIsoString(), 'PT0S');
        test.equal(lib_1.Duration.millis(5).toIsoString(), 'PT0.005S');
        test.equal(lib_1.Duration.seconds(5).toIsoString(), 'PT5S');
        test.equal(lib_1.Duration.minutes(5).toIsoString(), 'PT5M');
        test.equal(lib_1.Duration.hours(5).toIsoString(), 'PT5H');
        test.equal(lib_1.Duration.days(5).toIsoString(), 'PT5D');
        test.equal(lib_1.Duration.seconds(1 + 60 * (1 + 60 * (1 + 24))).toIsoString(), 'PT1D1H1M1S');
        test.done();
    },
    'parse'(test) {
        test.equal(lib_1.Duration.parse('PT0S').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT0M').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT0H').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT0D').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT5S').toSeconds(), 5);
        test.equal(lib_1.Duration.parse('PT5M').toSeconds(), 300);
        test.equal(lib_1.Duration.parse('PT5H').toSeconds(), 18000);
        test.equal(lib_1.Duration.parse('PT5D').toSeconds(), 432000);
        test.equal(lib_1.Duration.parse('PT1D1H1M1S').toSeconds(), 1 + 60 * (1 + 60 * (1 + 24)));
        test.done();
    },
    'to human string'(test) {
        test.equal(lib_1.Duration.minutes(0).toHumanString(), '0 minutes');
        test.equal(lib_1.Duration.minutes(lib_1.Lazy.numberValue({ produce: () => 5 })).toHumanString(), '<token> minutes');
        test.equal(lib_1.Duration.minutes(10).toHumanString(), '10 minutes');
        test.equal(lib_1.Duration.minutes(1).toHumanString(), '1 minute');
        test.equal(lib_1.Duration.minutes(62).toHumanString(), '1 hour 2 minutes');
        test.equal(lib_1.Duration.seconds(3666).toHumanString(), '1 hour 1 minute');
        test.equal(lib_1.Duration.millis(3000).toHumanString(), '3 seconds');
        test.equal(lib_1.Duration.millis(3666).toHumanString(), '3 seconds 666 millis');
        test.equal(lib_1.Duration.millis(3.6).toHumanString(), '3.6 millis');
        test.done();
    },
    'add two durations'(test) {
        test.equal(lib_1.Duration.minutes(1).plus(lib_1.Duration.seconds(30)).toSeconds(), lib_1.Duration.seconds(90).toSeconds());
        test.equal(lib_1.Duration.minutes(1).plus(lib_1.Duration.seconds(30)).toMinutes({ integral: false }), lib_1.Duration.seconds(90).toMinutes({ integral: false }));
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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