"use strict";
const lib_1 = require("../lib");
const util_1 = require("./util");
const template = {
    Parameters: {
        MyParam: {
            Type: 'String',
            Default: 'Hello',
        },
    },
    Resources: {
        MyResource1: {
            Type: 'ResourceType1',
            Properties: {
                P1: 1,
                P2: 2,
            },
        },
        MyResource2: {
            Type: 'ResourceType2',
        },
    },
};
/**
 * @param obj an object to clone
 * @returns a deep clone of ``obj`.
 */
function clone(obj) {
    switch (typeof obj) {
        case 'object':
            if (Array.isArray(obj)) {
                return obj.map(elt => clone(elt));
            }
            else {
                const cloned = {};
                for (const key of Object.keys(obj)) {
                    cloned[key] = clone(obj[key]);
                }
                return cloned;
            }
        default:
            return obj;
    }
}
module.exports = {
    'the Include construct can be used to embed an existing template as-is into a stack'(test) {
        const stack = new lib_1.Stack();
        new lib_1.CfnInclude(stack, 'T1', { template: clone(template) });
        test.deepEqual(util_1.toCloudFormation(stack), {
            Parameters: { MyParam: { Type: 'String', Default: 'Hello' } },
            Resources: {
                MyResource1: { Type: 'ResourceType1', Properties: { P1: 1, P2: 2 } },
                MyResource2: { Type: 'ResourceType2' }
            }
        });
        test.done();
    },
    'included templates can co-exist with elements created programmatically'(test) {
        const stack = new lib_1.Stack();
        new lib_1.CfnInclude(stack, 'T1', { template: clone(template) });
        new lib_1.CfnResource(stack, 'MyResource3', { type: 'ResourceType3', properties: { P3: 'Hello' } });
        new lib_1.CfnOutput(stack, 'MyOutput', { description: 'Out!', value: 'hey' });
        new lib_1.CfnParameter(stack, 'MyParam2', { type: 'Integer' });
        test.deepEqual(util_1.toCloudFormation(stack), {
            Parameters: {
                MyParam: { Type: 'String', Default: 'Hello' },
                MyParam2: { Type: 'Integer' }
            },
            Resources: {
                MyResource1: { Type: 'ResourceType1', Properties: { P1: 1, P2: 2 } },
                MyResource2: { Type: 'ResourceType2' },
                MyResource3: { Type: 'ResourceType3', Properties: { P3: 'Hello' } }
            },
            Outputs: {
                MyOutput: { Description: 'Out!', Value: 'hey' }
            }
        });
        test.done();
    },
    'exception is thrown in construction if an entity from an included template has the same id as a programmatic entity'(test) {
        const stack = new lib_1.Stack();
        new lib_1.CfnInclude(stack, 'T1', { template });
        new lib_1.CfnResource(stack, 'MyResource3', { type: 'ResourceType3', properties: { P3: 'Hello' } });
        new lib_1.CfnOutput(stack, 'MyOutput', { description: 'Out!', value: 'in' });
        new lib_1.CfnParameter(stack, 'MyParam', { type: 'Integer' }); // duplicate!
        test.throws(() => util_1.toCloudFormation(stack));
        test.done();
    },
    'correctly merges template sections that contain strings'(test) {
        const stack = new lib_1.Stack();
        new lib_1.CfnInclude(stack, 'T1', {
            template: {
                AWSTemplateFormatVersion: '2010-09-09',
                Description: 'Test 1',
            },
        });
        new lib_1.CfnInclude(stack, 'T2', {
            template: {
                AWSTemplateFormatVersion: '2010-09-09',
                Description: 'Test 2',
            },
        });
        test.deepEqual(util_1.toCloudFormation(stack), {
            AWSTemplateFormatVersion: '2010-09-09',
            Description: 'Test 1\nTest 2',
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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