"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCdkResolver = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const child_process_1 = require("child_process");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
class AwsCdkResolver {
    resolve(context) {
        if (!aws_cdk_lib_1.Token.isUnresolved(context.value)) {
            return;
        }
        if (typeof context.value !== 'string') {
            // should be ok because we only resolve CfnOutput values, which
            // must be strings.
            throw new Error(`Invalid value type: ${typeof (context.value)} (Expected 'string')`);
        }
        const output = this.findOutput(context.value);
        try {
            const outputValue = this.fetchOutputValue(output);
            context.replaceValue(outputValue);
        }
        catch (err) {
            // if both cdk8s and AWS CDK applications are defined within the same file,
            // a cdk8s synth is going to happen before the AWS CDK deployment.
            // in this case we must swallow the error, otherwise the AWS CDK deployment
            // won't be able to go through. we replace the value with something to indicate
            // that a fetching attempt was made and failed.
            context.replaceValue(`Failed fetching value for output ${output.node.path}: ${err}`);
        }
    }
    findOutput(value) {
        const inspectedStacks = [];
        for (const token of aws_cdk_lib_1.Tokenization.reverseString(value).tokens) {
            if (aws_cdk_lib_1.Reference.isReference(token)) {
                const stack = aws_cdk_lib_1.Stack.of(token.target);
                inspectedStacks.push(stack);
                const output = stack.node.findAll().filter(c => c instanceof aws_cdk_lib_1.CfnOutput && c.value === value)[0];
                // we don't really care if there are more outputs (possibly from different stacks)
                // that point to the same value. the first will suffice.
                if (output)
                    return output;
            }
        }
        // This can happen if either:
        // --------------------------
        //  1. User didn't define an output.
        //  2. Output was defined in a different stack than the tokens comprising its value.
        //  3. None of the tokens comprising the value are a Reference.
        throw new Error(`Unable to find output defined for ${value} (Inspected stacks: ${inspectedStacks.map(s => s.stackName).join(',')})`);
    }
    fetchOutputValue(output) {
        const script = path.join(__dirname, '..', 'lib', 'fetch-output-value.js');
        return JSON.parse((0, child_process_1.execFileSync)(process.execPath, [
            script,
            aws_cdk_lib_1.Stack.of(output).stackName,
            output.node.id,
        ], { encoding: 'utf-8', stdio: ['pipe'] }).toString().trim());
    }
}
exports.AwsCdkResolver = AwsCdkResolver;
_a = JSII_RTTI_SYMBOL_1;
AwsCdkResolver[_a] = { fqn: "@cdk8s/awscdk-resolver.AwsCdkResolver", version: "0.0.350" };
//# sourceMappingURL=data:application/json;base64,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