import os
import random
import unittest

from datetime import datetime, timedelta, timezone
from grid.utilities import (get_abs_time_difference, string_format_timedelta,
                            check_environment_variables, introspect_module,
                            convert_utc_string_to_local_datetime,
                            is_experiment, get_param_values,
                            get_experiment_duration_string, get_graphql_url)
from dateutil.parser import parse as date_string_parse


class UtilitiesTestCase(unittest.TestCase):
    """Test utilities in credentials."""
    def test_convert_utc_string_to_local_datetime(self):
        date_string = "2016-08-29T16:02:54.884Z"
        date_date = convert_utc_string_to_local_datetime(date_string)
        assert isinstance(date_date, datetime)

    def test_get_abs_time_difference(self):
        date_time_now = datetime.now()
        date_a = date_time_now - timedelta(hours=1)
        date_b = date_time_now
        result = get_abs_time_difference(date_1=date_a, date_2=date_b)
        inverse_result = get_abs_time_difference(date_1=date_b, date_2=date_a)
        assert isinstance(result, timedelta)
        assert int(result.total_seconds()) == 60 * 60
        assert int(inverse_result.total_seconds()) == 60 * 60

    def test_conversion_with_time_difference_now(self):
        finished_at = datetime.now(timezone.utc)
        created_at = date_string_parse('2020-05-16T21:21:10.745898+00:00')
        delta = get_abs_time_difference(finished_at, created_at)
        assert isinstance(delta, timedelta)

    def test_string_formating_duration(self):
        a = datetime.now()
        b = datetime.now() - timedelta(hours=1)
        c = a - b

        result = string_format_timedelta(c)
        assert result == '0d-00:59:59'

    def test_environment_variables(self):
        var = '_TEST_VAR'

        #  Checks that the env var isn't set.
        assert not os.getenv(var)

        #  Tests that function raises error.
        with self.assertRaises(ValueError):
            check_environment_variables(variables=[var])

        #  Tests that it doesn't raise an error
        #  if variable is set.
        os.environ[var] = 'foo'
        assert check_environment_variables(variables=[var])

    def test_introspect_module(self):
        #  Creates a class that represents the same
        #  module mechanics: it includes the __all__
        #  attribute that has a string reference to
        #  a class property (test).
        class M:
            @staticmethod
            def test():
                return True

            __all__ = ['test']

        #  Let's that the generator returns
        #  the defined method.
        for func in introspect_module(M):
            assert isinstance(func, staticmethod)

    def test_is_experiment(self):
        run_a = 'test-run-1'
        result_a = is_experiment(run_a)
        assert not result_a

        run_b = 'test-run-1-exp1'
        result_b = is_experiment(run_b)
        assert result_b

        run_c = 'test-run-1-exp1-foo'
        result_c = is_experiment(run_c)
        assert not result_c

    def test_is_experiment_many_ints(self):
        """is_experiment() returns true if experiment has many int digits."""

        for _ in range(10):
            digit = random.randint(10, 10000)
            run = f'test-run-1-exp{digit}'
            assert is_experiment(run)

    def test_get_param_vals(self):
        command = "command.py --flag1 val1 --flag2 --flag3 val3 --flag4"
        assert get_param_values(command) == ['val1', "True", "val3", "True"]

    def test_get_experiment_duration_string(self):
        created_at = str(datetime.now(timezone.utc) - timedelta(hours=3))
        started_running_at = str(
            datetime.now(timezone.utc) - timedelta(hours=2))
        finished_at = str(datetime.now(timezone.utc) - timedelta(hours=1))
        experiment_queued_result = get_experiment_duration_string(
            created_at=created_at, started_running_at=None, finished_at=None)
        assert experiment_queued_result >= f"{0}d-{3:02d}:{0:02d}:{0:02d}"
        assert experiment_queued_result < f"{0}d-{3:02d}:{1:02d}:{0:02d}"
        experiment_running_result = get_experiment_duration_string(
            created_at=created_at,
            started_running_at=started_running_at,
            finished_at=None)
        assert experiment_running_result >= f"{0}d-{2:02d}:{0:02d}:{0:02d}"
        assert experiment_running_result < f"{0}d-{2:02d}:{1:02d}:{0:02d}"
        experiment_finished_result = get_experiment_duration_string(
            created_at=created_at,
            started_running_at=started_running_at,
            finished_at=finished_at)
        assert experiment_finished_result >= f"{0}d-{1:02d}:{0:02d}:{0:02d}"
        assert experiment_finished_result < f"{0}d-{1:02d}:{1:02d}:{0:02d}"

    @staticmethod
    def test_get_graphql_url():
        url_with_graphql = 'http://localhost:8000/graphql'
        url_without_graphql = 'http://localhost:8000'
        assert get_graphql_url(url_without_graphql) == 'http://localhost:8000/graphql'
        assert get_graphql_url(url_with_graphql) == 'http://localhost:8000/graphql'
