# pylint: disable=missing-module-docstring,missing-class-docstring,missing-function-docstring
import numpy as np
from PySDM_examples.Lowe_et_al_2019 import aerosol
from PySDM import Formulae
from PySDM.physics import si, constants_defaults as const
from .constants import constants

assert hasattr(constants, '_pytestfixturefunction')

TRIVIA = Formulae().trivia
R_WET = np.logspace(np.log(150 * si.nm), np.log(3000 * si.nm), base=np.e, num=100)
R_DRY = 50 * si.nm
V_WET = TRIVIA.volume(R_WET)
V_DRY = TRIVIA.volume(R_DRY)
TEMPERATURE = 300 * si.K
aer = aerosol.AerosolBoreal()

class TestFig1:
    @staticmethod
    def test_bulk_surface_tension_is_sgm_w():
        # arrange
        formulae = Formulae(surface_tension='Constant')

        # act
        sigma = formulae.surface_tension.sigma(np.nan, V_WET, np.nan, np.nan)

        # assert
        assert sigma == const.sgm_w

    @staticmethod
    def test_ovad_surface_tension():
        # arrange
        formulae = Formulae(
            surface_tension='CompressedFilmOvadnevaite',
            constants={
                'sgm_org': 40 * si.mN / si.m,
                'delta_min': 0.1 * si.nm
            }
        )

        # act
        sigma = formulae.surface_tension.sigma.py_func(
            np.nan,
            V_WET,
            V_DRY,
            aer.aerosol_modes_per_cc[0]['f_org']
        )

        # assert
        test = np.array([
                    0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,
                    0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,
                    0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,0.04,
                    0.04,0.04,0.04,0.04,0.04,0.04145185206190185,0.043245940200403225,
                    0.04493465757799464,0.04652419321971498,0.04802037260212956,
                    0.049428679011056784,0.05075427364458418,0.05200201453477719,
                    0.053176474357604905,0.054281957196414696,0.055322514320391286,
                    0.056301959036044666,0.05722388066615342,0.05809165770721892,
                    0.05890847021403533,0.059677311456651734,0.06040099889241329,
                    0.06108218449344455,0.061723364467315724,0.06232688840684634,
                    0.06289496790207844,0.06342968464658641,0.06393299806742464,0.06440675250688739,
                    0.06485268398238773,0.06527242654921248,0.06566751828951929,0.06603940694949924,
                    0.06638945524541608,0.06671894585794522,0.06702908613316425,0.06732101250732575,
                    0.0675957946718376,0.06785443949346634,0.06809789470435344,0.06832705237523176,
                    0.06854275218466818,0.06874578449624996,0.06893689325503628,0.06911677871385602,
                    0.06928609999950004,0.06944547752817445,0.06959549527907691,0.06973670293439409,
                    0.06986961789368958,0.06999472716988156,0.0701124891739247,0.07022333539465174,
                    0.07032767197991878,0.07042588122494463,0.07051832297317082,0.07060533593488776,
                    0.0706872389283801,0.07076433204821643,0.07083689776487583,0.0709052019598351
                ])
        assert np.allclose(sigma, test, atol=1e-8)

    @staticmethod
    def test_ruehl_surface_tension():
        # arrange
        formulae = Formulae(
            surface_tension='CompressedFilmRuehl',
            constants={
                'RUEHL_nu_org': aer.aerosol_modes_per_cc[0]['nu_org'],
                'RUEHL_A0': 115e-20 * si.m * si.m,
                'RUEHL_C0': 6e-7,
                'RUEHL_m_sigma': 0.3e17 * si.J / si.m**2,
                'RUEHL_sgm_min': 40.0 * si.mN / si.m
            }
        )

        # act
        sigma = np.zeros(len(V_WET))
        for i,vw in enumerate(V_WET):
            sigma[i] = formulae.surface_tension.sigma.py_func(
                TEMPERATURE,
                vw,
                V_DRY,
                aer.aerosol_modes_per_cc[0]['f_org']
            )

        # assert
        test = np.array([
            0.04318990388863848,0.04355416769160859,0.0439421778272239,0.04435543913070772,
            0.04479551672881958,0.04526402672640544,0.045762623781862244,0.04629298561514597,
            0.04685679492085086,0.04745571976463471,0.048091394314671604,0.04876540264111237,
            0.04947926918577529,0.05023446017823065,0.05103240053506651,0.051874510426880245,
            0.052762264625474335,0.05369727599685779,0.054681402290397516,0.05571687298813886,
            0.056806430668716615,0.057953479146781645,0.05916222830454722,0.06043782250688807,
            0.06178643524956695,0.06321530723603938,0.06473269985853751,0.06634773509867063,
            0.0680701033701799,0.06990965125952682,0.07187591399036536,0.072,0.072,0.072,0.072,
            0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,
            0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,
            0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,
            0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,
            0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072,0.072
        ])
        assert np.allclose(sigma, test, atol=1e-8)

    @staticmethod
    def test_SL_surface_tension():
        # arrange
        formulae = Formulae(
            surface_tension='SzyszkowskiLangmuir',
            constants={
                'RUEHL_nu_org': aer.aerosol_modes_per_cc[0]['nu_org'],
                'RUEHL_A0': 115e-20 * si.m * si.m,
                'RUEHL_C0': 6e-7,
                'RUEHL_sgm_min': 40.0 * si.mN / si.m
            }
        )

        # act
        sigma = formulae.surface_tension.sigma.py_func(
            TEMPERATURE,
            V_WET,
            V_DRY,
            aer.aerosol_modes_per_cc[0]['f_org']
        )

        # assert
        test = np.array([
            0.04170545235068712,0.04207589632089122,0.04244962961646187,0.0428269485292672,
            0.04320818372912586,0.043593705564928045,0.04398393042681806,0.044379328434762944,
            0.04478043279935583,0.0451878513103144,0.04560228055915656,0.04602452371335611,
            0.046455512957773316,0.046896338148322336,0.047348283850285854,0.04781287786790929,
            0.048291955790905984,0.04878774828646771,0.049303001368475236,0.04984114559483559,
            0.05040653975448352,0.05100483126948219,0.05164350541293351,0.05233275085500362,
            0.0530868749743816,0.05392670762306413,0.05488381111799955,0.056007799064006464,
            0.0573767511936752,0.059091334616161374,0.06113470871082447,0.0630992101730827,
            0.06461657500220883,0.06572817193513107,0.06656632645301724,0.067223028931838,
            0.06775426559459131,0.0681947966802138,0.06856720046770072,0.06888681108136241,
            0.06916445633257737,0.06940803880572587,0.06962348937406575,0.06981536583363962,
            0.06998724243835744,0.07014197148608561,0.07028186391110573,0.07040881703533021,
            0.07052440690758358,0.0706299563370761,0.07072658588354233,0.07081525266495645,
            0.07089678030539454,0.0709718823377164,0.07104118070169195,0.07110522051890375,
            0.07116448200738978,0.07121939017482383,0.07127032276888652,0.07131761684754595,
            0.07136157424698915,0.07140246616194773,0.07144053700595183,0.07147600768332182,
            0.0715090783774165,0.07153993093863077,0.07156873093930102,0.07159562944989181,
            0.07162076458075536,0.07164426282575462,0.07166624023764522,0.07168680345997547,
            0.07170605063610594,0.0717240722125759,0.07174095165128262,0.071756766062675,
            0.07177158677029272,0.0717854798154341,0.07179850640944603,0.07181072334005235,
            0.0718221833372332,0.07183293540340689,0.07184302511202303,0.07185249487812974,
            0.07186138420401451,0.07186972990262169,0.07187756630111092,0.0718849254266297,
            0.07189183717612246,0.07189832947178237,0.07190442840356441,0.07191015836001612,
            0.07191554214854026,0.07192060110608092,0.07192535520111608,0.07192982312774573,
            0.0719340223925809,0.07193796939506661,0.07194167950180663,0.07194516711540175
        ])
        assert np.allclose(sigma, test, atol=1e-8)
