import numpy as np

import materia as mtr

from .dataseries import SPDSpectrum
from .cie_cmfs import (
    CIE1931ColorMatchingFunctionX,
    CIE1931ColorMatchingFunctionY,
    CIE1931ColorMatchingFunctionZ,
)


class ASTMG173(SPDSpectrum):
    def __init__(self):
        # wavelengths in nms, global tilt spectrum in watts per square-meter per nm
        # taken directly from https://rredc.nrel.gov/solar//spectra/am1.5/ASTMG173/ASTMG173.html, accessed at 16:52 EST 17 Feb. 2019
        x_unit = mtr.nm
        y_unit = mtr.watt / (mtr.m ** 2 * mtr.nm)

        wavs = np.hstack(
            [
                np.linspace(280, 400, 241),
                np.linspace(401, 1700, 1300),
                1702,
                np.linspace(1705, 4000, 460),
            ]
        )

        vals = [
            4.7309e-23,
            1.2307e-21,
            5.6895e-21,
            1.5662e-19,
            1.1946e-18,
            4.5436e-18,
            1.8452e-17,
            3.536e-17,
            7.267e-16,
            2.4856e-15,
            8.0142e-15,
            4.2613e-14,
            1.3684e-13,
            8.3823e-13,
            2.7367e-12,
            1.0903e-11,
            6.2337e-11,
            1.7162e-10,
            5.6265e-10,
            2.0749e-09,
            6.0168e-09,
            1.3783e-08,
            3.5052e-08,
            1.0913e-07,
            2.683e-07,
            4.2685e-07,
            8.6466e-07,
            2.2707e-06,
            4.1744e-06,
            6.5911e-06,
            1.229e-05,
            2.7826e-05,
            4.7904e-05,
            7.1345e-05,
            9.68e-05,
            0.00018608,
            0.00028988,
            0.00035789,
            0.00049211,
            0.00086068,
            0.0010205,
            0.001245,
            0.00193,
            0.0026914,
            0.0029209,
            0.004284,
            0.0070945,
            0.0089795,
            0.0094701,
            0.011953,
            0.016463,
            0.018719,
            0.018577,
            0.021108,
            0.027849,
            0.035635,
            0.037837,
            0.04143,
            0.040534,
            0.043306,
            0.050939,
            0.06554,
            0.082922,
            0.08408,
            0.093376,
            0.098984,
            0.10733,
            0.10757,
            0.11969,
            0.1306,
            0.13625,
            0.11838,
            0.12348,
            0.15036,
            0.17158,
            0.18245,
            0.17594,
            0.18591,
            0.2047,
            0.19589,
            0.20527,
            0.24525,
            0.25024,
            0.23843,
            0.22203,
            0.21709,
            0.21226,
            0.24861,
            0.27537,
            0.28321,
            0.27894,
            0.32436,
            0.3812,
            0.40722,
            0.39806,
            0.38465,
            0.35116,
            0.37164,
            0.42235,
            0.46878,
            0.47139,
            0.428,
            0.40262,
            0.41806,
            0.43623,
            0.43919,
            0.42944,
            0.40724,
            0.41497,
            0.44509,
            0.46388,
            0.45313,
            0.41519,
            0.38214,
            0.3738,
            0.40051,
            0.43411,
            0.45527,
            0.46355,
            0.47446,
            0.5018,
            0.50071,
            0.47139,
            0.46935,
            0.48934,
            0.50767,
            0.51489,
            0.48609,
            0.41843,
            0.40307,
            0.45898,
            0.48932,
            0.47778,
            0.48657,
            0.49404,
            0.47674,
            0.47511,
            0.48336,
            0.46564,
            0.47805,
            0.52798,
            0.56741,
            0.55172,
            0.53022,
            0.51791,
            0.48962,
            0.5204,
            0.57228,
            0.60498,
            0.61156,
            0.6114,
            0.59028,
            0.55387,
            0.51942,
            0.45673,
            0.46215,
            0.43006,
            0.39926,
            0.46953,
            0.56549,
            0.59817,
            0.56531,
            0.52024,
            0.50956,
            0.5342,
            0.5851,
            0.60191,
            0.58541,
            0.60628,
            0.60058,
            0.62359,
            0.68628,
            0.73532,
            0.73658,
            0.72285,
            0.70914,
            0.66759,
            0.6631,
            0.69315,
            0.74469,
            0.75507,
            0.68261,
            0.69338,
            0.72051,
            0.67444,
            0.64253,
            0.61886,
            0.55786,
            0.5564,
            0.55227,
            0.5893,
            0.65162,
            0.6748,
            0.6639,
            0.71225,
            0.79455,
            0.85595,
            0.83418,
            0.74389,
            0.66683,
            0.70077,
            0.75075,
            0.76383,
            0.68837,
            0.58678,
            0.50762,
            0.45499,
            0.44049,
            0.50968,
            0.61359,
            0.67355,
            0.64363,
            0.621,
            0.6457,
            0.65147,
            0.64204,
            0.63582,
            0.63136,
            0.68543,
            0.7597,
            0.79699,
            0.80371,
            0.85138,
            0.86344,
            0.79493,
            0.66257,
            0.47975,
            0.38152,
            0.49567,
            0.68385,
            0.80772,
            0.86038,
            0.75655,
            0.55017,
            0.42619,
            0.62945,
            0.85249,
            1.0069,
            1.0693,
            1.1021,
            1.1141,
            1.1603,
            1.2061,
            1.1613,
            1.1801,
            1.1511,
            1.1227,
            1.1026,
            1.1514,
            1.2299,
            1.0485,
            1.1738,
            1.2478,
            1.1971,
            1.1842,
            1.2258,
            1.2624,
            1.2312,
            1.1777,
            1.2258,
            1.1232,
            1.2757,
            1.2583,
            1.2184,
            1.2117,
            1.2488,
            1.2135,
            1.1724,
            1.1839,
            1.0963,
            0.87462,
            0.79394,
            1.3207,
            1.2288,
            1.1352,
            1.2452,
            1.3659,
            1.3943,
            1.2238,
            1.1775,
            1.3499,
            1.3313,
            1.425,
            1.4453,
            1.4084,
            1.4619,
            1.3108,
            1.4903,
            1.5081,
            1.5045,
            1.5595,
            1.6173,
            1.5482,
            1.4297,
            1.5335,
            1.5224,
            1.5724,
            1.5854,
            1.5514,
            1.5391,
            1.5291,
            1.5827,
            1.5975,
            1.6031,
            1.5544,
            1.535,
            1.5673,
            1.4973,
            1.5619,
            1.5682,
            1.5077,
            1.5331,
            1.6126,
            1.5499,
            1.5671,
            1.6185,
            1.5631,
            1.5724,
            1.623,
            1.5916,
            1.6181,
            1.6177,
            1.6236,
            1.6038,
            1.5734,
            1.5683,
            1.2716,
            1.4241,
            1.5413,
            1.4519,
            1.6224,
            1.5595,
            1.4869,
            1.5903,
            1.5525,
            1.6485,
            1.5676,
            1.5944,
            1.5509,
            1.5507,
            1.5451,
            1.4978,
            1.4966,
            1.5653,
            1.4587,
            1.5635,
            1.6264,
            1.556,
            1.5165,
            1.5893,
            1.5481,
            1.5769,
            1.6186,
            1.5206,
            1.4885,
            1.5314,
            1.5455,
            1.2594,
            1.4403,
            1.3957,
            1.5236,
            1.5346,
            1.569,
            1.4789,
            1.5905,
            1.5781,
            1.5341,
            1.3417,
            1.5357,
            1.6071,
            1.5446,
            1.6292,
            1.5998,
            1.4286,
            1.5302,
            1.5535,
            1.6199,
            1.4989,
            1.5738,
            1.5352,
            1.4825,
            1.4251,
            1.5511,
            1.5256,
            1.5792,
            1.5435,
            1.5291,
            1.549,
            1.5049,
            1.552,
            1.5399,
            1.5382,
            1.5697,
            1.525,
            1.5549,
            1.5634,
            1.5366,
            1.4988,
            1.531,
            1.4483,
            1.474,
            1.5595,
            1.4847,
            1.5408,
            1.5106,
            1.5201,
            1.4374,
            1.532,
            1.518,
            1.4807,
            1.4816,
            1.4331,
            1.5134,
            1.5198,
            1.5119,
            1.4777,
            1.4654,
            1.5023,
            1.456,
            1.477,
            1.502,
            1.5089,
            1.532,
            1.5479,
            1.5448,
            1.5324,
            1.4953,
            1.5281,
            1.4934,
            1.2894,
            1.3709,
            1.4662,
            1.4354,
            1.4561,
            1.4491,
            1.4308,
            1.4745,
            1.4788,
            1.4607,
            1.4606,
            1.4753,
            1.4579,
            1.436,
            1.4664,
            1.4921,
            1.4895,
            1.4822,
            1.4911,
            1.4862,
            1.4749,
            1.4686,
            1.4611,
            1.4831,
            1.4621,
            1.4176,
            1.4697,
            1.431,
            1.4128,
            1.4664,
            1.4733,
            1.4739,
            1.4802,
            1.4269,
            1.4165,
            1.4118,
            1.4026,
            1.4012,
            1.4417,
            1.3631,
            1.4114,
            1.3924,
            1.4161,
            1.3638,
            1.4508,
            1.4284,
            1.4458,
            1.4128,
            1.461,
            1.4707,
            1.4646,
            1.434,
            1.4348,
            1.4376,
            1.4525,
            1.4462,
            1.4567,
            1.415,
            1.4086,
            1.3952,
            1.3519,
            1.3594,
            1.4447,
            1.3871,
            1.4311,
            1.4153,
            1.3499,
            1.1851,
            1.2393,
            1.3855,
            1.3905,
            1.3992,
            1.3933,
            1.3819,
            1.3844,
            1.3967,
            1.4214,
            1.4203,
            1.4102,
            1.415,
            1.4394,
            1.4196,
            1.4169,
            1.3972,
            1.4094,
            1.4074,
            1.3958,
            1.412,
            1.3991,
            1.4066,
            1.3947,
            1.3969,
            1.3915,
            1.3981,
            1.383,
            1.3739,
            1.3748,
            1.3438,
            0.96824,
            1.1206,
            1.1278,
            1.1821,
            1.2333,
            1.2689,
            1.2609,
            1.2464,
            1.2714,
            1.2684,
            1.3403,
            1.3192,
            1.2918,
            1.2823,
            1.2659,
            1.2674,
            1.2747,
            1.3078,
            1.3214,
            1.3144,
            1.309,
            1.3048,
            1.3095,
            1.3175,
            1.3155,
            1.3071,
            1.2918,
            1.3029,
            1.2587,
            1.2716,
            1.1071,
            1.0296,
            0.92318,
            0.9855,
            1.0861,
            1.2407,
            1.1444,
            1.0555,
            1.038,
            1.0813,
            1.085,
            1.04,
            1.0466,
            1.1285,
            1.0703,
            1.1534,
            1.1962,
            1.2357,
            1.2178,
            1.2059,
            1.2039,
            1.2269,
            1.1905,
            1.2195,
            1.2148,
            1.2153,
            1.2405,
            1.2503,
            1.2497,
            1.247,
            1.2477,
            1.2401,
            1.2357,
            1.2341,
            1.2286,
            1.233,
            1.2266,
            1.242,
            1.2383,
            1.2232,
            1.2221,
            1.2295,
            1.1945,
            0.26604,
            0.15396,
            0.68766,
            0.37952,
            0.53878,
            0.68601,
            0.81461,
            0.97417,
            1.1138,
            1.1278,
            1.1608,
            1.1686,
            1.1778,
            1.1771,
            1.1771,
            1.1771,
            1.1798,
            1.1727,
            1.1713,
            1.1765,
            1.1636,
            1.1607,
            1.1662,
            1.1614,
            1.1536,
            1.1586,
            1.1592,
            1.145,
            1.1305,
            1.1257,
            1.091,
            1.1058,
            1.0953,
            1.0875,
            1.0972,
            1.0932,
            1.0742,
            1.0913,
            1.1121,
            1.0905,
            1.0725,
            1.0843,
            1.0856,
            1.0657,
            1.0782,
            1.0545,
            1.0974,
            1.0859,
            1.0821,
            1.0548,
            1.0559,
            1.0533,
            1.0268,
            1.0086,
            0.90356,
            0.89523,
            0.83216,
            0.85183,
            0.82259,
            0.90519,
            0.86188,
            0.99764,
            0.95157,
            0.67271,
            0.93506,
            0.96935,
            0.93381,
            0.98465,
            0.84979,
            0.9293,
            0.91601,
            0.92392,
            0.89426,
            0.9565,
            0.93412,
            1.0032,
            0.97234,
            1.0092,
            0.99901,
            1.0013,
            1.0157,
            1.0101,
            0.99703,
            1.0053,
            0.98631,
            1.0165,
            1.0187,
            0.9917,
            0.99217,
            0.98596,
            0.89372,
            0.97493,
            0.96927,
            0.96486,
            0.85112,
            0.913,
            0.97317,
            0.99166,
            0.99196,
            0.99171,
            0.98816,
            0.98679,
            0.99449,
            1.0005,
            0.97916,
            0.96324,
            0.849,
            0.91546,
            0.9592,
            0.94956,
            0.96755,
            0.95387,
            0.96686,
            0.95721,
            0.94042,
            0.92687,
            0.95277,
            0.95615,
            0.95237,
            0.93656,
            0.93957,
            0.90861,
            0.93245,
            0.92927,
            0.93305,
            0.94423,
            0.90752,
            0.91062,
            0.92228,
            0.93455,
            0.92393,
            0.92584,
            0.90881,
            0.87327,
            0.8513,
            0.81357,
            0.76253,
            0.66566,
            0.7178,
            0.54871,
            0.7426,
            0.59933,
            0.66791,
            0.68889,
            0.84457,
            0.81709,
            0.77558,
            0.63854,
            0.65217,
            0.70431,
            0.62467,
            0.66808,
            0.68893,
            0.62834,
            0.62649,
            0.67836,
            0.57646,
            0.73017,
            0.59271,
            0.73877,
            0.74414,
            0.78049,
            0.70026,
            0.74504,
            0.7215,
            0.7111,
            0.70331,
            0.78742,
            0.58968,
            0.55127,
            0.4321,
            0.40921,
            0.30086,
            0.24841,
            0.1438,
            0.25084,
            0.16142,
            0.16338,
            0.20058,
            0.39887,
            0.47181,
            0.37195,
            0.40532,
            0.27834,
            0.28579,
            0.36821,
            0.19461,
            0.37112,
            0.27423,
            0.49396,
            0.14726,
            0.48378,
            0.26891,
            0.34362,
            0.42411,
            0.34117,
            0.32821,
            0.27067,
            0.46101,
            0.37385,
            0.42066,
            0.4612,
            0.44174,
            0.50503,
            0.4586,
            0.50374,
            0.50275,
            0.5024,
            0.6521,
            0.68622,
            0.63461,
            0.71397,
            0.68765,
            0.60648,
            0.57529,
            0.58987,
            0.57191,
            0.63864,
            0.61509,
            0.63815,
            0.60468,
            0.71338,
            0.69218,
            0.66865,
            0.73732,
            0.68817,
            0.75083,
            0.73928,
            0.73462,
            0.74906,
            0.73227,
            0.75358,
            0.75102,
            0.73728,
            0.7541,
            0.75176,
            0.74884,
            0.73971,
            0.73887,
            0.73857,
            0.73532,
            0.74442,
            0.72805,
            0.73442,
            0.72336,
            0.68174,
            0.71252,
            0.72753,
            0.72685,
            0.71972,
            0.71914,
            0.72278,
            0.71877,
            0.71761,
            0.72068,
            0.70817,
            0.71129,
            0.70337,
            0.71422,
            0.68878,
            0.69896,
            0.70175,
            0.6897,
            0.69508,
            0.69058,
            0.69753,
            0.69636,
            0.69305,
            0.69385,
            0.68628,
            0.69055,
            0.68736,
            0.68787,
            0.67613,
            0.68015,
            0.68234,
            0.68202,
            0.67497,
            0.67172,
            0.67636,
            0.6717,
            0.67176,
            0.672,
            0.66525,
            0.66833,
            0.66452,
            0.64714,
            0.65694,
            0.66274,
            0.65896,
            0.65463,
            0.65521,
            0.65118,
            0.64919,
            0.64646,
            0.64847,
            0.64641,
            0.64482,
            0.63818,
            0.61875,
            0.63585,
            0.62121,
            0.63266,
            0.62239,
            0.63196,
            0.62913,
            0.61713,
            0.62032,
            0.61944,
            0.58626,
            0.60469,
            0.61661,
            0.61536,
            0.60363,
            0.62158,
            0.59252,
            0.61471,
            0.60434,
            0.60321,
            0.60474,
            0.59722,
            0.58083,
            0.5894,
            0.59814,
            0.57852,
            0.5933,
            0.5541,
            0.56697,
            0.59317,
            0.57919,
            0.55573,
            0.58835,
            0.58124,
            0.51058,
            0.53965,
            0.52067,
            0.50323,
            0.57852,
            0.50291,
            0.50772,
            0.48577,
            0.49696,
            0.46883,
            0.46637,
            0.46765,
            0.50644,
            0.39792,
            0.48304,
            0.41565,
            0.41278,
            0.47899,
            0.33154,
            0.41357,
            0.2685,
            0.29985,
            0.24987,
            0.20136,
            0.079618,
            0.21753,
            0.11317,
            0.14189,
            0.18586,
            0.081686,
            0.12817,
            0.1087,
            0.14428,
            0.051589,
            0.15725,
            0.099224,
            0.10591,
            0.070574,
            0.2956,
            0.23411,
            0.15331,
            0.04174,
            0.015462,
            0.12876,
            0.28785,
            0.20329,
            0.2985,
            0.25599,
            0.19337,
            0.22479,
            0.31183,
            0.11326,
            0.14604,
            0.15764,
            0.059176,
            0.27113,
            0.21854,
            0.12164,
            0.2034,
            0.24762,
            0.23812,
            0.14248,
            0.31316,
            0.2809,
            0.31458,
            0.31171,
            0.33693,
            0.28648,
            0.34753,
            0.35002,
            0.46857,
            0.40188,
            0.3886,
            0.37494,
            0.40996,
            0.41954,
            0.4231,
            0.45873,
            0.44831,
            0.45483,
            0.45642,
            0.33692,
            0.4524,
            0.47679,
            0.47235,
            0.36,
            0.48371,
            0.44069,
            0.45514,
            0.32318,
            0.4387,
            0.41985,
            0.40741,
            0.47715,
            0.45575,
            0.33504,
            0.41569,
            0.46239,
            0.4466,
            0.47336,
            0.45434,
            0.4689,
            0.44696,
            0.43131,
            0.47715,
            0.43392,
            0.36489,
            0.44825,
            0.43708,
            0.43717,
            0.43409,
            0.36247,
            0.43692,
            0.48086,
            0.42986,
            0.43346,
            0.41428,
            0.45336,
            0.42232,
            0.42489,
            0.46956,
            0.43407,
            0.4278,
            0.4664,
            0.45528,
            0.45934,
            0.44663,
            0.45805,
            0.46531,
            0.45139,
            0.44406,
            0.44808,
            0.46236,
            0.46819,
            0.43304,
            0.46658,
            0.46721,
            0.46003,
            0.47203,
            0.46633,
            0.45397,
            0.47016,
            0.46504,
            0.46908,
            0.46339,
            0.46797,
            0.46272,
            0.46077,
            0.46197,
            0.46247,
            0.45754,
            0.45528,
            0.45655,
            0.45945,
            0.45746,
            0.4586,
            0.45966,
            0.45705,
            0.45258,
            0.45097,
            0.44773,
            0.44363,
            0.4507,
            0.44023,
            0.43532,
            0.44496,
            0.42725,
            0.4311,
            0.41146,
            0.39567,
            0.40019,
            0.37148,
            0.3957,
            0.38527,
            0.38822,
            0.37051,
            0.24652,
            0.38744,
            0.40825,
            0.40879,
            0.40625,
            0.40614,
            0.41233,
            0.41693,
            0.42001,
            0.42763,
            0.42456,
            0.42204,
            0.41335,
            0.37305,
            0.40733,
            0.42078,
            0.42399,
            0.42714,
            0.42213,
            0.41989,
            0.40936,
            0.41285,
            0.41786,
            0.39618,
            0.41257,
            0.40421,
            0.40514,
            0.38957,
            0.3713,
            0.39183,
            0.40852,
            0.35312,
            0.36228,
            0.39181,
            0.34621,
            0.30062,
            0.38382,
            0.38453,
            0.30594,
            0.34696,
            0.38413,
            0.30114,
            0.33366,
            0.33337,
            0.31352,
            0.28833,
            0.28581,
            0.32419,
            0.31217,
            0.33328,
            0.26855,
            0.25872,
            0.29866,
            0.30217,
            0.23279,
            0.26249,
            0.32224,
            0.28051,
            0.26625,
            0.2345,
            0.17759,
            0.22923,
            0.1448,
            0.14579,
            0.20304,
            0.16925,
            0.23117,
            0.18348,
            0.16454,
            0.17804,
            0.17681,
            0.16831,
            0.17039,
            0.17798,
            0.12711,
            0.075645,
            0.10904,
            0.058186,
            0.060119,
            0.0047451,
            0.016159,
            0.016025,
            0.0046298,
            0.0015164,
            9.6096e-05,
            0.00029009,
            3.6034e-06,
            4.807e-05,
            7.1786e-05,
            4.1948e-06,
            7.3439e-07,
            2.1404e-06,
            4.8133e-09,
            1.8076e-11,
            3.1563e-06,
            1.3589e-06,
            9.0764e-12,
            1.2791e-05,
            4.9764e-06,
            1.481e-13,
            5.1667e-07,
            2.92e-07,
            1.9731e-08,
            2.7498e-06,
            4.4401e-05,
            0.00017917,
            0.00032332,
            0.00025748,
            0.0001227,
            0.0011089,
            5.2164e-05,
            8.1587e-05,
            2.3716e-06,
            2.5672e-06,
            4.4017e-08,
            6.1689e-07,
            2.0899e-06,
            2.5215e-06,
            0.00019896,
            4.0262e-06,
            0.00058098,
            0.00049328,
            0.00034384,
            2.3782e-05,
            0.00011586,
            7.5526e-05,
            6.7136e-07,
            6.3215e-09,
            4.9057e-05,
            0.0012704,
            0.00081226,
            3.2466e-09,
            1.0528e-08,
            0.0018353,
            0.00238,
            0.00073892,
            3.6444e-07,
            0.0020448,
            0.00017457,
            0.0016493,
            0.00061919,
            0.00046653,
            0.0021142,
            0.0026396,
            0.023353,
            0.00036378,
            0.00018366,
            0.035565,
            0.011759,
            0.013559,
            0.0021442,
            0.0082718,
            0.0091637,
            0.046314,
            0.0092198,
            0.016975,
            0.02585,
            0.027792,
            0.049546,
            0.0045588,
            0.03802,
            0.061601,
            0.050156,
            0.0025194,
            0.035834,
            0.020962,
            0.021416,
            0.038351,
            0.02988,
            0.013263,
            0.051039,
            0.039601,
            0.0318,
            0.036317,
            0.045063,
            0.061791,
            0.049751,
            0.023095,
            0.036215,
            0.11569,
            0.10213,
            0.027412,
            0.011271,
            0.062361,
            0.081978,
            0.13759,
            0.06615,
            0.088509,
            0.117,
            0.13643,
            0.16307,
            0.085421,
            0.090276,
            0.1306,
            0.043225,
            0.15184,
            0.093383,
            0.065197,
            0.036054,
            0.076942,
            0.094845,
            0.049678,
            0.017848,
            0.046771,
            0.070198,
            0.097339,
            0.18463,
            0.068778,
            0.069736,
            0.06348,
            0.12001,
            0.060637,
            0.11529,
            0.05849,
            0.14859,
            0.13747,
            0.12503,
            0.1234,
            0.060629,
            0.09418,
            0.18973,
            0.17478,
            0.19778,
            0.16441,
            0.18157,
            0.20367,
            0.18253,
            0.16852,
            0.2285,
            0.18968,
            0.21759,
            0.25061,
            0.26552,
            0.23356,
            0.18493,
            0.16029,
            0.18402,
            0.25773,
            0.25514,
            0.24302,
            0.1869,
            0.27052,
            0.26474,
            0.26068,
            0.24239,
            0.22571,
            0.26573,
            0.25683,
            0.24929,
            0.25211,
            0.24437,
            0.2645,
            0.27505,
            0.26378,
            0.28004,
            0.27539,
            0.25884,
            0.26745,
            0.2622,
            0.27928,
            0.27244,
            0.25522,
            0.26973,
            0.27839,
            0.27714,
            0.26892,
            0.26686,
            0.27464,
            0.27336,
            0.27202,
            0.27295,
            0.26491,
            0.26904,
            0.26927,
            0.27208,
            0.2721,
            0.27705,
            0.27481,
            0.27309,
            0.26675,
            0.27342,
            0.2699,
            0.27058,
            0.27182,
            0.27132,
            0.26474,
            0.26759,
            0.2631,
            0.27062,
            0.26848,
            0.26808,
            0.26568,
            0.27002,
            0.26756,
            0.26667,
            0.26264,
            0.26728,
            0.26245,
            0.26308,
            0.25722,
            0.25452,
            0.24175,
            0.23507,
            0.23775,
            0.23407,
            0.24145,
            0.23974,
            0.24678,
            0.21602,
            0.23516,
            0.23672,
            0.24464,
            0.2487,
            0.24195,
            0.24755,
            0.24904,
            0.25874,
            0.25569,
            0.25303,
            0.25107,
            0.23233,
            0.24179,
            0.24197,
            0.25225,
            0.25833,
            0.25624,
            0.25823,
            0.24452,
            0.24692,
            0.25421,
            0.24202,
            0.2381,
            0.22323,
            0.22413,
            0.22397,
            0.22842,
            0.23683,
            0.2414,
            0.23296,
            0.2299,
            0.22727,
            0.2176,
            0.2268,
            0.23076,
            0.23719,
            0.23838,
            0.24104,
            0.2305,
            0.23465,
            0.24352,
            0.241,
            0.23449,
            0.2343,
            0.23754,
            0.24246,
            0.24269,
            0.23782,
            0.23971,
            0.24078,
            0.24126,
            0.24137,
            0.23651,
            0.23806,
            0.23821,
            0.23267,
            0.23282,
            0.23367,
            0.23539,
            0.227,
            0.22007,
            0.22026,
            0.21511,
            0.2196,
            0.22082,
            0.21535,
            0.22355,
            0.21822,
            0.21749,
            0.22768,
            0.21655,
            0.21867,
            0.22526,
            0.20855,
            0.22373,
            0.22277,
            0.21583,
            0.22231,
            0.22101,
            0.22223,
            0.22487,
            0.2212,
            0.22332,
            0.22384,
            0.21908,
            0.22235,
            0.22098,
            0.21178,
            0.17884,
            0.21068,
            0.21459,
            0.21516,
            0.22168,
            0.21879,
            0.21147,
            0.21629,
            0.21575,
            0.2136,
            0.21145,
            0.21229,
            0.20915,
            0.21303,
            0.20558,
            0.19447,
            0.20366,
            0.20906,
            0.19797,
            0.21321,
            0.21026,
            0.20484,
            0.21013,
            0.20718,
            0.20523,
            0.19303,
            0.20708,
            0.21134,
            0.20477,
            0.20968,
            0.20922,
            0.18107,
            0.20739,
            0.20551,
            0.19975,
            0.20396,
            0.19778,
            0.1879,
            0.18965,
            0.18698,
            0.17808,
            0.17407,
            0.16154,
            0.16818,
            0.15481,
            0.16566,
            0.15301,
            0.15998,
            0.13284,
            0.14172,
            0.11484,
            0.1005,
            0.076981,
            0.088904,
            0.046931,
            0.031828,
            0.014815,
            0.0096911,
            0.0032816,
            0.00098755,
            0.0012744,
            5.2041e-06,
            6.419e-06,
            6.2703e-08,
            6.2658e-06,
            2.9993e-06,
            2.8396e-07,
            1.1151e-05,
            1.6982e-05,
            2.6662e-10,
            4.513e-10,
            7.7505e-05,
            4.389e-05,
            0.00022333,
            0.00012947,
            8.6221e-07,
            5.6667e-07,
            2.3045e-05,
            1.9947e-05,
            0.00045069,
            0.00093615,
            0.00055242,
            0.0035935,
            0.0032821,
            0.010863,
            0.016727,
            0.010036,
            0.021906,
            0.028563,
            0.048847,
            0.067857,
            0.075512,
            0.083063,
            0.085613,
            0.08119,
            0.038156,
            0.015001,
            0.039748,
            0.026648,
            0.044981,
            0.07401,
            0.084856,
            0.096386,
            0.089781,
            0.091074,
            0.067927,
            0.054906,
            0.069193,
            0.061875,
            0.065676,
            0.077443,
            0.086812,
            0.085102,
            0.0891,
            0.089747,
            0.086133,
            0.093153,
            0.089654,
            0.091673,
            0.087588,
            0.088632,
            0.089774,
            0.090044,
            0.090767,
            0.089486,
            0.084639,
            0.08484,
            0.08417,
            0.07631,
            0.081996,
            0.080448,
            0.081808,
            0.07455,
            0.079068,
            0.078992,
            0.071202,
            0.07401,
            0.079315,
            0.076273,
            0.07773,
            0.075453,
            0.075773,
            0.074299,
            0.073118,
            0.070838,
            0.071937,
            0.06769,
            0.066929,
            0.068137,
            0.064867,
            0.064021,
            0.066288,
            0.06308,
            0.06322,
            0.061265,
            0.058824,
            0.059171,
            0.06387,
            0.058141,
            0.052031,
            0.056215,
            0.056824,
            0.057967,
            0.045836,
            0.0514,
            0.041536,
            0.047473,
            0.050237,
            0.049409,
            0.030817,
            0.044147,
            0.042552,
            0.030826,
            0.037109,
            0.040594,
            0.04415,
            0.033599,
            0.033813,
            0.0273,
            0.02659,
            0.033078,
            0.045099,
            0.014878,
            0.043249,
            0.020798,
            0.013611,
            0.024853,
            0.033363,
            0.024148,
            0.016727,
            0.016455,
            0.0080395,
            0.0056102,
            0.0035113,
            0.0028772,
            0.0070642,
            0.0015191,
            0.0022163,
            0.0005188,
            0.00037054,
            4.1393e-05,
            6.3593e-07,
            1.7502e-07,
            3.7716e-07,
            5.3758e-11,
            2.8222e-13,
            1.0435e-09,
            3.102e-11,
            1.5955e-14,
            1.5258e-18,
            1.0786e-27,
            3.8214e-22,
            1.7194e-34,
            5.4793e-31,
            2.2838e-33,
            4.4912e-28,
            5.8053e-35,
            5.9447e-34,
            1.1196e-37,
            5.6505e-29,
            3.8687e-28,
            2.8026e-45,
            3.9027e-16,
            1.175e-16,
            8.9988e-19,
            1.4295e-19,
            1.3133e-27,
            2.6068e-25,
            1.1123e-37,
            0.0,
            0.0,
            0.0,
            0.0,
            1.0226e-29,
            7.1284e-33,
            0.0,
            2.9315e-42,
            1.125e-35,
            3.8557e-26,
            5.6052e-45,
            7.2935e-22,
            6.0734e-19,
            5.4888e-21,
            2.3314e-27,
            1.3146e-23,
            1.6648e-28,
            6.7262e-44,
            0.0,
            2.6777e-27,
            8.3791e-24,
            3.999e-38,
            4.8067e-34,
            3.8866e-27,
            1.217e-16,
            3.6205e-16,
            1.6484e-12,
            6.7478e-14,
            4.0233e-10,
            2.8685e-10,
            2.0548e-11,
            1.7605e-07,
            3.9008e-06,
            2.1276e-10,
            1.9609e-07,
            4.0575e-05,
            1.1566e-06,
            4.4867e-07,
            2.5356e-05,
            0.00016763,
            6.3129e-06,
            0.0003917,
            0.00024724,
            0.00045332,
            0.00018623,
            0.0026643,
            0.00081152,
            0.00011096,
            0.002722,
            0.0012581,
            0.0028948,
            0.0010835,
            0.0058858,
            0.0064903,
            0.0016273,
            0.0014489,
            0.0052276,
            0.0023361,
            0.0045971,
            0.0074379,
            0.00035233,
            0.00085429,
            0.0013381,
            0.0069628,
            0.01028,
            0.0042755,
            0.0078472,
            0.0028906,
            0.0068479,
            0.0055551,
            0.00063369,
            0.0075031,
            0.0060753,
            0.0024986,
            0.0020242,
            0.004209,
            0.0010321,
            0.00028947,
            0.0063012,
            0.0029113,
            0.0017492,
            0.0060221,
            0.0036224,
            0.0017671,
            0.0023805,
            0.0006551,
            0.004401,
            0.00092155,
            0.00084569,
            0.0022677,
            0.0098197,
            0.0030289,
            0.0057614,
            0.011446,
            0.0033241,
            0.0032517,
            0.0066744,
            0.0056366,
            0.009232,
            0.014017,
            0.012516,
            0.0092302,
            0.010621,
            0.0080823,
            0.0042388,
            0.0026927,
            0.00043843,
            0.00030973,
            0.00013634,
            0.00049752,
            0.0016089,
            0.00019875,
            0.0003408,
            0.007294,
            0.0037464,
            0.00073409,
            0.0026067,
            0.0099378,
            0.0012248,
            0.0024465,
            0.0012186,
            0.0059265,
            0.0028644,
            0.011128,
            0.0087571,
            0.0012234,
            0.0017794,
            0.0039416,
            0.0039235,
            1.6133e-05,
            5.9987e-05,
            0.0035187,
            0.0046616,
            0.0090694,
            0.0034602,
            0.0035408,
            0.0080277,
            0.0036308,
            0.0052402,
            0.0071907,
            0.0039389,
            0.008456,
            0.0051115,
            0.0074896,
            0.0098552,
            0.0095465,
            0.012509,
            0.0044594,
            0.0070802,
            0.0072774,
            0.013165,
            0.010006,
            0.0086892,
            0.011553,
            0.0080348,
            0.011318,
            0.011153,
            0.0083089,
            0.01253,
            0.0098179,
            0.012264,
            0.010943,
            0.011224,
            0.012094,
            0.010419,
            0.012265,
            0.011917,
            0.011809,
            0.011963,
            0.011494,
            0.012122,
            0.011428,
            0.011127,
            0.0094556,
            0.009031,
            0.0095432,
            0.010538,
            0.0090581,
            0.010795,
            0.010851,
            0.0083376,
            0.0086444,
            0.010187,
            0.0091671,
            0.0094523,
            0.00967,
            0.010262,
            0.010359,
            0.0094787,
            0.0094726,
            0.011614,
            0.010239,
            0.009955,
            0.010299,
            0.01148,
            0.010599,
            0.010123,
            0.010978,
            0.010914,
            0.010253,
            0.0079003,
            0.0048286,
            0.0083312,
            0.009438,
            0.0096922,
            0.010132,
            0.010878,
            0.01077,
            0.009364,
            0.0092254,
            0.010376,
            0.010698,
            0.0092707,
            0.0085837,
            0.0088494,
            0.010331,
            0.0092903,
            0.0089918,
            0.0088633,
            0.0085502,
            0.0091243,
            0.0090521,
            0.0095746,
            0.0088123,
            0.0077564,
            0.0088692,
            0.0098592,
            0.0093049,
            0.0082451,
            0.0077569,
            0.009655,
            0.0095056,
            0.0095925,
            0.0076916,
            0.0089756,
            0.0087801,
            0.0088274,
            0.0085085,
            0.007994,
            0.0080989,
            0.0073604,
            0.006762,
            0.006534,
            0.0067717,
            0.0068818,
            0.007476,
            0.0079254,
            0.0079269,
            0.0071353,
            0.0069868,
            0.0069466,
            0.006852,
            0.0070502,
            0.0073541,
            0.0074027,
            0.0075412,
            0.0076277,
            0.0077199,
            0.0077482,
            0.0078057,
            0.0076806,
            0.0075097,
            0.0073872,
            0.0074327,
            0.0073723,
            0.00721,
            0.0071043,
        ]

        super().__init__(x=wavs * x_unit, y=vals * y_unit)
