"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn_element_1 = require("./cfn-element");
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
/**
 * A CloudFormation parameter.
 *
 * Use the optional Parameters section to customize your templates.
 * Parameters enable you to input custom values to your template each time you create or
 * update a stack.
 */
class CfnParameter extends cfn_element_1.CfnElement {
    /**
     * Creates a parameter construct.
     * Note that the name (logical ID) of the parameter will derive from it's `coname` and location
     * within the stack. Therefore, it is recommended that parameters are defined at the stack level.
     *
     * @param scope The parent construct.
     * @param props The parameter properties.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.props = props;
        this.type = props.type || 'String';
    }
    /**
     * Indicates if this parameter is configured with "NoEcho" enabled.
     */
    get noEcho() {
        return !!this.props.noEcho;
    }
    /**
     * The parameter value as a Token
     */
    get value() {
        return cfn_reference_1.CfnReference.for(this, 'Ref');
    }
    /**
     * The parameter value, if it represents a string.
     */
    get valueAsString() {
        if (!isStringType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string type`);
        }
        return token_1.Token.asString(this.value);
    }
    /**
     * The parameter value, if it represents a string list.
     */
    get valueAsList() {
        if (!isListType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string list type`);
        }
        return token_1.Token.asList(this.value);
    }
    /**
     * The parameter value, if it represents a number.
     */
    get valueAsNumber() {
        if (!isNumberType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a number type`);
        }
        return token_1.Token.asNumber(this.value);
    }
    /**
     * @internal
     */
    _toCloudFormation() {
        return {
            Parameters: {
                [this.logicalId]: {
                    Type: this.type,
                    Default: this.props.default,
                    AllowedPattern: this.props.allowedPattern,
                    AllowedValues: this.props.allowedValues,
                    ConstraintDescription: this.props.constraintDescription,
                    Description: this.props.description,
                    MaxLength: this.props.maxLength,
                    MaxValue: this.props.maxValue,
                    MinLength: this.props.minLength,
                    MinValue: this.props.minValue,
                    NoEcho: this.props.noEcho,
                },
            },
        };
    }
    resolve(_context) {
        return this.value;
    }
}
exports.CfnParameter = CfnParameter;
/**
 * Whether the given parameter type looks like a list type
 */
function isListType(type) {
    return type.indexOf('List<') >= 0 || type.indexOf('CommaDelimitedList') >= 0;
}
/**
 * Whether the given parameter type looks like a number type
 */
function isNumberType(type) {
    return type === 'Number';
}
/**
 * Whether the given parameter type looks like a string type
 */
function isStringType(type) {
    return !isListType(type) && !isNumberType(type);
}
//# sourceMappingURL=data:application/json;base64,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