"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * A set of constructs to be used as a dependable
 *
 * This class can be used when a set of constructs which are disjoint in the
 * construct tree needs to be combined to be used as a single dependable.
 *
 * @experimental
 */
class ConcreteDependable {
    constructor() {
        this._dependencyRoots = new Array();
        const self = this;
        DependableTrait.implement(this, {
            get dependencyRoots() { return self._dependencyRoots; },
        });
    }
    /**
     * Add a construct to the dependency roots
     */
    add(construct) {
        this._dependencyRoots.push(construct);
    }
}
exports.ConcreteDependable = ConcreteDependable;
const DEPENDABLE_SYMBOL = Symbol.for('@aws-cdk/core.DependableTrait');
/**
 * Trait for IDependable
 *
 * Traits are interfaces that are privately implemented by objects. Instead of
 * showing up in the public interface of a class, they need to be queried
 * explicitly. This is used to implement certain framework features that are
 * not intended to be used by Construct consumers, and so should be hidden
 * from accidental use.
 *
 * @example
 *
 * // Usage
 * const roots = DependableTrait.get(construct).dependencyRoots;
 *
 * // Definition
 * DependableTrait.implement(construct, {
 *   get dependencyRoots() { return []; }
 * });
 *
 * @experimental
 */
class DependableTrait {
    /**
     * Register `instance` to have the given DependableTrait
     *
     * Should be called in the class constructor.
     */
    static implement(instance, trait) {
        // I would also like to reference classes (to cut down on the list of objects
        // we need to manage), but we can't do that either since jsii doesn't have the
        // concept of a class reference.
        instance[DEPENDABLE_SYMBOL] = trait;
    }
    /**
     * Return the matching DependableTrait for the given class instance.
     */
    static get(instance) {
        const ret = instance[DEPENDABLE_SYMBOL];
        if (!ret) {
            throw new Error(`${instance} does not implement DependableTrait`);
        }
        return ret;
    }
}
exports.DependableTrait = DependableTrait;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVwZW5kZW5jeS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImRlcGVuZGVuY3kudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFpQkE7Ozs7Ozs7R0FPRztBQUNILE1BQWEsa0JBQWtCO0lBRzdCO1FBRmlCLHFCQUFnQixHQUFHLElBQUksS0FBSyxFQUFjLENBQUM7UUFHMUQsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBQ2xCLGVBQWUsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFO1lBQzlCLElBQUksZUFBZSxLQUFLLE9BQU8sSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQztTQUN4RCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxHQUFHLENBQUMsU0FBcUI7UUFDOUIsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUN4QyxDQUFDO0NBQ0Y7QUFoQkQsZ0RBZ0JDO0FBRUQsTUFBTSxpQkFBaUIsR0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDLENBQUM7QUFFdEU7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBb0JHO0FBQ0gsTUFBc0IsZUFBZTtJQUNuQzs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLFNBQVMsQ0FBQyxRQUFxQixFQUFFLEtBQXNCO1FBQ25FLDZFQUE2RTtRQUM3RSw4RUFBOEU7UUFDOUUsZ0NBQWdDO1FBQy9CLFFBQWdCLENBQUMsaUJBQWlCLENBQUMsR0FBRyxLQUFLLENBQUM7SUFDL0MsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEdBQUcsQ0FBQyxRQUFxQjtRQUNyQyxNQUFNLEdBQUcsR0FBSSxRQUFnQixDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDakQsSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUNSLE1BQU0sSUFBSSxLQUFLLENBQUMsR0FBRyxRQUFRLHFDQUFxQyxDQUFDLENBQUM7U0FDbkU7UUFDRCxPQUFPLEdBQUcsQ0FBQztJQUNiLENBQUM7Q0FTRjtBQS9CRCwwQ0ErQkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBJQ29uc3RydWN0IH0gZnJvbSAnLi9jb25zdHJ1Y3QtY29tcGF0JztcblxuLyoqXG4gKiBUcmFpdCBtYXJrZXIgZm9yIGNsYXNzZXMgdGhhdCBjYW4gYmUgZGVwZW5kZWQgdXBvblxuICpcbiAqIFRoZSBwcmVzZW5jZSBvZiB0aGlzIGludGVyZmFjZSBpbmRpY2F0ZXMgdGhhdCBhbiBvYmplY3QgaGFzXG4gKiBhbiBgSURlcGVuZGFibGVUcmFpdGAgaW1wbGVtZW50YXRpb24uXG4gKlxuICogVGhpcyBpbnRlcmZhY2UgY2FuIGJlIHVzZWQgdG8gdGFrZSBhbiAob3JkZXJpbmcpIGRlcGVuZGVuY3kgb24gYSBzZXQgb2ZcbiAqIGNvbnN0cnVjdHMuIEFuIG9yZGVyaW5nIGRlcGVuZGVuY3kgaW1wbGllcyB0aGF0IHRoZSByZXNvdXJjZXMgcmVwcmVzZW50ZWQgYnlcbiAqIHRob3NlIGNvbnN0cnVjdHMgYXJlIGRlcGxveWVkIGJlZm9yZSB0aGUgcmVzb3VyY2VzIGRlcGVuZGluZyBPTiB0aGVtIGFyZVxuICogZGVwbG95ZWQuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSURlcGVuZGFibGUge1xuICAvLyBFbXB0eSwgdGhpcyBpbnRlcmZhY2UgaXMgYSB0cmFpdCBtYXJrZXJcbn1cblxuLyoqXG4gKiBBIHNldCBvZiBjb25zdHJ1Y3RzIHRvIGJlIHVzZWQgYXMgYSBkZXBlbmRhYmxlXG4gKlxuICogVGhpcyBjbGFzcyBjYW4gYmUgdXNlZCB3aGVuIGEgc2V0IG9mIGNvbnN0cnVjdHMgd2hpY2ggYXJlIGRpc2pvaW50IGluIHRoZVxuICogY29uc3RydWN0IHRyZWUgbmVlZHMgdG8gYmUgY29tYmluZWQgdG8gYmUgdXNlZCBhcyBhIHNpbmdsZSBkZXBlbmRhYmxlLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGNsYXNzIENvbmNyZXRlRGVwZW5kYWJsZSBpbXBsZW1lbnRzIElEZXBlbmRhYmxlIHtcbiAgcHJpdmF0ZSByZWFkb25seSBfZGVwZW5kZW5jeVJvb3RzID0gbmV3IEFycmF5PElDb25zdHJ1Y3Q+KCk7XG5cbiAgY29uc3RydWN0b3IoKSB7XG4gICAgY29uc3Qgc2VsZiA9IHRoaXM7XG4gICAgRGVwZW5kYWJsZVRyYWl0LmltcGxlbWVudCh0aGlzLCB7XG4gICAgICBnZXQgZGVwZW5kZW5jeVJvb3RzKCkgeyByZXR1cm4gc2VsZi5fZGVwZW5kZW5jeVJvb3RzOyB9LFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIGNvbnN0cnVjdCB0byB0aGUgZGVwZW5kZW5jeSByb290c1xuICAgKi9cbiAgcHVibGljIGFkZChjb25zdHJ1Y3Q6IElDb25zdHJ1Y3QpIHtcbiAgICB0aGlzLl9kZXBlbmRlbmN5Um9vdHMucHVzaChjb25zdHJ1Y3QpO1xuICB9XG59XG5cbmNvbnN0IERFUEVOREFCTEVfU1lNQk9MID0gU3ltYm9sLmZvcignQGF3cy1jZGsvY29yZS5EZXBlbmRhYmxlVHJhaXQnKTtcblxuLyoqXG4gKiBUcmFpdCBmb3IgSURlcGVuZGFibGVcbiAqXG4gKiBUcmFpdHMgYXJlIGludGVyZmFjZXMgdGhhdCBhcmUgcHJpdmF0ZWx5IGltcGxlbWVudGVkIGJ5IG9iamVjdHMuIEluc3RlYWQgb2ZcbiAqIHNob3dpbmcgdXAgaW4gdGhlIHB1YmxpYyBpbnRlcmZhY2Ugb2YgYSBjbGFzcywgdGhleSBuZWVkIHRvIGJlIHF1ZXJpZWRcbiAqIGV4cGxpY2l0bHkuIFRoaXMgaXMgdXNlZCB0byBpbXBsZW1lbnQgY2VydGFpbiBmcmFtZXdvcmsgZmVhdHVyZXMgdGhhdCBhcmVcbiAqIG5vdCBpbnRlbmRlZCB0byBiZSB1c2VkIGJ5IENvbnN0cnVjdCBjb25zdW1lcnMsIGFuZCBzbyBzaG91bGQgYmUgaGlkZGVuXG4gKiBmcm9tIGFjY2lkZW50YWwgdXNlLlxuICpcbiAqIEBleGFtcGxlXG4gKlxuICogLy8gVXNhZ2VcbiAqIGNvbnN0IHJvb3RzID0gRGVwZW5kYWJsZVRyYWl0LmdldChjb25zdHJ1Y3QpLmRlcGVuZGVuY3lSb290cztcbiAqXG4gKiAvLyBEZWZpbml0aW9uXG4gKiBEZXBlbmRhYmxlVHJhaXQuaW1wbGVtZW50KGNvbnN0cnVjdCwge1xuICogICBnZXQgZGVwZW5kZW5jeVJvb3RzKCkgeyByZXR1cm4gW107IH1cbiAqIH0pO1xuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIERlcGVuZGFibGVUcmFpdCB7XG4gIC8qKlxuICAgKiBSZWdpc3RlciBgaW5zdGFuY2VgIHRvIGhhdmUgdGhlIGdpdmVuIERlcGVuZGFibGVUcmFpdFxuICAgKlxuICAgKiBTaG91bGQgYmUgY2FsbGVkIGluIHRoZSBjbGFzcyBjb25zdHJ1Y3Rvci5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgaW1wbGVtZW50KGluc3RhbmNlOiBJRGVwZW5kYWJsZSwgdHJhaXQ6IERlcGVuZGFibGVUcmFpdCkge1xuICAgIC8vIEkgd291bGQgYWxzbyBsaWtlIHRvIHJlZmVyZW5jZSBjbGFzc2VzICh0byBjdXQgZG93biBvbiB0aGUgbGlzdCBvZiBvYmplY3RzXG4gICAgLy8gd2UgbmVlZCB0byBtYW5hZ2UpLCBidXQgd2UgY2FuJ3QgZG8gdGhhdCBlaXRoZXIgc2luY2UganNpaSBkb2Vzbid0IGhhdmUgdGhlXG4gICAgLy8gY29uY2VwdCBvZiBhIGNsYXNzIHJlZmVyZW5jZS5cbiAgICAoaW5zdGFuY2UgYXMgYW55KVtERVBFTkRBQkxFX1NZTUJPTF0gPSB0cmFpdDtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gdGhlIG1hdGNoaW5nIERlcGVuZGFibGVUcmFpdCBmb3IgdGhlIGdpdmVuIGNsYXNzIGluc3RhbmNlLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBnZXQoaW5zdGFuY2U6IElEZXBlbmRhYmxlKTogRGVwZW5kYWJsZVRyYWl0IHtcbiAgICBjb25zdCByZXQgPSAoaW5zdGFuY2UgYXMgYW55KVtERVBFTkRBQkxFX1NZTUJPTF07XG4gICAgaWYgKCFyZXQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgJHtpbnN0YW5jZX0gZG9lcyBub3QgaW1wbGVtZW50IERlcGVuZGFibGVUcmFpdGApO1xuICAgIH1cbiAgICByZXR1cm4gcmV0O1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBzZXQgb2YgY29uc3RydWN0cyB0aGF0IGZvcm0gdGhlIHJvb3Qgb2YgdGhpcyBkZXBlbmRhYmxlXG4gICAqXG4gICAqIEFsbCByZXNvdXJjZXMgdW5kZXIgYWxsIHJldHVybmVkIGNvbnN0cnVjdHMgYXJlIGluY2x1ZGVkIGluIHRoZSBvcmRlcmluZ1xuICAgKiBkZXBlbmRlbmN5LlxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGRlcGVuZGVuY3lSb290czogSUNvbnN0cnVjdFtdO1xufVxuIl19