"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const stack_1 = require("./stack");
const util_1 = require("./util");
/**
 * Adds a dependency between two resources or stacks, across stack and nested
 * stack boundaries.
 *
 * The algorithm consists of:
 * - Try to find the deepest common stack between the two elements
 * - If there isn't a common stack, it means the elements belong to two
 *   disjoined stack-trees and therefore we apply the dependency at the
 *   assembly/app level between the two topl-level stacks.
 * - If we did find a common stack, we apply the dependency as a CloudFormation
 *   "DependsOn" between the resources that "represent" our source and target
 *   either directly or through the AWS::CloudFormation::Stack resources that
 *   "lead" to them.
 *
 * @param source The source resource/stack (the depedent)
 * @param target The target resource/stack (the dependency)
 * @param reason Optional resource to associate with the dependency for
 * diagnostics
 */
function addDependency(source, target, reason) {
    if (source === target) {
        return;
    }
    const sourceStack = stack_1.Stack.of(source);
    const targetStack = stack_1.Stack.of(target);
    // find the deepest common stack between the two elements
    const sourcePath = util_1.pathToTopLevelStack(sourceStack);
    const targetPath = util_1.pathToTopLevelStack(targetStack);
    const commonStack = util_1.findLastCommonElement(sourcePath, targetPath);
    // if there is no common stack, then define an assembly-level dependency
    // between the two top-level stacks
    if (!commonStack) {
        const topLevelSource = sourcePath[0]; // first path element is the top-level stack
        const topLevelTarget = targetPath[0];
        topLevelSource._addAssemblyDependency(topLevelTarget, reason);
        return;
    }
    // assertion: at this point if source and target are stacks, both are nested stacks.
    // since we have a common stack, it is impossible that both are top-level
    // stacks, so let's examine the two cases where one of them is top-level and
    // the other is nested.
    // case 1 - source is top-level and target is nested: this implies that
    // `target` is a direct or indirect nested stack of `source`, and an explicit
    // dependency is not required because nested stacks will always be deployed
    // before their parents.
    if (commonStack === source) {
        return;
    }
    // case 2 - source is nested and target is top-level: this implies that
    // `source` is a direct or indirect nested stack of `target`, and this is not
    // possible (nested stacks cannot depend on their parents).
    if (commonStack === target) {
        throw new Error(`Nested stack '${sourceStack.node.path}' cannot depend on a parent stack '${targetStack.node.path}': ${reason}`);
    }
    // we have a common stack from which we can reach both `source` and `target`
    // now we need to find two resources which are defined directly in this stack
    // and which can "lead us" to the source/target.
    const sourceResource = resourceInCommonStackFor(source);
    const targetResource = resourceInCommonStackFor(target);
    sourceResource._addResourceDependency(targetResource);
    function resourceInCommonStackFor(element) {
        const resource = stack_1.Stack.isStack(element) ? element.nestedStackResource : element;
        if (!resource) {
            throw new Error('assertion failure'); // see "assertion" above
        }
        const resourceStack = stack_1.Stack.of(resource);
        // we reached a resource defined in the common stack
        if (commonStack === resourceStack) {
            return resource;
        }
        return resourceInCommonStackFor(resourceStack);
    }
}
exports.addDependency = addDependency;
//# sourceMappingURL=data:application/json;base64,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