"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto = require("crypto");
const stack_1 = require("../stack");
const token_1 = require("../token");
const token_map_1 = require("./token-map");
function generatePhysicalName(resource) {
    const stack = stack_1.Stack.of(resource);
    const stackPart = new PrefixNamePart(stack.stackName, 25);
    const idPart = new SuffixNamePart(resource.node.uniqueId, 24);
    const region = stack.region;
    if (token_1.Token.isUnresolved(region) || !region) {
        throw new Error(`Cannot generate a physical name for ${resource.node.path}, because the region is un-resolved or missing`);
    }
    const account = stack.account;
    if (token_1.Token.isUnresolved(account) || !account) {
        throw new Error(`Cannot generate a physical name for ${resource.node.path}, because the account is un-resolved or missing`);
    }
    const parts = [stackPart, idPart]
        .map(part => part.generate());
    const hashLength = 12;
    const sha256 = crypto.createHash('sha256')
        .update(stackPart.bareStr)
        .update(idPart.bareStr)
        .update(region)
        .update(account);
    const hash = sha256.digest('hex').slice(0, hashLength);
    const ret = [...parts, hash].join('');
    return ret.toLowerCase();
}
exports.generatePhysicalName = generatePhysicalName;
class NamePart {
    constructor(bareStr) {
        this.bareStr = bareStr;
    }
}
class PrefixNamePart extends NamePart {
    constructor(bareStr, prefixLength) {
        super(bareStr);
        this.prefixLength = prefixLength;
    }
    generate() {
        return this.bareStr.slice(0, this.prefixLength);
    }
}
class SuffixNamePart extends NamePart {
    constructor(str, suffixLength) {
        super(str);
        this.suffixLength = suffixLength;
    }
    generate() {
        const strLen = this.bareStr.length;
        const startIndex = Math.max(strLen - this.suffixLength, 0);
        return this.bareStr.slice(startIndex, strLen);
    }
}
const GENERATE_IF_NEEDED_SYMBOL = Symbol.for('@aws-cdk/core.<private>.GenerateIfNeeded');
/**
 * This marker token is used by PhysicalName.GENERATE_IF_NEEDED. When that token is passed to the
 * physicalName property of a Resource, it triggers different behavior in the Resource constructor
 * that will allow emission of a generated physical name (when the resource is used across
 * environments) or undefined (when the resource is not shared).
 *
 * This token throws an Error when it is resolved, as a way to prevent inadvertent mis-uses of it.
 */
class GeneratedWhenNeededMarker {
    constructor() {
        this.creationStack = [];
        Object.defineProperty(this, GENERATE_IF_NEEDED_SYMBOL, { value: true });
    }
    resolve(_ctx) {
        throw new Error('Invalid physical name passed to CloudFormation. Use "this.physicalName" instead');
    }
    toString() {
        return 'PhysicalName.GENERATE_IF_NEEDED';
    }
}
exports.GeneratedWhenNeededMarker = GeneratedWhenNeededMarker;
/**
 * Checks whether a stringified token resolves to a `GeneratedWhenNeededMarker`.
 */
function isGeneratedWhenNeededMarker(val) {
    const token = token_map_1.TokenMap.instance().lookupString(val);
    return !!token && GENERATE_IF_NEEDED_SYMBOL in token;
}
exports.isGeneratedWhenNeededMarker = isGeneratedWhenNeededMarker;
//# sourceMappingURL=data:application/json;base64,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