"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const token_1 = require("./token");
/**
 * Fragments of a concatenated string containing stringified Tokens
 */
class TokenizedStringFragments {
    constructor() {
        this.fragments = new Array();
    }
    get firstToken() {
        const first = this.fragments[0];
        if (first.type === 'token') {
            return first.token;
        }
        return undefined;
    }
    get firstValue() {
        return fragmentValue(this.fragments[0]);
    }
    get length() {
        return this.fragments.length;
    }
    addLiteral(lit) {
        this.fragments.push({ type: 'literal', lit });
    }
    addToken(token) {
        this.fragments.push({ type: 'token', token });
    }
    addIntrinsic(value) {
        this.fragments.push({ type: 'intrinsic', value });
    }
    /**
     * Return all Tokens from this string
     */
    get tokens() {
        const ret = new Array();
        for (const f of this.fragments) {
            if (f.type === 'token') {
                ret.push(f.token);
            }
        }
        return ret;
    }
    /**
     * Apply a transformation function to all tokens in the string
     */
    mapTokens(mapper) {
        const ret = new TokenizedStringFragments();
        for (const f of this.fragments) {
            switch (f.type) {
                case 'literal':
                    ret.addLiteral(f.lit);
                    break;
                case 'token':
                    const mapped = mapper.mapToken(f.token);
                    if (token_1.isResolvableObject(mapped)) {
                        ret.addToken(mapped);
                    }
                    else {
                        ret.addIntrinsic(mapped);
                    }
                    break;
                case 'intrinsic':
                    ret.addIntrinsic(f.value);
                    break;
            }
        }
        return ret;
    }
    /**
     * Combine the string fragments using the given joiner.
     *
     * If there are any
     */
    join(concat) {
        if (this.fragments.length === 0) {
            return concat.join(undefined, undefined);
        }
        if (this.fragments.length === 1) {
            return this.firstValue;
        }
        const values = this.fragments.map(fragmentValue);
        while (values.length > 1) {
            const prefix = values.splice(0, 2);
            values.splice(0, 0, concat.join(prefix[0], prefix[1]));
        }
        return values[0];
    }
}
exports.TokenizedStringFragments = TokenizedStringFragments;
/**
 * Resolve the value from a single fragment
 *
 * If the fragment is a Token, return the string encoding of the Token.
 */
function fragmentValue(fragment) {
    switch (fragment.type) {
        case 'literal': return fragment.lit;
        case 'token': return fragment.token.toString();
        case 'intrinsic': return fragment.value;
    }
}
//# sourceMappingURL=data:application/json;base64,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