"use strict";
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const lib_1 = require("../lib");
const util_1 = require("./util");
module.exports = {
    'addFileAsset correctly sets metadata and creates S3 parameters'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        stack.addFileAsset({
            fileName: 'file-name',
            packaging: lib_1.FileAssetPackaging.ZIP_DIRECTORY,
            sourceHash: 'source-hash',
        });
        // THEN
        const assetMetadata = stack.node.metadata.find(({ type }) => type === cxschema.ArtifactMetadataEntryType.ASSET);
        test.ok(assetMetadata && assetMetadata.data);
        if (assetMetadata && assetMetadata.data) {
            const data = assetMetadata.data;
            test.equal(data.path, 'file-name');
            test.equal(data.id, 'source-hash');
            test.equal(data.packaging, lib_1.FileAssetPackaging.ZIP_DIRECTORY);
            test.equal(data.sourceHash, 'source-hash');
        }
        test.deepEqual(util_1.toCloudFormation(stack), {
            Parameters: {
                AssetParameterssourcehashS3BucketE6E91E3E: {
                    Type: 'String',
                    Description: 'S3 bucket for asset "source-hash"',
                },
                AssetParameterssourcehashS3VersionKeyAC4157C3: {
                    Type: 'String',
                    Description: 'S3 key for asset version "source-hash"',
                },
                AssetParameterssourcehashArtifactHashADBAE418: {
                    Type: 'String',
                    Description: 'Artifact hash for asset "source-hash"',
                },
            },
        });
        test.done();
    },
    'addDockerImageAsset correctly sets metadata'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        stack.addDockerImageAsset({
            sourceHash: 'source-hash',
            directoryName: 'directory-name',
            repositoryName: 'repository-name',
        });
        // THEN
        const assetMetadata = stack.node.metadata.find(({ type }) => type === cxschema.ArtifactMetadataEntryType.ASSET);
        test.ok(assetMetadata && assetMetadata.data);
        if (assetMetadata && assetMetadata.data) {
            const data = assetMetadata.data;
            test.equal(data.packaging, 'container-image');
            test.equal(data.path, 'directory-name');
            test.equal(data.sourceHash, 'source-hash');
            test.equal(data.repositoryName, 'repository-name');
            test.equal(data.imageTag, 'source-hash');
        }
        test.deepEqual(util_1.toCloudFormation(stack), {});
        test.done();
    },
    'addDockerImageAsset uses the default repository name'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        // WHEN
        stack.addDockerImageAsset({
            sourceHash: 'source-hash',
            directoryName: 'directory-name',
        });
        // THEN
        const assetMetadata = stack.node.metadata.find(({ type }) => type === cxschema.ArtifactMetadataEntryType.ASSET);
        test.ok(assetMetadata && assetMetadata.data);
        if (assetMetadata && assetMetadata.data) {
            const data = assetMetadata.data;
            test.equal(data.packaging, 'container-image');
            test.equal(data.path, 'directory-name');
            test.equal(data.sourceHash, 'source-hash');
            test.equal(data.repositoryName, 'aws-cdk/assets');
            test.equal(data.imageTag, 'source-hash');
        }
        test.deepEqual(util_1.toCloudFormation(stack), {});
        test.done();
    },
    'addDockerImageAsset supports overriding repository name through a context key as a workaround until we have API for that'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        stack.node.setContext('assets-ecr-repository-name', 'my-custom-repo-name');
        // WHEN
        stack.addDockerImageAsset({
            sourceHash: 'source-hash',
            directoryName: 'directory-name',
        });
        // THEN
        const assetMetadata = stack.node.metadata.find(({ type }) => type === cxschema.ArtifactMetadataEntryType.ASSET);
        test.ok(assetMetadata && assetMetadata.data);
        if (assetMetadata && assetMetadata.data) {
            const data = assetMetadata.data;
            test.equal(data.packaging, 'container-image');
            test.equal(data.path, 'directory-name');
            test.equal(data.sourceHash, 'source-hash');
            test.equal(data.repositoryName, 'my-custom-repo-name');
            test.equal(data.imageTag, 'source-hash');
        }
        test.deepEqual(util_1.toCloudFormation(stack), {});
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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