"use strict";
const lib_1 = require("../lib");
const context_provider_1 = require("../lib/context-provider");
/**
 * Get the expected context key from a stack with missing parameters
 */
function expectedContextKey(stack) {
    const missing = lib_1.ConstructNode.synth(stack.node).manifest.missing;
    if (!missing || missing.length !== 1) {
        throw new Error('Expecting assembly to include a single missing context report');
    }
    return missing[0].key;
}
module.exports = {
    'AvailabilityZoneProvider returns a list with dummy values if the context is not available'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const azs = stack.availabilityZones;
        test.deepEqual(azs, ['dummy1a', 'dummy1b', 'dummy1c']);
        test.done();
    },
    'AvailabilityZoneProvider will return context list if available'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const before = stack.availabilityZones;
        test.deepEqual(before, ['dummy1a', 'dummy1b', 'dummy1c']);
        const key = expectedContextKey(stack);
        stack.node.setContext(key, ['us-east-1a', 'us-east-1b']);
        const azs = stack.availabilityZones;
        test.deepEqual(azs, ['us-east-1a', 'us-east-1b']);
        test.done();
    },
    'AvailabilityZoneProvider will complain if not given a list'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const before = stack.availabilityZones;
        test.deepEqual(before, ['dummy1a', 'dummy1b', 'dummy1c']);
        const key = expectedContextKey(stack);
        stack.node.setContext(key, 'not-a-list');
        test.throws(() => stack.availabilityZones);
        test.done();
    },
    'ContextProvider consistently generates a key'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const key = context_provider_1.ContextProvider.getKey(stack, {
            provider: 'ssm',
            props: {
                parameterName: 'foo',
                anyStringParam: 'bar',
            },
        });
        test.deepEqual(key, {
            key: 'ssm:account=12345:anyStringParam=bar:parameterName=foo:region=us-east-1',
            props: {
                account: '12345',
                region: 'us-east-1',
                parameterName: 'foo',
                anyStringParam: 'bar',
            },
        });
        const complexKey = context_provider_1.ContextProvider.getKey(stack, {
            provider: 'vpc',
            props: {
                cidrBlock: '192.168.0.16',
                tags: { Name: 'MyVPC', Env: 'Preprod' },
                igw: false,
            },
        });
        test.deepEqual(complexKey, {
            key: 'vpc:account=12345:cidrBlock=192.168.0.16:igw=false:region=us-east-1:tags.Env=Preprod:tags.Name=MyVPC',
            props: {
                account: '12345',
                region: 'us-east-1',
                cidrBlock: '192.168.0.16',
                tags: { Name: 'MyVPC', Env: 'Preprod' },
                igw: false,
            },
        });
        test.done();
    },
    'Key generation can contain arbitrarily deep structures'(test) {
        // GIVEN
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        // WHEN
        const key = context_provider_1.ContextProvider.getKey(stack, {
            provider: 'provider',
            props: {
                list: [
                    { key: 'key1', value: 'value1' },
                    { key: 'key2', value: 'value2' },
                ],
            },
        });
        // THEN
        test.deepEqual(key, {
            key: 'provider:account=12345:list.0.key=key1:list.0.value=value1:list.1.key=key2:list.1.value=value2:region=us-east-1',
            props: {
                account: '12345',
                region: 'us-east-1',
                list: [
                    { key: 'key1', value: 'value1' },
                    { key: 'key2', value: 'value2' },
                ],
            },
        });
        test.done();
    },
    'Keys with undefined values are not serialized'(test) {
        // GIVEN
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        // WHEN
        const result = context_provider_1.ContextProvider.getKey(stack, {
            provider: 'provider',
            props: {
                p1: 42,
                p2: undefined,
            },
        });
        // THEN
        test.deepEqual(result, {
            key: 'provider:account=12345:p1=42:region=us-east-1',
            props: {
                account: '12345',
                region: 'us-east-1',
                p1: 42,
                p2: undefined,
            },
        });
        test.done();
    },
    'context provider errors are attached to tree'(test) {
        const contextProps = { provider: 'bloop' };
        const contextKey = 'bloop:account=12345:region=us-east-1'; // Depends on the mangling algo
        // GIVEN
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        // NOTE: error key is inlined here because it's part of the CX-API
        // compatibility surface.
        stack.node.setContext(contextKey, { $providerError: 'I had a boo-boo' });
        const construct = new lib_1.Construct(stack, 'Child');
        // Verify that we got the right hardcoded key above, give a descriptive error if not
        test.equals(context_provider_1.ContextProvider.getKey(construct, contextProps).key, contextKey);
        // WHEN
        context_provider_1.ContextProvider.getValue(construct, {
            ...contextProps,
            dummyValue: undefined,
        });
        // THEN
        const error = construct.node.metadata.find(m => m.type === 'aws:cdk:error');
        test.equals(error && error.data, 'I had a boo-boo');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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