"use strict";
const lib_1 = require("../lib");
module.exports = {
    'By default, environment region and account are not defined and resolve to intrinsics'(test) {
        const stack = new lib_1.Stack();
        test.ok(lib_1.Token.isUnresolved(stack.account));
        test.ok(lib_1.Token.isUnresolved(stack.region));
        test.deepEqual(stack.resolve(stack.account), { Ref: 'AWS::AccountId' });
        test.deepEqual(stack.resolve(stack.region), { Ref: 'AWS::Region' });
        test.done();
    },
    'If only `env.region` or `env.account` are specified, Refs will be used for the other'(test) {
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'S1', { env: { region: 'only-region' } });
        const stack2 = new lib_1.Stack(app, 'S2', { env: { account: 'only-account' } });
        test.deepEqual(stack1.resolve(stack1.account), { Ref: 'AWS::AccountId' });
        test.deepEqual(stack1.resolve(stack1.region), 'only-region');
        test.deepEqual(stack2.resolve(stack2.account), 'only-account');
        test.deepEqual(stack2.resolve(stack2.region), { Ref: 'AWS::Region' });
        test.done();
    },
    'environment defaults': {
        'if "env" is not specified, it implies account/region agnostic'(test) {
            // GIVEN
            const app = new lib_1.App();
            // WHEN
            const stack = new lib_1.Stack(app, 'stack');
            // THEN
            test.deepEqual(stack.resolve(stack.account), { Ref: 'AWS::AccountId' });
            test.deepEqual(stack.resolve(stack.region), { Ref: 'AWS::Region' });
            test.deepEqual(app.synth().getStackByName(stack.stackName).environment, {
                account: 'unknown-account',
                region: 'unknown-region',
                name: 'aws://unknown-account/unknown-region',
            });
            test.done();
        },
        'only region is set'(test) {
            // GIVEN
            const app = new lib_1.App();
            // WHEN
            const stack = new lib_1.Stack(app, 'stack', { env: { region: 'explicit-region' } });
            // THEN
            test.deepEqual(stack.resolve(stack.account), { Ref: 'AWS::AccountId' });
            test.deepEqual(stack.resolve(stack.region), 'explicit-region');
            test.deepEqual(app.synth().getStackByName(stack.stackName).environment, {
                account: 'unknown-account',
                region: 'explicit-region',
                name: 'aws://unknown-account/explicit-region',
            });
            test.done();
        },
        'both "region" and "account" are set'(test) {
            // GIVEN
            const app = new lib_1.App();
            // WHEN
            const stack = new lib_1.Stack(app, 'stack', { env: {
                    account: 'explicit-account',
                    region: 'explicit-region',
                } });
            // THEN
            test.deepEqual(stack.resolve(stack.account), 'explicit-account');
            test.deepEqual(stack.resolve(stack.region), 'explicit-region');
            test.deepEqual(app.synth().getStackByName(stack.stackName).environment, {
                account: 'explicit-account',
                region: 'explicit-region',
                name: 'aws://explicit-account/explicit-region',
            });
            test.done();
        },
        'token-account and token-region'(test) {
            // GIVEN
            const app = new lib_1.App();
            // WHEN
            const stack = new lib_1.Stack(app, 'stack', {
                env: {
                    account: lib_1.Aws.ACCOUNT_ID,
                    region: lib_1.Aws.REGION,
                },
            });
            // THEN
            test.deepEqual(stack.resolve(stack.account), { Ref: 'AWS::AccountId' });
            test.deepEqual(stack.resolve(stack.region), { Ref: 'AWS::Region' });
            test.deepEqual(app.synth().getStackByName(stack.stackName).environment, {
                account: 'unknown-account',
                region: 'unknown-region',
                name: 'aws://unknown-account/unknown-region',
            });
            test.done();
        },
        'token-account explicit region'(test) {
            // GIVEN
            const app = new lib_1.App();
            // WHEN
            const stack = new lib_1.Stack(app, 'stack', {
                env: {
                    account: lib_1.Aws.ACCOUNT_ID,
                    region: 'us-east-2',
                },
            });
            // THEN
            test.deepEqual(stack.resolve(stack.account), { Ref: 'AWS::AccountId' });
            test.deepEqual(stack.resolve(stack.region), 'us-east-2');
            test.deepEqual(app.synth().getStackByName(stack.stackName).environment, {
                account: 'unknown-account',
                region: 'us-east-2',
                name: 'aws://unknown-account/us-east-2',
            });
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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