"use strict";
const lib_1 = require("../lib");
function floatEqual(test, actual, expected) {
    test.ok(
    // Floats are subject to rounding errors up to Number.ESPILON
    actual >= expected - Number.EPSILON && actual <= expected + Number.EPSILON, `${actual} == ${expected}`);
}
module.exports = {
    'negative amount'(test) {
        test.throws(() => lib_1.Size.kibibytes(-1), /negative/);
        test.done();
    },
    'unresolved amount'(test) {
        const stack = new lib_1.Stack();
        const lazySize = lib_1.Size.kibibytes(lib_1.Token.asNumber({ resolve: () => 1337 }));
        test.equals(stack.resolve(lazySize.toKibibytes()), 1337);
        test.throws(() => stack.resolve(lazySize.toMebibytes()), /Unable to perform time unit conversion on un-resolved token/);
        test.done();
    },
    'Size in kibibytes'(test) {
        const size = lib_1.Size.kibibytes(4294967296);
        test.equal(size.toKibibytes(), 4294967296);
        test.equal(size.toMebibytes(), 4194304);
        test.equal(size.toGibibytes(), 4096);
        test.equal(size.toTebibytes(), 4);
        test.throws(() => size.toPebibytes(), /'4294967296 kibibytes' cannot be converted into a whole number/);
        floatEqual(test, size.toPebibytes({ rounding: lib_1.SizeRoundingBehavior.NONE }), 4294967296 / (1024 * 1024 * 1024 * 1024));
        test.equal(lib_1.Size.kibibytes(4 * 1024 * 1024).toGibibytes(), 4);
        test.done();
    },
    'Size in mebibytes'(test) {
        const size = lib_1.Size.mebibytes(4194304);
        test.equal(size.toKibibytes(), 4294967296);
        test.equal(size.toMebibytes(), 4194304);
        test.equal(size.toGibibytes(), 4096);
        test.equal(size.toTebibytes(), 4);
        test.throws(() => size.toPebibytes(), /'4194304 mebibytes' cannot be converted into a whole number/);
        floatEqual(test, size.toPebibytes({ rounding: lib_1.SizeRoundingBehavior.NONE }), 4194304 / (1024 * 1024 * 1024));
        test.equal(lib_1.Size.mebibytes(4 * 1024).toGibibytes(), 4);
        test.done();
    },
    'Size in gibibyte'(test) {
        const size = lib_1.Size.gibibytes(5);
        test.equal(size.toKibibytes(), 5242880);
        test.equal(size.toMebibytes(), 5120);
        test.equal(size.toGibibytes(), 5);
        test.throws(() => size.toTebibytes(), /'5 gibibytes' cannot be converted into a whole number/);
        floatEqual(test, size.toTebibytes({ rounding: lib_1.SizeRoundingBehavior.NONE }), 5 / 1024);
        test.throws(() => size.toPebibytes(), /'5 gibibytes' cannot be converted into a whole number/);
        floatEqual(test, size.toPebibytes({ rounding: lib_1.SizeRoundingBehavior.NONE }), 5 / (1024 * 1024));
        test.equal(lib_1.Size.gibibytes(4096).toTebibytes(), 4);
        test.done();
    },
    'Size in tebibyte'(test) {
        const size = lib_1.Size.tebibytes(5);
        test.equal(size.toKibibytes(), 5368709120);
        test.equal(size.toMebibytes(), 5242880);
        test.equal(size.toGibibytes(), 5120);
        test.equal(size.toTebibytes(), 5);
        test.throws(() => size.toPebibytes(), /'5 tebibytes' cannot be converted into a whole number/);
        floatEqual(test, size.toPebibytes({ rounding: lib_1.SizeRoundingBehavior.NONE }), 5 / 1024);
        test.equal(lib_1.Size.tebibytes(4096).toPebibytes(), 4);
        test.done();
    },
    'Size in pebibyte'(test) {
        const size = lib_1.Size.pebibyte(5);
        test.equal(size.toKibibytes(), 5497558138880);
        test.equal(size.toMebibytes(), 5368709120);
        test.equal(size.toGibibytes(), 5242880);
        test.equal(size.toTebibytes(), 5120);
        test.equal(size.toPebibytes(), 5);
        test.done();
    },
    'rounding behavior'(test) {
        const size = lib_1.Size.mebibytes(5200);
        test.throws(() => size.toGibibytes(), /cannot be converted into a whole number/);
        test.throws(() => size.toGibibytes({ rounding: lib_1.SizeRoundingBehavior.FAIL }), /cannot be converted into a whole number/);
        test.equals(size.toGibibytes({ rounding: lib_1.SizeRoundingBehavior.FLOOR }), 5);
        test.equals(size.toTebibytes({ rounding: lib_1.SizeRoundingBehavior.FLOOR }), 0);
        floatEqual(test, size.toKibibytes({ rounding: lib_1.SizeRoundingBehavior.FLOOR }), 5324800);
        test.equals(size.toGibibytes({ rounding: lib_1.SizeRoundingBehavior.NONE }), 5.078125);
        test.equals(size.toTebibytes({ rounding: lib_1.SizeRoundingBehavior.NONE }), 5200 / (1024 * 1024));
        test.equals(size.toKibibytes({ rounding: lib_1.SizeRoundingBehavior.NONE }), 5324800);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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