"use strict";
const cfn_resource_1 = require("../lib/cfn-resource");
const tag_manager_1 = require("../lib/tag-manager");
module.exports = {
    'TagManagerOptions can set tagPropertyName'(test) {
        const tagPropName = 'specialName';
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.MAP, 'Foo', undefined, { tagPropertyName: tagPropName });
        test.deepEqual(mgr.tagPropertyName, tagPropName);
        test.done();
    },
    '#setTag() supports setting a tag regardless of Type'(test) {
        const notTaggable = new tag_manager_1.TagManager(cfn_resource_1.TagType.NOT_TAGGABLE, 'AWS::Resource::Type');
        notTaggable.setTag('key', 'value');
        test.deepEqual(notTaggable.renderTags(), undefined);
        test.done();
    },
    'when a tag does not exist': {
        '#removeTag() does not throw an error'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            test.doesNotThrow(() => (mgr.removeTag('dne', 0)));
            test.done();
        },
        '#setTag() creates the tag'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            mgr.setTag('dne', 'notanymore');
            test.deepEqual(mgr.renderTags(), [{ key: 'dne', value: 'notanymore' }]);
            test.done();
        },
    },
    'when a tag does exist': {
        '#removeTag() deletes the tag'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            mgr.setTag('dne', 'notanymore', 0);
            mgr.removeTag('dne', 0);
            test.deepEqual(mgr.renderTags(), undefined);
            test.done();
        },
        '#setTag() overwrites the tag'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            mgr.setTag('dne', 'notanymore');
            mgr.setTag('dne', 'iwin');
            test.deepEqual(mgr.renderTags(), [{ key: 'dne', value: 'iwin' }]);
            test.done();
        },
    },
    'when there are no tags': {
        '#renderTags() returns undefined'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            test.deepEqual(mgr.renderTags(), undefined);
            test.done();
        },
        '#hasTags() returns false'(test) {
            const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
            test.equal(mgr.hasTags(), false);
            test.done();
        },
    },
    '#renderTags() handles standard, map, keyValue, and ASG tag formats'(test) {
        const tagged = [];
        const standard = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
        const asg = new tag_manager_1.TagManager(cfn_resource_1.TagType.AUTOSCALING_GROUP, 'AWS::Resource::Type');
        const keyValue = new tag_manager_1.TagManager(cfn_resource_1.TagType.KEY_VALUE, 'AWS::Resource::Type');
        const mapper = new tag_manager_1.TagManager(cfn_resource_1.TagType.MAP, 'AWS::Resource::Type');
        tagged.push(standard);
        tagged.push(asg);
        tagged.push(keyValue);
        tagged.push(mapper);
        for (const res of tagged) {
            res.setTag('foo', 'bar');
            res.setTag('asg', 'only', 0, false);
        }
        test.deepEqual(standard.renderTags(), [
            { key: 'foo', value: 'bar' },
            { key: 'asg', value: 'only' },
        ]);
        test.deepEqual(asg.renderTags(), [
            { key: 'foo', value: 'bar', propagateAtLaunch: true },
            { key: 'asg', value: 'only', propagateAtLaunch: false },
        ]);
        test.deepEqual(keyValue.renderTags(), [
            { Key: 'foo', Value: 'bar' },
            { Key: 'asg', Value: 'only' },
        ]);
        test.deepEqual(mapper.renderTags(), {
            foo: 'bar',
            asg: 'only',
        });
        test.done();
    },
    'when there are tags it hasTags returns true'(test) {
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
        mgr.setTag('key', 'myVal', 2);
        mgr.setTag('key', 'newVal', 1);
        test.equal(mgr.hasTags(), true);
        test.done();
    },
    'tags with higher or equal priority always take precedence'(test) {
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Resource::Type');
        mgr.setTag('key', 'myVal', 2);
        mgr.setTag('key', 'newVal', 1);
        test.deepEqual(mgr.renderTags(), [
            { key: 'key', value: 'myVal' },
        ]);
        mgr.removeTag('key', 1);
        test.deepEqual(mgr.renderTags(), [
            { key: 'key', value: 'myVal' },
        ]);
        mgr.removeTag('key', 2);
        test.deepEqual(mgr.renderTags(), undefined);
        test.done();
    },
    'excludeResourceTypes only tags resources that do not match'(test) {
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Fake::Resource');
        test.equal(false, mgr.applyTagAspectHere([], ['AWS::Fake::Resource']));
        test.equal(true, mgr.applyTagAspectHere([], ['AWS::Wrong::Resource']));
        test.done();
    },
    'includeResourceTypes only tags resources that match'(test) {
        const mgr = new tag_manager_1.TagManager(cfn_resource_1.TagType.STANDARD, 'AWS::Fake::Resource');
        test.equal(true, mgr.applyTagAspectHere(['AWS::Fake::Resource'], []));
        test.equal(false, mgr.applyTagAspectHere(['AWS::Wrong::Resource'], []));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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