Официальный Python SDK для легкой интеграции вашего Telegram-бота с рекламной платформой **AnyAds.online**. Начните монетизировать вашу аудиторию уже сегодня!

## 🚀 Быстрый старт

Этот гайд поможет вам интегрировать SDK и подготовить вашего бота к заработку менее чем за 5 минут.

### Шаг 1: Установка

Установите пакет, используя `pip`. Если вы работаете с фреймворком **Aiogram**, мы настоятельно рекомендуем установить SDK с дополнительными зависимостями для упрощенной интеграции.

```bash
# Для Aiogram
pip install "anyads[aiogram]"

# Стандартная установка (для других фреймворков)
pip install anyads
```

### Шаг 2: Получение API Key

1.  Зарегистрируйтесь на [**anyads.online**](https://anyads.online).
2.  Перейдите в раздел "Площадки" и добавьте вашего бота, указав его `@username`.
3.  Для вашей новой площадки будет сгенерирован уникальный **`API_KEY`**, начинающийся с `anyads_...`. Скопируйте его. Он понадобится на следующем шаге.

### Шаг 3: Интеграция в код (Пример для Aiogram 3.x)

В главном файле вашего бота, где происходит его запуск, добавьте следующий код:

```python
import asyncio
import os
import logging
from aiogram import Bot, Dispatcher, types

# 1. Импортируем AnyAds SDK
import anyads

# --- Конфигурация ---
# Рекомендуется хранить ключи в переменных окружения, а не в коде
ANYADS_API_KEY = os.getenv("ANYADS_API_KEY")
TELEGRAM_BOT_TOKEN = os.getenv("TELEGRAM_BOT_TOKEN")

# Настраиваем базовое логирование для отладки
logging.basicConfig(level=logging.INFO)

# --- Инициализация Aiogram ---
dp = Dispatcher()
bot = Bot(TELEGRAM_BOT_TOKEN)

# 2. Инициализируем SDK AnyAds
# SDK автоматически запустит фоновый опрос рекламных задач
sdk = anyads.init(api_key=ANYADS_API_KEY)

# 3. Регистрируем обработчики AnyAds (для верификации)
# Эта простая строчка позволит нашим модераторам верифицировать вашего бота
anyads.integrations.aiogram.register_anyads_handlers(dp)


# 4. Создаем обработчик для полученной рекламы
@sdk.on_broadcast_received
async def handle_ad_task(task_data: dict):
    """
    Эта функция будет автоматически вызвана SDK, когда с сервера
    придет новая задача на рассылку.
    """
    logging.info(f"Получена новая задача на рассылку: {task_data.get('task_id')}")

    creative = task_data.get("creative", {}).get("content", {})
    text = creative.get("text")

    if not text:
        logging.error("Ошибка: в задаче отсутствует текст для рассылки.")
        return

    # !!! ВАЖНО: Здесь должна быть ваша логика получения списка ID пользователей
    # user_ids = await get_all_active_users_from_your_database()
    user_ids = [] # Заглушка, замените на реальный код

    sent_count = 0
    for user_id in user_ids:
        try:
            await bot.send_message(user_id, text)
            sent_count += 1
            await asyncio.sleep(0.05)
        except Exception as e:
            logging.warning(f"Не удалось отправить сообщение пользователю {user_id}: {e}")

    logging.info(f"Рассылка завершена. Отправлено {sent_count} из {len(user_ids)} сообщений.")


# --- Основная функция запуска ---
async def main():
    # Запускаем фоновый опрос AnyAds
    await sdk.start()

    # Запускаем бота
    try:
        await dp.start_polling(bot)
    finally:
        # Корректно останавливаем SDK при завершении работы бота
        await sdk.stop()
        logging.info("Бот и AnyAds SDK остановлены.")

if __name__ == "__main__":
    if not ANYADS_API_KEY or not TELEGRAM_BOT_TOKEN:
        raise RuntimeError("Необходимо установить переменные окружения ANYADS_API_KEY и TELEGRAM_BOT_TOKEN")

    try:
        asyncio.run(main())
    except KeyboardInterrupt:
        print("Бот остановлен вручную.")

```

### 4. Верификация

После того как вы запустите бота с интегрированным SDK, его статус в личном кабинете [**AnyAds.online**](https://anyads.online) будет "Ожидает верификации". Наша команда модераторов проверит интеграцию и активирует вашу площадку. После этого ваш бот начнет получать рекламные задачи.

## ⚙️ Универсальная интеграция (для других фреймворков)

Если вы не используете `aiogram`, вы можете легко интегрировать SDK вручную.

1.  **Установите ядро SDK:** `pip install anyads`
2.  **Инициализируйте и запустите SDK**, как в примере выше (`anyads.init(...)`, `await sdk.start()`).
3.  **Обработайте команду верификации:** Вам нужно самостоятельно поймать сообщение, начинающееся с `/verify_anyads_`, и передать его в SDK.

```python
# Пример обработчика
async def handle_messages(message):
    if message.text and message.text.startswith('/verify_anyads_'):
        sdk = anyads.get_sdk_instance()
        success = await sdk.process_verification_code(message.text)
        if success:
            # Ответить пользователю об успехе
            pass
```

## ❓ Нужна помощь?

Если у вас возникли вопросы, обратитесь в нашу поддержку или изучите полную документацию на [**docs.anyads.online**](https://docs.anyads.online).
