"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithCondition = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const operators_1 = require("../operators");
const _1_base_1 = require("./1-base");
/**
 * Adds "condition" functionality to the Policy Statement
 */
class PolicyStatementWithCondition extends _1_base_1.PolicyStatementBase {
    constructor() {
        super(...arguments);
        this.floydConditions = {};
        this.cdkConditionsApplied = false;
    }
    /**
     * Injects conditions into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyConditions();
            return super.toJSON();
        }
        const statement = super.toJSON();
        if (this.hasConditions()) {
            statement.Condition = this.floydConditions;
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyConditions();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    freeze() {
        // @ts-ignore only available after swapping 1-base
        if (!this.frozen) {
            this.cdkApplyConditions();
        }
        // @ts-ignore only available after swapping 1-base
        return super.freeze();
    }
    cdkApplyConditions() {
        if (this.hasConditions() && !this.cdkConditionsApplied) {
            Object.keys(this.floydConditions).forEach((operator) => {
                Object.keys(this.floydConditions[operator]).forEach((key) => {
                    const condition = {};
                    condition[key] = this.floydConditions[operator][key];
                    // @ts-ignore only available after swapping 1-base
                    this.addCondition(operator, condition);
                });
            });
            this.cdkConditionsApplied = true;
        }
    }
    /**
     * Checks weather a condition was applied to the policy.
     */
    hasConditions() {
        return Object.keys(this.floydConditions).length > 0;
    }
    /**
     * Adds a condition to the statement
     *
     * @param key The key of the condition
     * @param value The value(s) to check for
     * @param operator [Operator](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html) of the condition. **Default:** `StringLike`
     */
    if(key, value, operator) {
        if (this.servicePrefix.length && key.indexOf(':') < 0) {
            key = this.servicePrefix + ':' + key;
        }
        if (typeof operator === 'undefined') {
            operator = new operators_1.Operator().stringLike();
        }
        var op = '';
        if (typeof operator === 'string') {
            op = operator;
        }
        else {
            op = operator.toString();
        }
        // For boolean/number operators, IAM accepts both, booleans/numbers and
        // their string representation. To be consistent with how the IAM console
        // displays/generates the values, we convert them to strings.
        if (['boolean', 'number'].includes(typeof value)) {
            value = value.toString();
        }
        if (!(op in this.floydConditions)) {
            this.floydConditions[op] = {};
        }
        this.floydConditions[op][key] = value;
        return this;
    }
    /**
     * Compare the services with the services that made requests on behalf of the IAM principal (user or role). When a principal makes a request to an AWS service, that service might use the principal's credentials to make subsequent requests to other services.
     *
     * The `aws:CalledVia` key contains an ordered list of each service in the chain that made requests on the principal's behalf.
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ForAnyValue:StringEquals`
     */
    ifAwsCalledVia(value, operator) {
        return this.if('aws:CalledVia', value, operator || new operators_1.Operator().forAnyValue().stringEquals());
    }
    /**
     * Compare the services with the first service that made a request on behalf of the IAM principal (user or role). For more information, see [aws:CalledVia](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia).
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledviafirst
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsCalledViaFirst(value, operator) {
        return this.if('aws:CalledViaFirst', value, operator);
    }
    /**
     * Compare the services with the last service that made a request on behalf of the IAM principal (user or role). For more information, see [aws:CalledVia](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia).
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvialast
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsCalledViaLast(value, operator) {
        return this.if('aws:CalledViaLast', value, operator);
    }
    /**
     * Compare the date and time of the request with the date and time that you specify. To view an example policy that uses this condition key, see [AWS: Allows Access Within Specific Dates](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_examples_aws-dates.html).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-currenttime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateLessThanEquals`
     */
    ifAwsCurrentTime(value, operator) {
        if (typeof value.getMonth === 'function') {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === 'function') {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if('aws:CurrentTime', value, operator || new operators_1.Operator().dateLessThanEquals());
    }
    /**
     * This key identifies the VPC to which Amazon EC2 IAM role credentials were delivered to. You can use this key in a policy with the [aws:SourceVPC](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcevpc) global key to check if a call is made from a VPC (`aws:SourceVPC`) that matches the VPC where a credential was delivered to (`aws:Ec2InstanceSourceVpc`).
     *
     * **Availability:** This key is included in the request context whenever the requester is signing requests with an Amazon EC2 role credential. It can be used in IAM policies, service control policies, VPC endpoint policies, and resource policies.
     *
     * This key can be used with VPC identifier values, but is most useful when used as a variable combined with the `aws:SourceVpc` context key. The `aws:SourceVpc` context key is included in the request context only if the requester uses a VPC endpoint to make the request. Using `aws:Ec2InstanceSourceVpc` with `aws:SourceVpc` allows you to use `aws:Ec2InstanceSourceVpc` more broadly since it compares values that typically change together.
     *
     * **Note:** This condition key is not available in EC2-Classic.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-ec2instancesourcevpc
     *
     * @param value The VPS ID
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsEc2InstanceSourceVpc(value, operator) {
        return this.if('aws:Ec2InstanceSourceVpc', value, operator);
    }
    /**
     * This key identifies the private IPv4 address of the primary elastic network interface to which Amazon EC2 IAM role credentials were delivered. You must use this condition key with its companion key `aws:Ec2InstanceSourceVpc` to ensure that you have a globally unique combination of VPC ID and source private IP. Use this key with `aws:Ec2InstanceSourceVpc` to ensure that a request was made from the same private IP address that the credentials were delivered to.
     *
     * **Availability:** This key is included in the request context whenever the requester is signing requests with an Amazon EC2 role credential. It can be used in IAM policies, service control policies, VPC endpoint policies, and resource policies.
     *
     * **Note:** This condition key is not available in EC2-Classic.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-vpcsourceip
     *
     * @param value The private IPv4 address
     * @param operator Works with IP [address operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IPAddress). **Default:** `IpAddress`
     */
    ifAwsEc2InstanceSourcePrivateIPv4(value, operator) {
        return this.if('aws:Ec2InstanceSourcePrivateIPv4', value, operator || new operators_1.Operator().ipAddress());
    }
    /**
     * Compare the date and time of the request in epoch or Unix time with the value that you specify. This key also accepts the number of seconds since January 1, 1970.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-epochtime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date) and [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `DateLessThanEquals`
     */
    ifAwsEpochTime(value, operator) {
        if (typeof value.getMonth === 'function') {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === 'function') {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if('aws:EpochTime', value, operator || new operators_1.Operator().dateLessThanEquals());
    }
    /**
     * Use this key to compare the principal's issuing identity provider (IdP) with the IdP that you specify in the policy. This means that an IAM role was assumed using the `AssumeRoleWithWebIdentity` or `AssumeRoleWithSAML` AWS STS operations. When the resulting role session's temporary credentials are used to make a request, the request context identifies the IdP that authenticated the original federated identity.
     *
     * **Availability:** This key is present when the principal is a role session principal and that session was issued using a third-party identity provider.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-federatedprovider
     *
     * @param value The principal's issuing identity provider (IdP)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsFederatedProvider(value, operator) {
        return this.if('aws:FederatedProvider', value, operator);
    }
    /**
     * Compare the number of seconds since the requesting principal was authorized using MFA with the number that you specify. For more information about MFA, see [Using Multi-Factor Authentication (MFA) in AWS](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_mfa.html).
     *
     * **Availability:** This key is included in the request context only if the principal was authenticated using MFA. If MFA was not used, this key is not present.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-multifactorauthage
     *
     * @param value Number of seconds
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericLessThan`
     */
    ifAwsMultiFactorAuthAge(value, operator) {
        return this.if('aws:MultiFactorAuthAge', value, operator || new operators_1.Operator().numericLessThan());
    }
    /**
     * Check whether multi-factor authentication (MFA) was used to validate the temporary security credentials that made the request.
     *
     * **Availability:** This key is included in the request context only when the principal uses temporary credentials to make the request. The key is not present in AWS CLI, AWS API, or AWS SDK requests that are made using long-term credentials.
     *
     * Temporary credentials are used to authenticate IAM roles, federated users, IAM users with temporary tokens from `sts:GetSessionToken`, and users of the AWS Management Console. IAM users in the AWS Management Console unknowingly use temporary credentials. Users sign into the console using their user name and password, which are long-term credentials. However, in the background, the console generates temporary credentials on behalf of the user. To learn which services support using temporary credentials, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html).
     *
     * The `aws:MultiFactorAuthPresent` key is not present when an API or CLI command is called with long-term credentials, such as user access key pairs. Therefore we recommend that when you check for this key that you use the [...IfExists](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IfExists) versions of the condition operators.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-multifactorauthpresent
     *
     * @param value Weather the MFA should be present or absent. **Default:** `true`
     */
    ifAwsMultiFactorAuthPresent(value) {
        return this.if(`aws:MultiFactorAuthPresent`, typeof value !== 'undefined' ? value : true, new operators_1.Operator().bool());
    }
    /**
     * Compare the account to which the requesting principal belongs with the account identifier that you specify.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalaccount
     *
     * @param value Account identifier(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalAccount(value, operator) {
        return this.if('aws:PrincipalAccount', value, operator);
    }
    /**
     * Compare the [Amazon Resource Name](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns) (ARN) of the principal that made the request with the ARN that you specify.
     *
     * For IAM roles, the request context returns the ARN of the role, not the ARN of the user that assumed the role. To learn which types of principals you can specify in this condition key, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalarn
     *
     * @param value Principle ARN(s)
     * @param operator Works with [ARN operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN) and [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ArnLike`
     */
    ifAwsPrincipalArn(value, operator) {
        return this.if('aws:PrincipalArn', value, operator || new operators_1.Operator().arnLike());
    }
    /**
     * Check whether the call to your resource is being made directly by an AWS [service principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#principal-services). For example, AWS CloudTrail uses the service principal `cloudtrail.amazonaws.com` to write logs to your Amazon S3 bucket. The request context key is set to true when a service uses a service principal to perform a direct action on your resources. The context key is set to false if the service uses the credentials of an IAM principal to make a request on the principal's behalf. It is also set to false if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf.
     *
     * **Availability:** This key is present in the request context for all signed API requests that use AWS credentials.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalisawsservice
     *
     * @param value Weather the call to your resource is being made directly by an AWS service principal. **Default:** `true`
     */
    ifAwsPrincipalIsAWSService(value) {
        return this.if(`aws:PrincipalIsAWSService`, typeof value !== 'undefined' ? value : true, new operators_1.Operator().bool());
    }
    /**
     * Compare the identifier of the organization in AWS Organizations to which the requesting principal belongs with the identifier you specify.
     *
     * **Availability:** This key is included in the request context only if the principal is a member of an organization.
     *
     * This global key provides an alternative to listing all the account IDs for all AWS accounts in an organization. You can use this condition key to simplify specifying the `Principal` element in a resource-based policy. You can specify the organization ID in the condition element. When you add and remove accounts, policies that include the `aws:PrincipalOrgID` key automatically include the correct accounts and don't require manual updating.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalorgid
     *
     * @param value Organization ID(s) in format `o-xxxxxxxxxxx`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalOrgID(value, operator) {
        return this.if('aws:PrincipalOrgID', value, operator);
    }
    /**
     * Compare the AWS Organizations path for the principal who is making the request to the path you provide. That principal can be an IAM user, IAM role, federated user, or AWS account root user.
     *
     * This condition ensures that the requester is an account member within the specified organization root or organizational units (OUs) in AWS Organizations. An AWS Organizations path is a text representation of the structure of an Organizations entity. For more information about using and understanding paths, see Understand the [AWS Organizations Entity Path](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor-view-data-orgs.html#access_policies_access-advisor-viewing-orgs-entity-path).
     *
     * **Availability:** This key is included in the request context only if the principal is a member of an organization.
     *
     * **Note:** Organization IDs are globally unique but OU IDs and root IDs are unique only within an organization. This means that no two organizations share the same organization ID. However, another organization might have an OU or root with the same ID as yours. We recommend that you always include the organization ID when you specify an OU or root.
     *
     * `aws:PrincipalOrgPaths` is a multivalued condition key. Multivalued keys include one or more values in a list format. The result is a logical `OR`. When you use multiple values with the `ForAnyValue:` condition operator, the principal's path must match one of the paths provided. For policies that include multiple values for a single key, you must enclose the conditions within brackets like an array (`"Key":["Value1", "Value2"]`). You should also include these brackets when there is a single value. For more information about multivalued condition keys, see [Creating a Condition with Multiple Keys or Values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalorgpaths
     *
     * @param value Organization path(s) in the format of `o-xxxxxxxxxxx/r-xxxxxxxxxx/ou-xxxx-xxxxxxxx/ou-xxxx-xxxxxxxx/`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsPrincipalOrgPaths(value, operator) {
        return this.if('aws:PrincipalOrgPaths', value, operator);
    }
    /**
     * Compare the [service principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#principal-services) name in the policy with the service principal that is making requests to your resources. You can use this key to check whether this call is made by a specific service principal. When a service principal makes a direct request to your resource, the `aws:PrincipalServiceName` key contains the name of the service principal. For example, the AWS CloudTrail service principal name is `cloudtrail.amazonaws.com`.
     *
     * **Availability:** This key is present in the request when the call is made by an AWS service principal. This key is not present in any other situation, including the following:
     *
     * - If the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf.
     * - If the service uses the credentials of an IAM principal to make a request on the principal's behalf.
     * - If the call is made directly by an IAM principal.
     *
     * You can use this condition key to limit access to your trusted identities and expected network locations, while safely granting access to an AWS service.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalservicename
     *
     * @param value AWS service name, e.g. `cloudtrail.amazonaws.com`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalServiceName(value, operator) {
        return this.if('aws:PrincipalServiceName', value, operator);
    }
    /**
     * This key provides a list of all [service principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#principal-services) names that belong to the service. This is an advanced condition key. You can use it to restrict the service from accessing your resource from a specific Region only. Some services may create Regional service principals to indicate a particular instance of the service within a specific Region. You can limit access to a resource to a particular instance of the service. When a service principal makes a direct request to your resource, the `aws:PrincipalServiceNamesList` contains an unordered list of all service principal names associated with the Regional instance of the service.
     *
     * **Availability:** This key is present in the request when the call is made by an AWS service principal. This key is not present in any other situation, including the following:
     *
     * - If the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf.
     * - If the service uses the credentials of an IAM principal to make a request on the principal's behalf.
     * - If the call is made directly by an IAM principal.
     *
     * `aws:PrincipalServiceNamesList` is a multivalued condition key. Multivalued keys include one or more values in a list format. The result is a logical `OR`. You must use the `ForAnyValue` or `ForAllValues` set operators with the `StringLike` [condition operator](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String) when you use this key. For policies that include multiple values for a single key, you must enclose the conditions within brackets like an array, such as `("Key":["Value1", "Value2"])`. You should also include these brackets when there is a single value. For more information about multivalued condition keys, see [Using multiple keys and values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html#reference_policies_multi-key-or-value-conditions).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalservicenameslist
     *
     * @param value AWS service name list
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ForAnyValue:StringLike`
     */
    ifAwsPrincipalServiceNamesList(value, operator) {
        return this.if('aws:PrincipalServiceNamesList', value, operator || new operators_1.Operator().stringLike().forAnyValue());
    }
    /**
     * Compare the tag attached to the principal making the request with the tag that you specify. If the principal has more than one tag attached, the request context includes one aws:PrincipalTag key for each attached tag key.
     *
     * **Availability:** This key is included in the request context if the principal is using an IAM user with attached tags. It is included for a principal using an IAM role with attached tags or [session tags](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_session-tags.html).
     *
     * You can add custom attributes to a user or role in the form of a key-value pair. For more information about IAM tags, see [Tagging IAM Users and Roles](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html). You can use `aws:PrincipalTag` to [control access](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html#access_iam-tags_control-resources) for AWS principals.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalTag(key, value, operator) {
        return this.if(`aws:PrincipalTag/${key}`, value, operator);
    }
    /**
     * Compare the type of principal making the request with the principal type that you specify. For details about how the information appears in the request context for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltype
     *
     * @param value The principal type(s). Any of `Account`, `User`, `FederatedUser`, `AssumedRole`, `Anonymous`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsPrincipalType(value, operator) {
        return this.if('aws:PrincipalType', value, operator);
    }
    /**
     * Compare who referred the request in the client browser with the referer that you specify. The `aws:referer` request context value is provided by the caller in an HTTP header.
     *
     * **Availability:** This key is included in the request context only if the request was invoked using a URL in the browser.
     *
     * For example, you can call [Amazon S3 API operations directly using a web browser](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies.html#example-bucket-policies-use-case-4). This means that you can view S3 objects, such as images and documents, directly through a web browser. The `aws:referer` condition allows you to restrict access to specific values in the HTTP or HTTPS request based on the value of the referrer header.
     *
     * **Warning:** This condition should be used carefully. It is dangerous to include a publicly known referer header value. Unauthorized parties can use modified or custom browsers to provide any `aws:referer` value that they choose. As a result, `aws:referer` should not be used to prevent unauthorized parties from making direct AWS requests. It is offered only to allow customers to protect their digital content, such as content stored in Amazon S3, from being referenced on unauthorized third-party sites.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-referer
     *
     * @param value The referer url(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsReferer(value, operator) {
        return this.if('aws:Referer', value, operator);
    }
    /**
     * Compare the AWS Region that was called in the request with the Region that you specify. You can use this global condition key to control which Regions can be requested. To view the AWS Regions for each service, see [Service endpoints and quotas](https://docs.aws.amazon.com/general/latest/gr/aws-service-information.html) in the Amazon Web Services General Reference.
     *
     * **Availability:** This key is always included in the request context.
     *
     * Some global services, such as IAM, have a single endpoint. Because this endpoint is physically located in the US East (N. Virginia) Region, IAM calls are always made to the us-east-1 Region. For example, if you create a policy that denies access to all services if the requested Region is not us-west-2, then IAM calls always fail. To view an example of how to work around this, see [NotAction with Deny](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * **Note:** The `aws:RequestedRegion` condition key allows you to control which endpoint of a service is invoked but does not control the impact of the operation. Some services have cross-Region impacts. For example, Amazon S3 has API operations that control cross-Region replication. You can invoke `s3:PutBucketReplication` in one Region (which is affected by the `aws:RequestedRegion` condition key), but other Regions are affected based on the replications configuration settings.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requestedregion
     *
     * @param value The region(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsRequestedRegion(value, operator) {
        return this.if('aws:RequestedRegion', value, operator);
    }
    /**
     * Compare the tag key-value pair that was passed in the request with the tag pair that you specify. For example, you could check whether the request includes the tag key `Dept` and that it has the value `Accounting`. For more information, see [Controlling Access During AWS Requests](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-requests).
     *
     * **Availability:** This key is included in the request context when tags are passed in the request. When multiple tags are passed in the request, there is one context key for each tag key-value pair.
     *
     * Because you can include multiple tag key-value pairs in a request, the request content could be a multivalued request. In this case, you should consider using the `ForAllValues` or `ForAnyValue` set operators. For more information, see [Using Multiple Keys and Values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html#reference_policies_multi-key-or-value-conditions).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(key, value, operator) {
        return this.if(`aws:RequestTag/${key}`, value, operator);
    }
    /**
     * Use this key to compare the requested resource owner's [AWS account ID](https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html) with the resource account in the policy. You can then allow or deny access to that resource based on the account that owns the resource.
     *
     * This key is equal to the AWS account ID for the account with the resources evaluated in the request.
     *
     * For most resources in your account, the ARN contains the owner account ID for that resource. For certain resources, such as Amazon S3 buckets, the resource ARN does not include the account ID. The following two examples show the difference between a resource with an account ID in the ARN, and an Amazon S3 ARN without an account ID:
     *
     * - `arn:aws:iam::123456789012:role/AWSExampleRole` - IAM role created and owned within the account 123456789012.
     * - `arn:aws:s3:::DOC-EXAMPLE-BUCKET2` - Amazon S3 bucket created and owned within the account 111122223333, not displayed in the ARN.
     *
     * **Availability:** This key is always included in the request context for most service actions. The following actions don't support this key:
     *
     *   - Amazon Elastic Block Store - All actions
     *   - Amazon EC2
     *     - `ec2:CopyFpgaImage`
     *     - `ec2:CopyImage`
     *     - `ec2:CopySnapshot`
     *     - `ec2:CreateTransitGatewayPeeringAttachment`
     *     - `ec2:CreateVolume`
     *     - `ec2:CreateVpcPeeringConnection`
     *   - Amazon EventBridge - All actions
     *   - Amazon WorkSpaces
     *     - `workspaces:CopyWorkspaceImage`
     *     - `workspaces:DescribeWorkspaceImages`
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourceaccount
     *
     * @param value The account ID
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceAccount(value, operator) {
        return this.if('aws:ResourceAccount', value, operator);
    }
    /**
     * Use this key to compare the identifier of the organization in AWS Organizations to which the requested resource belongs with the identifier specified in the policy.
     *
     * This global key returns the resource organization ID for a given request. It allows you to create rules that apply to all resources in an organization that are specified in the Resource element of an [identity-based policy](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_identity-vs-resource.html). You can specify the [organization ID](https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_org_details.html) in the condition element. When you add and remove accounts, policies that include the aws:ResourceOrgID key automatically include the correct accounts and you don't have to manually update it.
     *
     * **Note:** Some AWS services require access to AWS owned resources that are hosted in another AWS account. Using `aws:ResourceOrgID` in your identity-based policies might impact your identity's ability to access these resources.
     *
     * **Availability:** This key is included in the request context only if the account that owns the resource is a member of an organization. This global condition key does not support the following actions:
     *
     * - Amazon Elastic Block Store - All actions
     * - Amazon EC2
     *   - `ec2:CopyFpgaImage`
     *   - `ec2:CopyImage`
     *   - `ec2:CopySnapshot`
     *   - `ec2:CreateTransitGatewayPeeringAttachment`
     *   - `ec2:CreateVolume`
     *   - `ec2:CreateVpcPeeringConnection`
     * - Amazon EventBridge - All actions
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourceorgid
     *
     * @param value ID of an organization
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceOrgID(value, operator) {
        return this.if('aws:ResourceOrgID', value, operator);
    }
    /**
     * Use this key to compare the AWS Organizations path for the accessed resource to the path in the policy. In a policy, this condition key ensures that the resource belongs to an account member within the specified organization root or organizational units (OUs) in AWS Organizations. An AWS Organizations path is a text representation of the structure of an Organizations entity. For more information about using and understanding paths, see [Understand the AWS Organizations entity path](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor-view-data-orgs.html#access_policies_access-advisor-viewing-orgs-entity-path).
     *
     * `aws:ResourceOrgPaths` is a multivalued condition key. Multivalued keys can have multiple values in the request context. You must use the `ForAnyValue` or `ForAllValues` set operators with [string condition operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String) for this key. For more information about multivalued condition keys, see [Using multiple keys and values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html#reference_policies_multi-key-or-value-conditions).
     *
     * **Note:** Some AWS services require access to AWS owned resources that are hosted in another AWS account. Using aws:ResourceOrgPaths in your identity-based policies might impact your identity's ability to access these resources.
     *
     * **Availability:** This key is included in the request context only if the account that owns the resource is a member of an organization. This global condition key does not support the following actions:
     *
     * - Amazon Elastic Block Store - All actions
     * - Amazon EC2
     *   - `ec2:CopyFpgaImage`
     *   - `ec2:CopyImage`
     *   - `ec2:CopySnapshot`
     *   - `ec2:CreateTransitGatewayPeeringAttachment`
     *   - `ec2:CreateVolume`
     *   - `ec2:CreateVpcPeeringConnection`
     * - Amazon EventBridge - All actions
     * - Amazon WorkSpaces
     *   - `workspaces:CopyWorkspaceImage`
     *   - `workspaces:DescribeWorkspaceImages`
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourceorgpaths
     *
     * @param value The path of an organization
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceOrgPaths(value, operator) {
        return this.if('aws:ResourceOrgPaths', value, operator);
    }
    /**
     * Compare the tag key-value pair that you specify with the key-value pair that is attached to the resource. For example, you could require that access to a resource is allowed only if the resource has the attached tag key `Dept` with the value `Marketing`. For more information, see [Controlling Access to AWS Resources](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-resources).
     *
     * **Availability:** This key is included in the request context when the requested resource already has attached tags. This key is returned only for resources that [support authorization based on tags](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html). There is one context key for each tag key-value pair.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(key, value, operator) {
        return this.if(`aws:ResourceTag/${key}`, value, operator);
    }
    /**
     * Check whether the request was sent using SSL. The request context returns `true` or `false`. In a policy, you can allow specific actions only if the request is sent using SSL.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-securetransport
     *
     * @param value Weather request was sent using SSL. **Default:** `true`
     */
    ifAwsSecureTransport(value) {
        return this.if(`aws:SecureTransport`, typeof value !== 'undefined' ? value : true, new operators_1.Operator().bool());
    }
    /**
     * Compare the account ID of the resource making a service-to-service request with the account ID that you specify.
     *
     * **Availability:** This key is included in the request context only if accessing a resource triggers an AWS service to call another service on behalf of the resource owner. The calling service must pass the resource ARN of the source to the called service. This ARN includes the source account ID.
     *
     * You can use this condition key to check that Amazon S3 is not being used as a [confused deputy](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html#confused-deputy). For example, when an Amazon S3 bucket update triggers an Amazon SNS topic post, the Amazon S3 service invokes the `sns:Publish` API operation. The bucket is considered the source of the SNS request and the value of the key is the account ID from the bucket's ARN.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourceaccount
     *
     * @param value The account ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsSourceAccount(value, operator) {
        return this.if('aws:SourceAccount', value, operator);
    }
    /**
     * Compare the [Amazon Resource Name (ARN)](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns) of the resource making a service-to-service request with the ARN that you specify.
     *
     * This key does not work with the ARN of the principal making the request. Instead, use [aws:PrincipalArn](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalarn). The source's ARN includes the account ID, so it is not necessary to use `aws:SourceAccount` with `aws:SourceArn`.
     *
     * **Availability:** This key is included in the request context only if accessing a resource triggers an AWS service to call another service on behalf of the resource owner. The calling service must pass the ARN of the original resource to the called service.
     *
     * You can use this condition key to check that Amazon S3 is not being used as a [confused deputy](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html#confused-deputy). For example, when an Amazon S3 bucket update triggers an Amazon SNS topic post, the Amazon S3 service invokes the `sns:Publish` API operation. The bucket is considered the source of the SNS request and the value of the key is the bucket's ARN.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcearn
     *
     * @param value The source ARN(s)
     * @param operator Works with [ARN operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN) and [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ArnLike`
     */
    ifAwsSourceArn(value, operator) {
        return this.if('aws:SourceArn', value, operator || new operators_1.Operator().arnLike());
    }
    /**
     * Compare the source identity that was set by the principal with the source identity that you specify in the policy.
     *
     * **Availability:** This key is included in the request context after a source identity has been set when a role is assumed using any AWS STS assume-role CLI command, or AWS STS `AssumeRole` API operation.
     *
     * You can use this key in a policy to allow actions in AWS by principals that have set a source identity when assuming a role. Activity for the role's specified source identity appears in [AWS CloudTrail](https://docs.aws.amazon.com/IAM/latest/UserGuide/cloudtrail-integration.html#cloudtrail-integration_signin-tempcreds). This makes it easier for administrators to determine who or what performed actions with a role in AWS.
     *
     * Unlike [sts:RoleSessionName](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_rolesessionname), after the source identity is set, the value cannot be changed. It is present in the request context for all actions taken by the role. The value persists into subsequent role sessions when you use the session credentials to assume another role. Assuming one role from another is called [role chaining](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-role-chaining).
     *
     * The [sts:SourceIdentity](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_sourceidentity) key is present in the request when the principal initially sets a source identity while assuming a role using any AWS STS assume-role CLI command, or AWS STS `AssumeRole` API operation. The `aws:SourceIdentity` key is present in the request for any actions that are taken with a role session that has a source identity set.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourceidentity
     *
     * @param value The source identity
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsSourceIdentity(value, operator) {
        return this.if('aws:SourceIdentity', value, operator);
    }
    /**
     * Compare the requester's IP address with the IP address that you specify.
     *
     * **Availability:** This key is included in the request context, except when the requester uses a VPC endpoint to make the request.
     *
     * The `aws:SourceIp` condition key can be used in a policy to allow principals to make requests only from within a specified IP range. However, this policy denies access if an AWS service makes calls on the principal's behalf. In this case, you can use `aws:SourceIp` with the [aws:ViaAWSService](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-viaawsservice) key to ensure that the source IP restriction applies only to requests made directly by a principal.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourceip
     *
     * @param value The source IP(s)
     * @param operator Works with IP [address operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IPAddress). **Default:** `IpAddress`
     */
    ifAwsSourceIp(value, operator) {
        return this.if('aws:SourceIp', value, operator || new operators_1.Operator().ipAddress());
    }
    /**
     * Check whether the request comes from the VPC that you specify. In a policy, you can use this condition to allow access to only a specific VPC. For more information, see [Restricting Access to a Specific VPC](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies-vpc-endpoint.html#example-bucket-policies-restrict-access-vpc) in the *Amazon Simple Storage Service Developer Guide*.
     *
     * **Availability:** This key is included in the request context only if the requester uses a VPC endpoint to make the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcevpc
     *
     * @param value The VPS ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsSourceVpc(value, operator) {
        return this.if('aws:SourceVpc', value, operator);
    }
    /**
     * Compare the VPC endpoint identifier of the request with the endpoint ID that you specify. In a policy, you can use this condition to restrict access to a specific VPC endpoint. For more information, see [Restricting Access to a Specific VPC Endpoint](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies-vpc-endpoint.html#example-bucket-policies-restrict-access-vpc-endpoint) in the *Amazon Simple Storage Service Developer Guide*.
     *
     * **Availability:** This key is included in the request context only if the requester uses a VPC endpoint to make the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcevpce
     *
     * @param value The VPC Endpoint ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsSourceVpce(value, operator) {
        return this.if('aws:SourceVpce', value, operator);
    }
    /**
     * Compare the tag keys in a request with the keys that you specify. As a best practice when you use policies to control access using tags, use the `aws:TagKeys` condition key to define what tag keys are allowed. For example policies and more information, see [Controlling Access Based on Tag Keys](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-tag-keys).
     *
     * **Availability:** This key is included in the request context only if the operation supports attaching tags to resources.
     *
     * Because you can include multiple tag key-value pairs in a request, the request content could be a [multivalued](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html) request. In this case, you should consider using the `ForAllValues` or `ForAnyValue` set operators. For more information, see Using Multiple Keys and Values.
     *
     * Some services support tagging with resource operations, such as creating, modifying, or deleting a resource. To allow tagging and operations as a single call, you must create a policy that includes both the tagging action and the resource-modifying action. You can then use the `aws:TagKeys` condition key to enforce using specific tag keys in the request. For example, to limit tags when someone creates an Amazon EC2 snapshot, you must include the `ec2:CreateSnapshot` creation action ***and*** the `ec2:CreateTags` tagging action in the policy. To view a policy for this scenario that uses `aws:TagKeys`, see [Creating a Snapshot with Tags](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ExamplePolicies_EC2.html#iam-creating-snapshot-with-tags) in the *Amazon EC2 User Guide for Linux Instances*.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * @param value The tag key(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if('aws:TagKeys', value, operator);
    }
    /**
     * Compare the date and time that temporary security credentials were issued with the date and time that you specify.
     *
     * **Availability:** This key is included in the request context only when the principal uses temporary credentials to make the request. They key is not present in AWS CLI, AWS API, or AWS SDK requests that are made using access keys.
     *
     * To learn which services support using temporary credentials, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tokenissuetime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateGreaterThanEquals`
     */
    ifAwsTokenIssueTime(value, operator) {
        return this.if('aws:TokenIssueTime', dateToString(value), operator || new operators_1.Operator().dateGreaterThanEquals());
    }
    /**
     * Compare the requester's client application with the application that you specify.
     *
     * **Availability:** This key is always included in the request context.
     *
     * **Warning:** This key should be used carefully. Since the `aws:UserAgent` value is provided by the caller in an HTTP header, unauthorized parties can use modified or custom browsers to provide any `aws:UserAgent` value that they choose. As a result, `aws:UserAgent` should not be used to prevent unauthorized parties from making direct AWS requests. You can use it to allow only specific client applications, and only after testing your policy.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-useragent
     *
     * @param value The User Agent string(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsUserAgent(value, operator) {
        return this.if('aws:UserAgent', value, operator);
    }
    /**
     * Compare the requester's principal identifier with the ID that you specify. For IAM users, the request context value is the user ID. For IAM roles, this value format can vary. For details about how the information appears for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is included in the request context for all signed requests. Anonymous requests do not include this key.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-userid
     *
     * @param value The principal identifier(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsUserid(value, operator) {
        return this.if('aws:userid', value, operator);
    }
    /**
     * Compare the requester's user name with the user name that you specify. For details about how the information appears for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context for IAM users. Anonymous requests and requests that are made using the AWS account root user or IAM roles do not include this key.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-username
     *
     * @param value The user name(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsUsername(value, operator) {
        return this.if('aws:username', value, operator);
    }
    /**
     * Check whether an AWS service makes a request to another service on your behalf.
     *
     * The request context key returns `true` when a service uses the credentials of an IAM principal to make a request on behalf of the principal. The context key returns `false` if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. The request context key also returns `false` when the principal makes the call directly.
     *
     * **Availability:** This key is always included in the request context for most services.
     *
     *The following services do not currently support `aws:ViaAWSService`:
     *- Amazon EC2
     *- AWS Glue
     *- AWS Lake Formation
     *- AWS OpsWorks
     *
     * You can use this condition key to allow or deny access based on whether a request was made by a service. To view an example policy, see [AWS: Denies Access to AWS Based on the Source IP](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_examples_aws_deny-ip.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-viaawsservice
     *
     * @param value Whether a request was made by a service. **Default:** `true`
     */
    ifAwsViaAWSService(value) {
        return this.if(`aws:ViaAWSService`, typeof value !== 'undefined' ? value : true, new operators_1.Operator().bool());
    }
    /**
     * Compare the IP address from which a request was made with the IP address that you specify. In a policy, the key matches only if the request originates from the specified IP address and it goes through a VPC endpoint.
     *
     * **Availability:** This key is included in the request context only if the request is made using a VPC endpoint.
     *
     * For more information, see [Controlling Access to Services with VPC Endpoints](https://docs.aws.amazon.com/vpc/latest/userguide/vpc-endpoints-access.html) in the *Amazon VPC User Guide*.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-vpcsourceip
     *
     * @param value The VPC source IP(s)
     * @param operator Works with IP [address operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IPAddress). **Default:** `IpAddress`
     */
    ifAwsVpcSourceIp(value, operator) {
        return this.if('aws:VpcSourceIp', value, operator || new operators_1.Operator().ipAddress());
    }
}
exports.PolicyStatementWithCondition = PolicyStatementWithCondition;
_a = JSII_RTTI_SYMBOL_1;
PolicyStatementWithCondition[_a] = { fqn: "iam-floyd.PolicyStatementWithCondition", version: "0.505.0" };
function dateToString(value) {
    if (typeof value.getMonth === 'function') {
        value = value.toISOString();
    }
    return value;
}
//# sourceMappingURL=data:application/json;base64,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