"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrossAccountDestination = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const logs_generated_1 = require("./logs.generated");
/**
 * A new CloudWatch Logs Destination for use in cross-account scenarios.
 *
 * CrossAccountDestinations are used to subscribe a Kinesis stream in a
 * different account to a CloudWatch Subscription.
 *
 * Consumers will hardly ever need to use this class. Instead, directly
 * subscribe a Kinesis stream using the integration class in the
 * `@aws-cdk/aws-logs-destinations` package; if necessary, a
 * `CrossAccountDestination` will be created automatically.
 *
 * @stability stable
 * @resource AWS::Logs::Destination
 */
class CrossAccountDestination extends cdk.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.destinationName ||
                // In the underlying model, the name is not optional, but we make it so anyway.
                cdk.Lazy.string({ produce: () => this.generateUniqueName() }),
        });
        /**
         * Policy object of this CrossAccountDestination object.
         *
         * @stability stable
         */
        this.policyDocument = new iam.PolicyDocument();
        this.resource = new logs_generated_1.CfnDestination(this, 'Resource', {
            destinationName: this.physicalName,
            // Must be stringified policy
            destinationPolicy: this.lazyStringifiedPolicyDocument(),
            roleArn: props.role.roleArn,
            targetArn: props.targetArn,
        });
        this.destinationArn = this.getResourceArnAttribute(this.resource.attrArn, {
            service: 'logs',
            resource: 'destination',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.destinationName = this.getResourceNameAttribute(this.resource.ref);
    }
    /**
     * @stability stable
     */
    addToPolicy(statement) {
        this.policyDocument.addStatements(statement);
    }
    /**
     * Return the properties required to send subscription events to this destination.
     *
     * If necessary, the destination can use the properties of the SubscriptionFilter
     * object itself to configure its permissions to allow the subscription to write
     * to it.
     *
     * The destination may reconfigure its own permissions in response to this
     * function call.
     *
     * @stability stable
     */
    bind(_scope, _sourceLogGroup) {
        return { arn: this.destinationArn };
    }
    /**
     * Generate a unique Destination name in case the user didn't supply one
     */
    generateUniqueName() {
        // Combination of stack name and LogicalID, which are guaranteed to be unique.
        return cdk.Stack.of(this).stackName + '-' + this.resource.logicalId;
    }
    /**
     * Return a stringified JSON version of the PolicyDocument
     */
    lazyStringifiedPolicyDocument() {
        return cdk.Lazy.string({
            produce: () => this.policyDocument.isEmpty ? '' : cdk.Stack.of(this).toJsonString(this.policyDocument),
        });
    }
}
exports.CrossAccountDestination = CrossAccountDestination;
_a = JSII_RTTI_SYMBOL_1;
CrossAccountDestination[_a] = { fqn: "@aws-cdk/aws-logs.CrossAccountDestination", version: "1.110.1" };
//# sourceMappingURL=data:application/json;base64,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