"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildElasticSearchCWAlarms = exports.buildElasticSearch = void 0;
const elasticsearch = require("@aws-cdk/aws-elasticsearch");
const elasticsearch_defaults_1 = require("./elasticsearch-defaults");
const utils_1 = require("./utils");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
function buildElasticSearch(scope, domainName, options, cfnDomainProps) {
    // Setup the IAM Role & policy for ES to configure Cognito User pool and Identity pool
    const cognitoKibanaConfigureRole = new iam.Role(scope, 'CognitoKibanaConfigureRole', {
        assumedBy: new iam.ServicePrincipal('es.amazonaws.com')
    });
    const cognitoKibanaConfigureRolePolicy = new iam.Policy(scope, 'CognitoKibanaConfigureRolePolicy', {
        statements: [
            new iam.PolicyStatement({
                actions: [
                    "cognito-idp:DescribeUserPool",
                    "cognito-idp:CreateUserPoolClient",
                    "cognito-idp:DeleteUserPoolClient",
                    "cognito-idp:DescribeUserPoolClient",
                    "cognito-idp:AdminInitiateAuth",
                    "cognito-idp:AdminUserGlobalSignOut",
                    "cognito-idp:ListUserPoolClients",
                    "cognito-identity:DescribeIdentityPool",
                    "cognito-identity:UpdateIdentityPool",
                    "cognito-identity:SetIdentityPoolRoles",
                    "cognito-identity:GetIdentityPoolRoles",
                    "es:UpdateElasticsearchDomainConfig"
                ],
                resources: [
                    options.userpool.userPoolArn,
                    `arn:aws:cognito-identity:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:identitypool/${options.identitypool.ref}`,
                    `arn:aws:es:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:domain/${domainName}`
                ]
            }),
            new iam.PolicyStatement({
                actions: [
                    "iam:PassRole"
                ],
                conditions: {
                    StringLike: { 'iam:PassedToService': 'cognito-identity.amazonaws.com' }
                },
                resources: [
                    cognitoKibanaConfigureRole.roleArn
                ]
            })
        ]
    });
    cognitoKibanaConfigureRolePolicy.attachToRole(cognitoKibanaConfigureRole);
    let _cfnDomainProps = elasticsearch_defaults_1.DefaultCfnDomainProps(domainName, cognitoKibanaConfigureRole, options);
    if (cfnDomainProps) {
        _cfnDomainProps = utils_1.overrideProps(_cfnDomainProps, cfnDomainProps);
    }
    const esDomain = new elasticsearch.CfnDomain(scope, 'ElasticsearchDomain', _cfnDomainProps);
    esDomain.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W28',
                    reason: `The ES Domain is passed dynamically as as parameter and explicitly specified to ensure that IAM policies are configured to lockdown access to this specific ES instance only`
                }]
        }
    };
    return [esDomain, cognitoKibanaConfigureRole];
}
exports.buildElasticSearch = buildElasticSearch;
function buildElasticSearchCWAlarms(scope) {
    // Setup CW Alarms for ES
    const alarms = new Array();
    // ClusterStatus.red maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusRedAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.red'
        }),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(60),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one primary shard and its replicas are not allocated to a node. '
    }));
    // ClusterStatus.yellow maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusYellowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.yellow'
        }),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(60),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one replica shard is not allocated to a node.'
    }));
    // FreeStorageSpace minimum is <= 20480 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'FreeStorageSpaceTooLowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'FreeStorageSpace'
        }),
        threshold: 2000,
        evaluationPeriods: 1,
        statistic: 'Minimum',
        period: cdk.Duration.seconds(60),
        comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'A node in your cluster is down to 20 GiB of free storage space.'
    }));
    // ClusterIndexWritesBlocked is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'IndexWritesBlockedTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterIndexWritesBlocked'
        }),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Your cluster is blocking write requests.'
    }));
    // AutomatedSnapshotFailure maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'AutomatedSnapshotFailureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'AutomatedSnapshotFailure'
        }),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(60),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'An automated snapshot failed. This failure is often the result of a red cluster health status.'
    }));
    // CPUUtilization maximum is >= 80% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'CPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'CPUUtilization'
        }),
        threshold: 80,
        evaluationPeriods: 3,
        statistic: 'Average',
        period: cdk.Duration.seconds(900),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: '100% CPU utilization is not uncommon, but sustained high usage is problematic. Consider using larger instance types or adding instances.'
    }));
    // JVMMemoryPressure maximum is >= 80% for 5 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'JVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'JVMMemoryPressure'
        }),
        threshold: 80,
        evaluationPeriods: 1,
        statistic: 'Average',
        period: cdk.Duration.seconds(900),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    // MasterCPUUtilization maximum is >= 50% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'MasterCPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterCPUUtilization'
        }),
        threshold: 50,
        evaluationPeriods: 3,
        statistic: 'Average',
        period: cdk.Duration.seconds(900),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average CPU utilization over last 45 minutes too high. Consider using larger instance types for your dedicated master nodes.'
    }));
    // MasterJVMMemoryPressure maximum is >= 80% for 15 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'MasterJVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterJVMMemoryPressure'
        }),
        threshold: 50,
        evaluationPeriods: 1,
        statistic: 'Average',
        period: cdk.Duration.seconds(900),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    return alarms;
}
exports.buildElasticSearchCWAlarms = buildElasticSearchCWAlarms;
//# sourceMappingURL=data:application/json;base64,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