"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addPermission = exports.deployLambdaFunction = exports.buildLambdaFunction = void 0;
const lambda = require("@aws-cdk/aws-lambda");
const iam = require("@aws-cdk/aws-iam");
const lambda_defaults_1 = require("./lambda-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const security_group_helper_1 = require("./security-group-helper");
function buildLambdaFunction(scope, props) {
    // Conditional lambda function creation
    if (!props.existingLambdaObj) {
        if (props.lambdaFunctionProps) {
            return deployLambdaFunction(scope, props.lambdaFunctionProps, undefined, props.vpc);
        }
        else {
            throw Error('Either existingLambdaObj or lambdaFunctionProps is required');
        }
    }
    else {
        if (props.vpc) {
            if (!props.existingLambdaObj.isBoundToVpc) {
                throw Error('A Lambda function must be bound to a VPC upon creation, it cannot be added to a VPC in a subsequent construct');
            }
        }
        return props.existingLambdaObj;
    }
}
exports.buildLambdaFunction = buildLambdaFunction;
function deployLambdaFunction(scope, lambdaFunctionProps, functionId, vpc) {
    var _a, _b;
    const _functionId = functionId ? functionId : 'LambdaFunction';
    const _functionRoleId = _functionId + 'ServiceRole';
    if (vpc && lambdaFunctionProps.vpc) {
        throw new Error("Cannot provide a VPC in both the lambdaFunctionProps and the function argument");
    }
    // Setup the IAM Role for Lambda Service
    const lambdaServiceRole = new iam.Role(scope, _functionRoleId, {
        assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        inlinePolicies: {
            LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:PutLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`]
                    })]
            })
        }
    });
    // If this Lambda function is going to access resoures in a
    // VPC, then it needs privileges to access an ENI in that VPC
    if (lambdaFunctionProps.vpc || vpc) {
        lambdaServiceRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                "ec2:CreateNetworkInterface",
                "ec2:DescribeNetworkInterfaces",
                "ec2:DeleteNetworkInterface",
                "ec2:AssignPrivateIpAddresses",
                "ec2:UnassignPrivateIpAddresses"
            ],
            resources: ["*"]
        }));
    }
    // Override the DefaultFunctionProps with user provided  lambdaFunctionProps
    let finalLambdaFunctionProps = utils_1.overrideProps(lambda_defaults_1.DefaultLambdaFunctionProps(lambdaServiceRole), lambdaFunctionProps);
    if (vpc) {
        // This is literally setting up what would be the default SG, but
        // we need to to it explicitly to disable the cfn_nag error
        const lambdaSecurityGroup = security_group_helper_1.buildSecurityGroup(scope, "ReplaceDefaultSecurityGroup", {
            vpc,
            allowAllOutbound: true,
        }, [], []);
        finalLambdaFunctionProps = utils_1.overrideProps(finalLambdaFunctionProps, {
            securityGroups: [lambdaSecurityGroup],
            vpc,
        });
    }
    const lambdafunction = new lambda.Function(scope, _functionId, finalLambdaFunctionProps);
    if (lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_10_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_12_X) {
        lambdafunction.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
    }
    const cfnLambdafunction = lambdafunction.node.findChild('Resource');
    cfnLambdafunction.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W58',
                    reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with more tighter permissions.`
                }]
        }
    };
    if (cfnLambdafunction.tracingConfig) {
        // Find the X-Ray IAM Policy
        const cfnLambdafunctionDefPolicy = (_b = (_a = lambdafunction.role) === null || _a === void 0 ? void 0 : _a.node.tryFindChild('DefaultPolicy')) === null || _b === void 0 ? void 0 : _b.node.findChild('Resource');
        // Add the CFN NAG suppress to allow for "Resource": "*" for AWS X-Ray
        cfnLambdafunctionDefPolicy.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W12',
                        reason: `Lambda needs the following minimum required permissions to send trace data to X-Ray and access ENIs in a VPC.`
                    }]
            }
        };
    }
    return lambdafunction;
}
exports.deployLambdaFunction = deployLambdaFunction;
// A wrapper above Function.addPermision that
// prevents two different calls to addPermission using
// the same construct id.
function addPermission(targetFunction, name, permission) {
    targetFunction.addPermission(GetNextId(targetFunction.permissionsNode.children, name), permission);
}
exports.addPermission = addPermission;
// Scan the current permissions for any entries with this core name and
// return the first available synthesized name. Names are coreName-suffix.
function GetNextId(children, coreName) {
    let lastSuffix = 0;
    children.forEach(child => {
        // if (compare right side of string)
        if (child.node.id.indexOf(coreName) === 0) {
            const components = child.node.id.split('-');
            if (components.length !== 2) {
                throw new Error("Incorrectly formatted synthesized construct ID");
            }
            const usedSuffix = Number(components[1]);
            if (usedSuffix > lastSuffix) {
                lastSuffix = usedSuffix;
            }
        }
    });
    return `${coreName}-${lastSuffix + 1}`;
}
//# sourceMappingURL=data:application/json;base64,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