"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addCfnNagS3BucketNotificationRulesToSuppress = exports.createLoggingBucket = exports.applySecureBucketPolicy = exports.buildS3Bucket = void 0;
const s3 = require("@aws-cdk/aws-s3");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const rule_1 = require("@aws-cdk/aws-s3/lib/rule");
const duration_1 = require("@aws-cdk/core/lib/duration");
function buildS3Bucket(scope, props, bucketId) {
    if (props.bucketProps) {
        return s3BucketWithLogging(scope, props.bucketProps, bucketId);
    }
    else {
        return s3BucketWithLogging(scope, s3_bucket_defaults_1.DefaultS3Props(), bucketId);
    }
}
exports.buildS3Bucket = buildS3Bucket;
function applySecureBucketPolicy(s3Bucket) {
    // Apply bucket policy to enforce encryption of data in transit
    s3Bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${s3Bucket.bucketArn}/*`
        ],
        actions: ['*'],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
exports.applySecureBucketPolicy = applySecureBucketPolicy;
function createLoggingBucket(scope, bucketId) {
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, bucketId, s3_bucket_defaults_1.DefaultS3Props());
    applySecureBucketPolicy(loggingBucket);
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    // Override accessControl configuration and add metadata for the logging bucket
    loggingBucketResource.addPropertyOverride('AccessControl', 'LogDeliveryWrite');
    // Turn off Versioning for the logging bucket as objects will be written only ONCE
    loggingBucketResource.addPropertyDeletionOverride('VersioningConfiguration.Status');
    // Remove the default LifecycleConfiguration for the Logging Bucket
    loggingBucketResource.addPropertyDeletionOverride('LifecycleConfiguration.Rules');
    let _reason = "This S3 bucket is used as the access logging bucket for another bucket";
    if (bucketId === 'CloudfrontLoggingBucket') {
        _reason = "This S3 bucket is used as the access logging bucket for CloudFront Distribution";
    }
    loggingBucketResource.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W35',
                    reason: _reason
                }]
        }
    };
    return loggingBucket;
}
exports.createLoggingBucket = createLoggingBucket;
function s3BucketWithLogging(scope, s3BucketProps, bucketId) {
    /** Default Life Cycle policy to transition older versions to Glacier after 90 days */
    const lifecycleRules = [{
            noncurrentVersionTransitions: [{
                    storageClass: rule_1.StorageClass.GLACIER,
                    transitionAfter: duration_1.Duration.days(90)
                }]
        }];
    // Create the Application Bucket
    let bucketprops;
    let loggingBucket;
    const _bucketId = bucketId ? bucketId + 'S3Bucket' : 'S3Bucket';
    const _loggingBucketId = bucketId ? bucketId + 'S3LoggingBucket' : 'S3LoggingBucket';
    if (s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.serverAccessLogsBucket) {
        // Attach the Default Life Cycle policy ONLY IF the versioning is ENABLED
        if (s3BucketProps.versioned === undefined || s3BucketProps.versioned) {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(undefined, lifecycleRules);
        }
        else {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props();
        }
    }
    else {
        // Create the Logging Bucket
        loggingBucket = createLoggingBucket(scope, _loggingBucketId);
        // Attach the Default Life Cycle policy ONLY IF the versioning is ENABLED
        if ((s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.versioned) === undefined || s3BucketProps.versioned) {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(loggingBucket, lifecycleRules);
        }
        else {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(loggingBucket);
        }
    }
    if (s3BucketProps) {
        bucketprops = utils_1.overrideProps(bucketprops, s3BucketProps);
    }
    const s3Bucket = new s3.Bucket(scope, _bucketId, bucketprops);
    applySecureBucketPolicy(s3Bucket);
    return [s3Bucket, loggingBucket];
}
function addCfnNagS3BucketNotificationRulesToSuppress(stackRoot, logicalId) {
    const notificationsResourceHandler = stackRoot.node.tryFindChild(logicalId);
    const notificationsResourceHandlerRoleRole = notificationsResourceHandler.node.findChild('Role');
    const notificationsResourceHandlerRolePolicy = notificationsResourceHandlerRoleRole.node.findChild('DefaultPolicy');
    // Extract the CfnFunction from the Function
    const fnResource = notificationsResourceHandler.node.findChild('Resource');
    fnResource.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W58',
                    reason: `Lambda function has permission to write CloudWatch Logs via AWSLambdaBasicExecutionRole policy attached to the lambda role`
                }]
        }
    };
    // Extract the CfnPolicy from the iam.Policy
    const policyResource = notificationsResourceHandlerRolePolicy.node.findChild('Resource');
    policyResource.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W12',
                    reason: `Bucket resource is '*' due to circular dependency with bucket and role creation at the same time`
                }]
        }
    };
}
exports.addCfnNagS3BucketNotificationRulesToSuppress = addCfnNagS3BucketNotificationRulesToSuppress;
//# sourceMappingURL=data:application/json;base64,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