"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildSagemakerNotebook = void 0;
const sagemaker = require("@aws-cdk/aws-sagemaker");
const ec2 = require("@aws-cdk/aws-ec2");
const kms_helper_1 = require("./kms-helper");
const sagemaker_defaults_1 = require("./sagemaker-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const vpc_helper_1 = require("./vpc-helper");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const vpc_defaults_1 = require("./vpc-defaults");
const security_group_helper_1 = require("./security-group-helper");
function addPermissions(_role) {
    // Grant permissions to NoteBookInstance for creating and training the model
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${core_1.Aws.PARTITION}:sagemaker:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:*`],
        actions: [
            "sagemaker:CreateTrainingJob",
            "sagemaker:DescribeTrainingJob",
            "sagemaker:CreateModel",
            "sagemaker:DescribeModel",
            "sagemaker:DeleteModel",
            "sagemaker:CreateEndpoint",
            "sagemaker:CreateEndpointConfig",
            "sagemaker:DescribeEndpoint",
            "sagemaker:DescribeEndpointConfig",
            "sagemaker:DeleteEndpoint",
            "sagemaker:DeleteEndpointConfig",
            "sagemaker:InvokeEndpoint"
        ],
    }));
    // Grant CloudWatch Logging permissions
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/sagemaker/*`],
        actions: [
            'logs:CreateLogGroup',
            'logs:CreateLogStream',
            'logs:DescribeLogStreams',
            'logs:GetLogEvents',
            'logs:PutLogEvents'
        ],
    }));
    // Grant GetRole permissions to the SageMaker service
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [_role.roleArn],
        actions: [
            'iam:GetRole'
        ],
    }));
    // Grant PassRole permissions to the SageMaker service
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [_role.roleArn],
        actions: [
            'iam:PassRole'
        ],
        conditions: {
            StringLike: { 'iam:PassedToService': 'sagemaker.amazonaws.com' }
        },
    }));
}
function buildSagemakerNotebook(scope, props) {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j;
    // Setup the notebook properties
    let sagemakerNotebookProps;
    let vpcInstance;
    let securityGroup;
    let kmsKeyId;
    let subnetId;
    // Conditional Sagemaker Notebook creation
    if (!props.existingNotebookObj) {
        if ((((_a = props.sagemakerNotebookProps) === null || _a === void 0 ? void 0 : _a.subnetId) && ((_b = props.sagemakerNotebookProps) === null || _b === void 0 ? void 0 : _b.securityGroupIds) === undefined) ||
            (((_c = props.sagemakerNotebookProps) === null || _c === void 0 ? void 0 : _c.subnetId) === undefined && ((_d = props.sagemakerNotebookProps) === null || _d === void 0 ? void 0 : _d.securityGroupIds))) {
            throw new Error('Must define both sagemakerNotebookProps.subnetId and sagemakerNotebookProps.securityGroupIds');
        }
        addPermissions(props.role);
        if (((_e = props.sagemakerNotebookProps) === null || _e === void 0 ? void 0 : _e.kmsKeyId) === undefined) {
            kmsKeyId = kms_helper_1.buildEncryptionKey(scope).keyId;
        }
        else {
            kmsKeyId = props.sagemakerNotebookProps.kmsKeyId;
        }
        if (props.deployInsideVpc === undefined || props.deployInsideVpc) {
            if (((_f = props.sagemakerNotebookProps) === null || _f === void 0 ? void 0 : _f.subnetId) === undefined && ((_g = props.sagemakerNotebookProps) === null || _g === void 0 ? void 0 : _g.securityGroupIds) === undefined) {
                vpcInstance = vpc_helper_1.buildVpc(scope, {
                    defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps()
                });
                securityGroup = security_group_helper_1.buildSecurityGroup(scope, "SecurityGroup", {
                    vpc: vpcInstance,
                    allowAllOutbound: false
                }, [], [{ peer: ec2.Peer.anyIpv4(), connection: ec2.Port.tcp(443) }]);
                subnetId = vpcInstance.privateSubnets[0].subnetId;
                sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId, subnetId, [securityGroup.securityGroupId]);
            }
            else {
                sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId, (_h = props.sagemakerNotebookProps) === null || _h === void 0 ? void 0 : _h.subnetId, (_j = props.sagemakerNotebookProps) === null || _j === void 0 ? void 0 : _j.securityGroupIds);
            }
        }
        else {
            sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId);
        }
        if (props.sagemakerNotebookProps) {
            sagemakerNotebookProps = utils_1.overrideProps(sagemakerNotebookProps, props.sagemakerNotebookProps);
        }
        // Create the notebook
        const sagemakerInstance = new sagemaker.CfnNotebookInstance(scope, "SagemakerNotebook", sagemakerNotebookProps);
        if (vpcInstance) {
            return [sagemakerInstance, vpcInstance, securityGroup];
        }
        else {
            return [sagemakerInstance];
        }
    }
    else {
        // Return existing notebook object
        return [props.existingNotebookObj];
    }
}
exports.buildSagemakerNotebook = buildSagemakerNotebook;
//# sourceMappingURL=data:application/json;base64,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